#ifndef INCLUDE_VCA_MEDIA_FOUR_CC_H_
#define INCLUDE_VCA_MEDIA_FOUR_CC_H_
/**
 * @file       vca/media/four_cc.h
 * @copyright  VCA Technology
 * @~english
 * @brief      Video frame data type
 */

#include <vca/media/macros.h>

#include <stdint.h>

/**
 * @~english
 * A convenience macro for defining a fourcc value
 */
#define VCA_MAKE_FOUR_CC(a, b, c, d) \
  (((VcaMediaFourCc)(d) << 24) | ((VcaMediaFourCc)(c) << 16) | ((VcaMediaFourCc)(b) << 8) | (VcaMediaFourCc)(a))

/**
 * @~english
 * A typedef for a fourcc value
 */
typedef uint32_t VcaMediaFourCc;  // NOLINT(modernize-use-using)

/**
 * @~english
 * An enum of supported fourcc values
 */
enum VcaMediaFourCcs {
  /**
   * @~english
   * 8-bit Y-plane for monochrome images.
   */
  VcaMediaFourCcY800 = VCA_MAKE_FOUR_CC('Y', '8', '0', '0'),

  /**
   * @~english
   * Planar 4:2:0 YUV
   */
  VcaMediaFourCcI420 = VCA_MAKE_FOUR_CC('I', '4', '2', '0'),

  /**
   * @~english
   * Planar 4:2:0 YUV. Same as @c VcaMediaFourCcI420 but with U and V planes swapped.
   */
  VcaMediaFourCcYV12 = VCA_MAKE_FOUR_CC('Y', 'V', '1', '2'),

  /**
   * @~english
   * Packed 4:2:2 YUV
   */
  VcaMediaFourCcYUY2 = VCA_MAKE_FOUR_CC('Y', 'U', 'Y', '2'),

  /**
   * @~english
   * Packed 4:2:2 YUV. Same as @c VcaMediaFourCcYuy2 but with U and V planes swapped.
   */
  VcaMediaFourCcYVYU = VCA_MAKE_FOUR_CC('Y', 'V', 'Y', 'U'),

  /**
   * @~english
   * Packed 4:2:2 YUV
   */
  VcaMediaFourCcUYVY = VCA_MAKE_FOUR_CC('U', 'Y', 'V', 'Y'),

  /**
   * @~english
   * Packed 4:4:4 YUV with alpha channel
   */
  VcaMediaFourCcAYUV = VCA_MAKE_FOUR_CC('A', 'Y', 'U', 'V'),

  /**
   * @~english
   * RGB 8-bits per channel.
   */
  VcaMediaFourCcRGB = VCA_MAKE_FOUR_CC('R', 'G', 'B', 24),

  /**
   * @~english
   * Reverse RGB 8-bits per channel.
   */
  VcaMediaFourCcBGR = VCA_MAKE_FOUR_CC('B', 'G', 'R', 24),

  /**
   * @~english
   * "NV12" planar 4:2:0 YUV with interleaved UV plane.
   */
  VcaMediaFourCcNV12 = VCA_MAKE_FOUR_CC('N', 'V', '1', '2'),

  /**
   * @~english
   * "NV21" planar 4:2:0 YUV with interleaved VU plane.
   */
  VcaMediaFourCcNV21 = VCA_MAKE_FOUR_CC('N', 'V', '2', '1')
};

#endif /* INCLUDE_VCA_MEDIA_FOUR_CC_H_ */
