#ifndef INCLUDE_VCA_CORE_SDK_SERVICE_CONFIGURE_H_
#define INCLUDE_VCA_CORE_SDK_SERVICE_CONFIGURE_H_
/**
 * @file      vca/core_sdk/service/configure.h
 * @copyright VCA Technology
 * @~english
 * @brief     Provides a means of interacting with the VCA Configuration via JSON
 */

#include <vca/core_sdk/macros.h>

VCA_CORE_SDK_EXTERN_C_BEGIN

struct VcaCoreService;

/**
 * @~english
 * The method of interaction with configuration
 */
enum VcaCoreConfigureMethod {
  /**
   * @~english
   * Gets the current JSON representation of the configuration at the given path
   */
  VcaCoreConfigureMethodGet,

  /**
   * @~english
   * Modifies the configuration node at the given path
   */
  VcaCoreConfigureMethodPut,

  /**
   * @~english
   * Adds a new element to the configuration at the given path. Only allowed for dynamic nodes, e.g. channels, zones,
   * actions etc. When successful, the JSON string in the result will contain an object with two key/value pairs:
   * 'index', which is the index of the added element, and 'object', which is the JSON representation of the added
   * object.
   */
  VcaCoreConfigureMethodPost,

  /**
   * @~english
   * Deletes the element at the given path from the configuration. Only allowed for children of dynamic nodes, e.g.
   * channels, zones, actions etc.
   */
  VcaCoreConfigureMethodDelete
};

/**
 * @~english
 * The status of a request
 */
enum VcaCoreConfigureStatus {
  /**
   * @~english
   * The request was successful
   */
  VcaCoreConfigureStatusSuccess,

  /**
   * @~english
   * The specified path was not found in the configuration
   */
  VcaCoreConfigureStatusNotFound,

  /**
   * @~english
   * The method used is not applicable for the specified path
   */
  VcaCoreConfigureStatusNotAllowed,

  /**
   * @~english
   * The input data supplied is not valid JSON
   */
  VcaCoreConfigureStatusParseError,

  /**
   * @~english
   * An error occurred while trying to make the requested changes to the configuration
   */
  VcaCoreConfigureStatusInternalError
};

/**
 * @~english
 * The result of a configuration request
 */
struct VcaCoreConfigureResult {
  /**
   * @~english
   * The status of the request
   */
  enum VcaCoreConfigureStatus status;

  /**
   * @~english
   * String containing the results of the request in JSON format. The contents will depend on the request method (may
   * be an empty string)
   */
  char* json_data;
};

/**
 * @~english
 * Interacts with the VCA Core configuration. MT safe.
 * @param service the service to configure
 * @param method the method of interaction with the configuration
 * @param path dot-separated path to configuration node, e.g. 'channels.0.enabled'
 * @param json_data the input JSON data. Not required for all request methods.
 * @returns the result of the request. This structure must be freed using VcaCoreConfigureResultFree.
 */
VCA_CORE_SDK_DLL_PUBLIC
struct VcaCoreConfigureResult VcaCoreServiceConfigure(
  struct VcaCoreService* const service,
  enum VcaCoreConfigureMethod method,
  const char* const path,
  const char* const json_data
);

/**
 * @~english
 * Frees the memory allocated for the result of a configuration request.
 * @param result the configuration request result structure to free
 */
VCA_CORE_SDK_DLL_PUBLIC
void VcaCoreConfigureResultFree(struct VcaCoreConfigureResult* result);

VCA_CORE_SDK_EXTERN_C_END

#endif /* INCLUDE_VCA_CORE_SDK_SERVICE_CONFIGURE_H_ */
