const YUVBuffer = require('yuv-buffer')
const EventEmitter = require('events').EventEmitter
const path = require('path')
const binding_path = require('node-pre-gyp').find(path.resolve(path.join(__dirname,'./package.json')))
const VN_NativePlayer = require(binding_path).VN_Player
const buf_num = 1
//const fs = require('fs')

function sleep (ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
}

function wait(ms) {
    var start = new Date().getTime(),
        end = start;
    while(end < start + ms)
        end = new Date().getTime();
}

function forceGC() {
    if (global.gc)
      global.gc();
    else
      console.warn('No GC hook! Add `--expose-gc` to execution options.');
}

class VN_Player {
    constructor(options) {
        const cell_w = ('cell_w' in options) ? options.cell_w : undefined
        const cell_h = ('cell_h' in options) ? options.cell_h : undefined
        const scale = ('scale' in options) ? options.scale : false
        const vsync = ('vsync' in options) ? options.vsync : false

        var objid = this._objid = ('objid' in options) ? options.objid : undefined
        this._emitter = new EventEmitter()
        this._frame   = new Array(buf_num)

        // next variables MUST be available for the draw() function
        var frame
        var player = this
        var _yuvCanvas = this._yuvCanvas = ('yuvCanvas' in options) ? options.yuvCanvas : undefined

        function draw(time) {
            //console.log(`[${objid}] draw`) //this.reqId, window)
            if(frame !== undefined
                && vn_player !== undefined
                && vn_player.is_new_frame_ready())
            {
                //console.log(`[${objid}] new frame is ready`)
                _yuvCanvas.drawFrame(frame)
                player.onNewFrame()
            }
            window.requestAnimationFrame(draw)
        }

        this._emitter.on('state_changed', (state) => {
            console.log(`[${objid}] state_changed to ${state}`)
            this.onStateChanged(state)
        });

        this._emitter.on('create_image_buffer', (w, h, ext_buf) => {
            const width = w*1,
                  height = h*1,
                  chromaWidth = Math.round(w/2),
                  chromaHeight = Math.round(h/2),
                  luma_len = width*height,
                  chroma_len = chromaWidth*chromaHeight,
                  total_len = luma_len + 2 * chroma_len // luma_len*1.5
            console.log(`[${objid}] create_image_buffer ${w}x${h}`)

            this._format = YUVBuffer.format({
                width: width,
                height: height,
                chromaWidth: chromaWidth,
                chromaHeight: chromaHeight,
                displayWidth: scale && cell_w !== undefined ? cell_w : width,
                displayHeight: scale && cell_h !== undefined ? cell_h : height
                //cropWidth: width
            })

            this._buf = new ArrayBuffer(buf_num*total_len)
            for(let i=0; i<buf_num; i++) {
                const offset = i*total_len

                this._frame[i] = YUVBuffer.frame(this._format)
                this._frame[i].y.bytes = new Uint8Array(this._buf, offset, luma_len)
                this._frame[i].u.bytes = new Uint8Array(this._buf, offset + luma_len, chroma_len)
                this._frame[i].v.bytes = new Uint8Array(this._buf, offset + luma_len + chroma_len, chroma_len)
            }

            this._native_vn_player.set_img_buffer(this._buf, ext_buf)
            frame = this._frame[0]
        });

        this._emitter.on('new_frame', (ts, metadata) => {
            //console.log(`[${objid}] new_frame (ts: ${ts}, metadata: ${metadata})`)
            if(frame !== undefined) {
                if(this._yuvCanvas !== undefined && !vsync) {
                    this._yuvCanvas.drawFrame(frame)
                    this.onNewFrame(ts, metadata)
                }

                /**
                 * For DEBUG purposes
                 *
                if(this.cnt < 10) {
                    let fd = fs.openSync(`/tmp/${this.cnt}.yuv`, 'w')
                    //fs.writeFileSync(`/tmp/${this.cnt}.y`, frame.y.bytes)
                    //fs.writeFileSync(`/tmp/${this.cnt}.u`, frame.u.bytes)
                    //fs.writeFileSync(`/tmp/${this.cnt}.v`, frame.v.bytes)
                    fs.writeSync(fd, frame.y.bytes)
                    fs.writeSync(fd, frame.u.bytes)
                    fs.writeSync(fd, frame.v.bytes)
                    fs.closeSync(fd)
                    this.cnt++
                }*/
            }
        });

        var vn_player = this._native_vn_player = new VN_NativePlayer(this._emitter.emit.bind(this._emitter), objid, vsync, buf_num)
        if(vsync)
            window.requestAnimationFrame(draw)
    }

    play(url) {
        console.log(`[${this._objid}] Let's play url: ${url}`)
        this._native_vn_player.play(url)
    }

    pause() {
        console.log(`[${this._objid}] pause`)
        this._native_vn_player.pause()
    }

    resume(direction) {
        console.log(`[${this._objid}] resume, direction: ${direction}`)
        this._native_vn_player.resume(direction)
    }

    teardown() {
        console.log(`[${this._objid}] teardown`)
        this._native_vn_player.teardown()
        this._native_vn_player = null
        if(this._yuvCanvas !== undefined)
            this._yuvCanvas.clear()
        forceGC(); //after forceGC, the C++ destructor function will call
    }

    // May be extended with useful logic in the derived classes
    onNewFrame(...args) {
        console.log(`[${this._objid}] pure onNewFrame() from super class. It would be better to extend it in a derived class.`)
    }

    onStateChanged(...args) {
        console.log(`[${this._objid}] pure onStateChanged() from super class. It would be better to extend it in a derived class.`)
    }
};

module.exports = VN_Player;
