const YUVBuffer = require('yuv-buffer')
const EventEmitter = require('events').EventEmitter
const path = require('path')
const binding_path = require('node-pre-gyp').find(path.resolve(path.join(__dirname,'./package.json')))
const VN_NativePlayer = require(binding_path).VN_Player
//const fs = require('fs')

// Only add setZeroTimeout to the window object, and hide everything
// else in a closure. (http://dbaron.org/log/20100309-faster-timeouts)
const fnSetZeroTimeout = () => {
    //if(typeof window === 'undefined' || typeof window.setZeroTimeout === 'function')
    //    return

    const timeouts = [];
    const messageName = "zero-timeout-message";

    // Like setTimeout, but only takes a function argument.  There's
    // no time argument (always zero) and no arguments (you have to
    // use a closure).
    function setZeroTimeout(fn) {
      timeouts.push(fn);
      window.postMessage(messageName, "*");
    }

    function handleMessage(event) {
      if (event.source == window && event.data == messageName) {
        event.stopPropagation();
        if (timeouts.length > 0) {
          const fn = timeouts.shift();
          fn();
        }
      }
    }

    window.addEventListener("message", handleMessage, true);

    // Add the one thing we want added to the window object.
    window.setZeroTimeout = setZeroTimeout;
};

if(typeof window !== 'undefined' && typeof window.setZeroTimeout !== 'function')
    fnSetZeroTimeout();
const zero_timeout_enabled
    = typeof window !== 'undefined' && typeof window.setZeroTimeout === 'function'

function wait(ms) {
    const start = new Date().getTime(),
          end = start
    while(end < start + ms)
        end = new Date().getTime()
}

function forceGC() {
    if (global.gc)
      global.gc()
    else
      console.warn('No GC hook! Add `--expose-gc` to execution options.')
}

class VN_Player {
    constructor(options) {
        //let cnt = 0 // counter of new frames for debugging
        let rendering = false
        let cell_w = ('cell_w' in options) ? options.cell_w : undefined // may be resized
        let cell_h = ('cell_h' in options) ? options.cell_h : undefined // may be resized
        const scale = ('scale' in options) ? options.scale : false
        const vsync = ('vsync' in options) ? options.vsync : false
        const objid = ('objid' in options) ? options.objid : undefined
        const yuvCanvas = ('yuvCanvas' in options) ? options.yuvCanvas : undefined
        const cache_size = ('cache_size' in options) ? options.cache_size : 2 // Mb
        const jbuf_len = ('jitter_buffer_len' in options) ? options.jitter_buffer_len : 1000 // ms
        const buf_num = ('frame_buffers_num' in options) ? options.frame_buffers_num : 1 // number of frame buffers to use

        const emitter = new EventEmitter()
        const frame = new Array(buf_num)
        let native_vn_player = new VN_NativePlayer({
            emit: emitter.emit.bind(emitter),
            objid: objid, vsync: vsync,
            cache_size: cache_size,      // Mb
            jitter_buffer_len: jbuf_len, // ms
            frame_buffers_num: buf_num
        })

        // used in 'new_frame' callback when vsync disabled
        function drawSingleFrameAsync(num, callback) {
            if(!zero_timeout_enabled /*|| !yuvCanvas*/) { // console app?
                callback()
                return
            }

            //const now = performance.now()
            setZeroTimeout(() => {
                try {
                    //console.log(`[${objid}]`, 'timeout before drawing frame:', Math.round((performance.now() - now)*1000), 'usecs')
                    yuvCanvas.drawFrame(frame[num])
                    //console.log(`[${objid}]`, 'drawing the frame took', Math.round((performance.now() - now)*1000), 'usecs')
                } catch (error) {
                    console.log('ERROR:', error.message)
                } finally {
                    callback()
                }
            })
        }

        // Used when vsync enabled.
        // This is the most preferred rendering method, since, apart from synchronization
        // with the monitor refresh rate, it does not depend on the overflow of the frame
        // queue.
        async function render() {
            console.log(`[${objid}] rendering started`)
            rendering = true
            do {
                if(frame[0] && native_vn_player)
                {
                    const obj = native_vn_player.get_info_and_lock_frame_if_ready();
                    //console.log(`[${objid}] new frame is ready:`, obj)
                    if(obj.new) {
                        yuvCanvas.drawFrame(frame[0])
                        native_vn_player.frame_unlock()
                        this.onNewFrame(obj.ts, obj.meta)
                    }
                }

                await new Promise(requestAnimationFrame)
            } while(rendering);

            console.log(`[${objid}] rendering stopped`)
        }

        emitter.on('state_changed', (state) => {
            if(!native_vn_player) return // sanity check;)
            console.log(`[${objid}] state_changed to ${state}`)
            this.onStateChanged(state)
            if(!vsync || !yuvCanvas)
                return

            switch(state) {
                //case 0:
                case 3:
                    rendering = false
                    break;
                case 6:
                    render.call(this)
                    break;
            }
        });

        emitter.on('create_image_buffer', (w, h, ext_buf) => {
            if(!native_vn_player) return // sanity check;)
            const width = w*1,
                  height = h*1,
                  chromaWidth = Math.round(w/2),
                  chromaHeight = Math.round(h/2),
                  luma_len = width*height,
                  chroma_len = chromaWidth*chromaHeight,
                  total_len = luma_len + 2 * chroma_len // luma_len*1.5
            console.log(`[${objid}] create_image_buffer ${w}x${h}`)

            const format = YUVBuffer.format({
                width: width,
                height: height,
                chromaWidth: chromaWidth,
                chromaHeight: chromaHeight,
                displayWidth: scale && cell_w !== undefined ? cell_w : width,
                displayHeight: scale && cell_h !== undefined ? cell_h : height
                //cropWidth: width
            })

            const buf = new ArrayBuffer(buf_num*total_len)
            for(let i=0; i<buf_num; i++) {
                const offset = i*total_len

                frame[i] = YUVBuffer.frame(format)
                frame[i].y.bytes = new Uint8Array(buf, offset, luma_len)
                frame[i].u.bytes = new Uint8Array(buf, offset + luma_len, chroma_len)
                frame[i].v.bytes = new Uint8Array(buf, offset + luma_len + chroma_len, chroma_len)
            }

            native_vn_player.set_img_buffer(buf, ext_buf)
        });

        emitter.on('new_frame', (ts, metadata) => {
            if(!native_vn_player) return // sanity check;)
            //console.log(`cnt: ${cnt}`)
            drawSingleFrameAsync(0, () => {
                native_vn_player.frame_unlock()
                //const now = performance.now()
                this.onNewFrame(ts, metadata)
                //console.log('onNewFrame() took:', Math.round((performance.now() - now)*1000), 'usecs')
            })
            //console.log(`[${objid}] new_frame_${cnt++} (ts: ${ts}, metadata: ${metadata})`)
            return

            // TODO: the same logic using promise. Decide to remove later.
            // The use of promise in this case is redundant,
            // because we have no nested callbacks, but only one.
            // But perhaps it is more clearly because all in one place.
            const now = performance.now()
            new Promise((resolve) => setZeroTimeout(() => {
                console.log(`[${objid}]`, 'timeout before drawing frame:', Math.round((performance.now() - now)*1000), 'usecs')
                yuvCanvas.drawFrame(frame[num])
                console.log(`[${objid}]`, 'drawing the frame took', Math.round((performance.now() - now)*1000), 'usecs')
                resolve()
            })).then(() => {
                native_vn_player.frame_unlock()
                //const now = performance.now()
                this.onNewFrame(ts, metadata)
                //console.log('onNewFrame() took:', Math.round((performance.now() - now)*1000), 'usecs')
            }).catch(error => {
                if(native_vn_player)
                    native_vn_player.frame_unlock()
                console.log('ERROR:', error.message)
                this.onNewFrame(ts, metadata)
            })

            // LEGACY
            /* if(frame[0] !== undefined) {
                if(yuvCanvas !== undefined && !vsync)
                    yuvCanvas.drawFrame(frame[0])
                this.onNewFrame(ts, metadata)
                */

                /**
                 * For DEBUG purposes
                 *
                if(this.cnt < 10) {
                    let fd = fs.openSync(`/tmp/${this.cnt}.yuv`, 'w')
                    //fs.writeFileSync(`/tmp/${this.cnt}.y`, frame.y.bytes)
                    //fs.writeFileSync(`/tmp/${this.cnt}.u`, frame.u.bytes)
                    //fs.writeFileSync(`/tmp/${this.cnt}.v`, frame.v.bytes)
                    fs.writeSync(fd, frame.y.bytes)
                    fs.writeSync(fd, frame.u.bytes)
                    fs.writeSync(fd, frame.v.bytes)
                    fs.closeSync(fd)
                    this.cnt++
                }*/
            //}
        });

        this._play = (url) => {
            if(!native_vn_player) return // sanity check;)
            console.log(`[${objid}] Let's play url: ${url}`)
            native_vn_player.play(url)
        };

        this._pause = () => {
            rendering = false
            if(!native_vn_player) return // sanity check;)
            console.log(`[${objid}] pause`)
            native_vn_player.pause()
        };
    
        this._resume = (direction) => {
            if(!native_vn_player) return // sanity check;)
            if(vsync && yuvCanvas)
                render.call(this)
            console.log(`[${objid}] resume, direction: ${direction}`)
            native_vn_player.resume(direction)
        };
    
        this._teardown = () => {
            rendering = false
            console.log(`[${objid}] teardown`)
            if(yuvCanvas)
                yuvCanvas.clear()
            if(!native_vn_player) return // sanity check;)
            native_vn_player.teardown()
            native_vn_player = null
            forceGC(); //after forceGC, the C++ destructor function will call
        };

        this._resize = (cw, ch) => {
            cell_w = cw
            cell_h = ch
            if(!native_vn_player || !scale)
                return
            console.log(`[${objid}] cell resized to ${cw}x${ch}`)

            if(yuvCanvas)
                yuvCanvas.clear()

            for(let i=0; i<buf_num; i++) {
                if(!frame[i])
                    continue
                frame[i].format.displayWidth = cw
                frame[i].format.displayHeight = ch
            }
        };
    }

    // Following methods may be extended with useful logic in the derived classes
    play(url) {
        this._play(url)
    }

    pause() {
        this._pause()
    }

    resume(direction) {
        this._resume(direction)
    }

    teardown() {
        this._teardown()
    }

    // Since VN_Player base class can be used also in a node console application,
    // the resize method must be called explicitly on cell resizing.
    // Please use following ds-electron's commit as an example of handling cells resize:
    // https://bb.videonext.com/projects/DIS/repos/ds-electron/commits/60cacfdad8e821571a26307d15b2f02da9af1cf0
    resize(cw, ch) {
        this._resize(cw, ch)
    }

    onNewFrame(...args) {
        console.log('Pure VN_Player.onNewFrame() method called. It would be better to extend it in a derived class.')
        //console.log(args[0], args[1])
    }

    onStateChanged(...args) {
        console.log('Pure VN_Player.onStateChanged() method called. It would be better to extend it in a derived class.')
    }
};

module.exports = VN_Player;
