import {TimeLineData} from "../vistimeline";
import {CoverageStream, Granularity, TimeStamp} from "@solid/types/coverage";

export interface StreamList<T> extends Array<T> {
	[stream: number]: T
}

// like in /api/cm/coverage response
export type Coverage = {
	[granularity in Granularity]: StreamList<TimeStamp[]>
}

export type ConvertedCoverage = {
	[granularity in Granularity]: StreamList<CoverageStream>
}

export class DataCoverage {
	initial: {
		coverage: Partial<Coverage>,
		partialCoverage: Partial<Coverage>,
		eventCoverage: TimeStamp[]
	} = {
		coverage: {},
		partialCoverage: {},
		eventCoverage: []
	};

	converted: {
		coverage: Partial<ConvertedCoverage>,
		partialCoverage: Partial<ConvertedCoverage>,
		eventCoverage: CoverageStream
	} = {
		coverage: {},
		partialCoverage: {},
		eventCoverage: []
	};

	data: StreamList<TimeLineData> = [];

	coverageProbability: number = 80; // percent
	partialCoverageProbability: number = 5; // percent
	eventCoverageProbability: number = 2; // percent

	generate(start: number, end: number, streamCount: number, granularity: Granularity = Granularity.chunk) {
		let time = performance.now();

		const coverageList: StreamList<TimeStamp[]> = new Array(streamCount);
		const partialCoverageList: StreamList<TimeStamp[]> = new Array(streamCount);

		for (let stream = 0; stream < streamCount; stream++) {
			if (!coverageList[stream]) {
				coverageList[stream] = [];
			}
			if (!partialCoverageList[stream]) {
				partialCoverageList[stream] = [];
			}

			coverageList[stream] = this._generateList(start, end, this.coverageProbability, granularity);
			partialCoverageList[stream] = this._generateList(start, end, this.partialCoverageProbability, granularity);
		}

		this.initial.coverage[granularity] = coverageList;
		this.initial.partialCoverage[granularity] = partialCoverageList;
		this.initial.eventCoverage = this._generateList(start, end, this.eventCoverageProbability, Granularity.second);

		console.log(
			'DataCoverage.generate', `${performance.now() - time}ms`,
			'coverage count', Object.keys(coverageList[0]).length, Object.keys(coverageList[1]).length,
			'partialCoverage count', Object.keys(partialCoverageList[0]).length, Object.keys(partialCoverageList[1]).length,
			'eventCoverage count', Object.keys(this.initial.eventCoverage).length
		);

		this._convert();
	}

	_convert(granularity: Granularity = Granularity.chunk) {
		let time = performance.now();

		const coverage = this.initial.coverage[granularity];
		if (!coverage) {
			return;
		}

		const partialCoverage = this.initial.partialCoverage[granularity];
		if (!partialCoverage) {
			return;
		}

		const coverageList: StreamList<CoverageStream> = [];
		const partialCoverageList: StreamList<CoverageStream> = [];

		for (let stream: number = 0; stream < coverage.length; stream++) {
			const coverageStream: CoverageStream = {};
			for (let i = 0; i < coverage[stream].length; i++) {
				const timeStamp = coverage[stream][i];
				coverageStream[timeStamp] = {};
			}
			coverageList.push(coverageStream);

			const partialCoverageStream: CoverageStream = {};
			for (let i = 0; i < partialCoverage[stream].length; i++) {
				const timeStamp = partialCoverage[stream][i];
				partialCoverageStream[timeStamp] = {};
			}
			partialCoverageList.push(partialCoverageStream);
		}

		const eventCoverage = this.initial.eventCoverage;
		const eventCoverageList: CoverageStream = {};
		for (let i = 0; i < eventCoverage.length; i++) {
			const timeStamp = eventCoverage[i];
			eventCoverageList[timeStamp] = {isEvent: true};
		}

		this.converted.coverage[granularity] = coverageList;
		this.converted.partialCoverage[granularity] = partialCoverageList;
		this.converted.eventCoverage = eventCoverageList;

		console.log('DataCoverage._convert', `${performance.now() - time}ms`);
	}

	_generateList(start: number, end: number, probability: number, granularity: Granularity = Granularity.chunk) {
		const list: TimeStamp[] = [];

		const {step, date} = DataCoverage.getStep(granularity, start);
		if (step) {
			for (let timeStamp = date.getTime(); timeStamp < end; timeStamp += step * 1000) {
				if (Math.random() * 100 > probability) {
					continue;
				}

				if (timeStamp % (1000 * step) !== 0) {
					continue;
				}

				list.push(timeStamp);
			}
		}

		return list;
	}

	static getStep(granularity: Granularity, time: number): {step?: number, date: Date} {
		let date = new Date();
		date.setTime(time);

		let step: number | undefined;
		switch (granularity) {
			case 'year':
				break;
			case 'mon':
				break;
			case 'day':
				step = 24 * 3600;
				date.setUTCHours(0, 0, 0, 0);
				break;
			case 'hour':
				step = 3600;
				date.setUTCMinutes(0, 0, 0);
				break;
			case 'min':
				step = 60;
				date.setUTCSeconds(0, 0);
				break;
			case 'chunk':
				step = 30;
				date.setUTCSeconds(0, 0);
				break;
			case 'second':
				step = 1;
				date.setUTCSeconds(0, 0);
				break;
		}

		return {
			step,
			date
		}
	}

	get(start: number, end: number, granularity: Granularity = Granularity.chunk): CoverageStream[] {
		let time = performance.now();

		const list: CoverageStream[] = [];
		const {step, date} = DataCoverage.getStep(granularity, start);
		if (step) {
			const coverage = this.converted.coverage[granularity];
			const partialCoverage = this.converted.partialCoverage[granularity];
			if (coverage && partialCoverage) {
				for (let stream = 0; stream < coverage.length; stream++) {
					if (!list[stream]) {
						list[stream] = {};
					}

					for (let timeStamp = date.getTime(); timeStamp < end; timeStamp += step * 1000) {
						if (partialCoverage[stream][timeStamp]) {
							list[stream][timeStamp] = {
								...partialCoverage[stream][timeStamp],
								isPartialChunk: true
							};
						} else
						if (coverage[stream][timeStamp]) {
							list[stream][timeStamp] = {
								...coverage[stream][timeStamp],
								isChunk: true
							};
						}
					}
				}
			}
		}

		let {step: eventStep, date: eventDate} = DataCoverage.getStep(Granularity.second, start);
		if (eventStep) {
			const eventCoverage = this.converted.eventCoverage;
			const coverageStream: CoverageStream = {};
			for (let timeStamp = eventDate.getTime(); timeStamp < end; timeStamp += eventStep * 1000) {
				if (eventCoverage[timeStamp]) {
					coverageStream[timeStamp] = {
						...eventCoverage[timeStamp],
						isEvent: true
					}
				}
			}
			list.push(coverageStream);
		}

		console.log('DataCoverage.get', `${performance.now() - time}ms`);
		return list;
	}
}
