import {FileTransport} from "./filetransport";
import {__, AjaxError, API, isElectron, Logger, timeout, TimeoutError} from "@solid/libs";
import {TransportInitParameters} from "./msemediaplayer";
import { StreamParams, UUID } from "@solid/types";

const isUseFileTransport = false && isElectron();

// Chrome / Firefox compatibility
// let RTCPeerConnection = window.RTCPeerConnection || window.mozRTCPeerConnection || window.webkitRTCPeerConnection;
// let RTCIceCandidate = window.RTCIceCandidate || window.mozRTCIceCandidate;
// let RTCSessionDescription = window.RTCSessionDescription || window.mozRTCSessionDescription;
// TODO: Firefox appears to require the offering peer to send both the
// offer + candidate(s) to the answerer, before having the answer applied.
// Chrome seems to be more forgiving of mixing the ordering.
// I have successfully gotten Firefox and Chrome to create a data channel using
// this code, (either can start). I've also gotten both Firefox and Chrome to
// successfully connect to the Go client, but chat messages from the Go client so
// far do not appear for Firefox, while they do for Chrome.
// The signaling semantics should probably be combined in any case, for ease
// of use, but all the data channel interoperability needs more investigation.

type WebRTCStatus = {
  time: number,
  fail?: {
    code: number,
    message: string
  } | null
}

type WebRTCMediaTransportParameters = {
  logger?: Logger | Console,
  debug?: boolean,
  debugData?: boolean
}

export class WebRTCMediaTransport {
  _fileTransport: FileTransport = new FileTransport("webrtc");

  _logger: Logger | Console = console;

  _api: API = new API();

  _peerConnection: RTCPeerConnection | null = null;
  _channel: RTCDataChannel | null = null;

  _requestLog: string[] = [];
  _requestLogMaxSize = 50;

  _debug = false;
  _debugData = false;

  _parameters: TransportInitParameters = {
    isUseFileTransport: false,
    isLocal: false
  };

  _stream: StreamParams = {
    mime: "",
    width: 0,
    height: 0,
  };

  _heartBeatInterval: number = 0;
  _lastReceivedTime: number = 0;
  _currentChunk: Uint8Array | null = null;
  _endEvent: {time: number | null, event: string} | null = null
  _endKeyTime: number = 0;
  _doPlay: boolean = false;
  _initMIME: boolean = false;
  _isChunkSplitted: boolean = false;
  _skipChunks: boolean = false;
  _chunkPosition: number = 0;
  _chunkNumber?: number;
  _chunkNumber2?: number;
  _chunkSize?: number;
  _chunkMD5?: string;

  _isCloseCallback = true;
  _isCloseCallbackCalled = false;

  _connectTimeout = 30 * 1000; // ms
  _checkConnectionTimeout = 15 * 60 * 1000; // ms

  _avatarId: UUID | null = null;

  constructor(parameters: WebRTCMediaTransportParameters = {}) {
    if (typeof parameters.logger !== "undefined") {
      this._logger = parameters.logger;
    }
    if (typeof parameters.debug !== "undefined") {
      this._debug = parameters.debug;
    }
      if (typeof parameters.debugData !== "undefined") {
      this._debugData = parameters.debugData;
    }
  }

  /**
   *
   * @param parameters
   * @returns {Promise<boolean>}
   */
  async init(parameters: TransportInitParameters) {
    this.setParameters(parameters);

    this._isCloseCallback = true;
    this._isCloseCallbackCalled = false;

    if (this.isOpen()) {
      return true;
    }

    isUseFileTransport && await this._fileTransport.init();

    const avatarId = await this.getAvatarId();
    if (!avatarId) {
      throw {message: `Cannot get avatarId for ${this._parameters.obj}`};
    }
    const isPreviousWebRTCSuccess = this.isPreviousWebRTCSuccess(avatarId);
    let webRTCStatus = this.getWebRTCStatus(avatarId);

    if (!this._parameters.alwaysConnect && !isPreviousWebRTCSuccess && webRTCStatus.fail) {
      this._parameters.close && this._parameters.close(0);
      throw {code: webRTCStatus.fail.code, message: webRTCStatus.fail.message};
    }

    let webRTCException = null;
    try {
      await timeout(this.start(), this._connectTimeout);
      webRTCStatus.time = Date.now();
      webRTCStatus.fail = null;
    }
    catch (e: any) {
      let message = e.message;
      if (e instanceof TimeoutError) {
        message = __("Failed to open local stream: Connection timed out");
        e.message = message;
        this._logger.warn(message);
      }
      webRTCException = e;

      webRTCStatus.time = 0;
      webRTCStatus.fail = {
        code: e.code,
        message
      };
    }

    this.setWebRTCStatus(avatarId, webRTCStatus);

    if (webRTCException) {
      throw webRTCException;
    }

    return true;
  }

  setParameters(parameters: TransportInitParameters) {
    if (parameters.obj && parameters.obj !== this._parameters.obj) {
      this._avatarId = null;
    }

    this._parameters = Object.assign(this._parameters, parameters);
  }

  isPreviousWebRTCSuccess(avatarId: UUID) {
    const webRTCStatus = this.getWebRTCStatus(avatarId);
    return Date.now() - webRTCStatus.time < this._checkConnectionTimeout;
  }

  getWebRTCStatus(avatarId: UUID): WebRTCStatus {
    const status = localStorage.getItem(`local_connection_status:${avatarId}`);

    let localConnectionStatus: WebRTCStatus = {
      time: 0
    };
    try {
      if (status) {
        localConnectionStatus = JSON.parse(status);
      }
    }
    catch (e) {}

    return localConnectionStatus;
  }

  setWebRTCStatus(avatarId: UUID, webRTCStatus: WebRTCStatus) {
    const localConnectionStatus = JSON.stringify(webRTCStatus);
    localStorage.setItem(`local_connection_status:${avatarId}`, localConnectionStatus);
  }

  async getAvatarId(obj: UUID | null = this._parameters.obj ?? null): Promise<string | null> {
    if (this._avatarId !== null) {
      return this._avatarId;
    }

    let avatarId: UUID | null = null;
    try {
      const response = await this._api.getAttributes({obj: obj});
      avatarId = response.list.AVATARID;
    }
    catch (e) {
      if (e instanceof AjaxError) {
        this._logger.error(e.message);
      }

      return null;
    }

    this._avatarId = avatarId;

    return this._avatarId;
  }

  send(data: object): boolean {
    if (!this._channel) {
      this._logger.error("this._channel is undefined");
      return false;
    }

    if (!this.isOpen()) {
      this._logger.error("channel is " + this._channel.readyState);
      return false;
    }

    const dataJSON = JSON.stringify(data);

    this._logger.log("send local>", dataJSON);

    isUseFileTransport && this._fileTransport.send(data);

    this._requestLog.push(dataJSON);
    if (this._requestLog.length > this._requestLogMaxSize) {
      this._requestLog.shift();
    }

    this._channel.send(dataJSON);

    return true;
  }

  /**
   *
   * @returns {Array}
   */
  getLog() {
    return this._requestLog;
  }

  /**
   *
   * @returns {boolean}
   */
  isOpen() {
    return !!(this._channel && this._channel.readyState === "open");
  }

  /**
   * @param {boolean} [isCallback]
   */
  close(isCallback = true) {
    this._logger.log("WebRTCMediaTransport close");

    if (!isCallback) {
      this._isCloseCallback = false;
    }

    clearInterval(this._heartBeatInterval);

    isUseFileTransport && this._fileTransport.close(isCallback);

    if (this._peerConnection && this._peerConnection.iceConnectionState !== "closed") {
      this._peerConnection.close();
      return;
    }

    if (
      !this._isCloseCallbackCalled
      && this._peerConnection
      && this._peerConnection.iceConnectionState === "closed"
    ) {
      this._isCloseCallbackCalled = true;
      this._isCloseCallback && this._parameters.close && this._parameters.close(0);
    }
  }

  async sendOffer() {
    if (!this._peerConnection) {
      this._logger.error("this._peerConnection is undefined");
      return;
    }

    try {
      const description = await this._peerConnection.createOffer();
      // this._logger.log(description.sdp);

      this._peerConnection.setLocalDescription(description);
    }
    catch (e: any) {
      this._logger.error("Failed to create/set session description", e.message);
    }
  }

  start(): Promise<void> {
    this._logger.log("Starting up RTCPeerConnection...");

    return new Promise((resolve, reject) => {
      this._peerConnection = new RTCPeerConnection();

      this._peerConnection.addEventListener("iceconnectionstatechange", (e) => {
        if (!e.target) {
          return;
        }
        const state = (e.target as RTCPeerConnection).iceConnectionState;
        this._logger.log("RTCPeerConnection iceconnectionstatechange", state);
        if (state === "disconnected" || state === "failed" || state === "closed") {
          this.close();
          reject({message: `iceConnectionState ${state}`});
        }
      });
      this._peerConnection.addEventListener("icegatheringstatechange", (e) => this._logger.log("RTCPeerConnection icegatheringstatechange"));
      this._peerConnection.addEventListener("removestream", (e) => this._logger.log("RTCPeerConnection removestream"));
      this._peerConnection.addEventListener("signalingstatechange", (e) => {
        this._logger.log("RTCPeerConnection signalingstatechange", (e.target as RTCPeerConnection).signalingState);
      });
      this._peerConnection.addEventListener("datachannel", (e) => this._logger.log("RTCPeerConnection connectionstatechange"));
      this._peerConnection.addEventListener("message", (e) => this._logger.log("RTCPeerConnection message"));
      this._peerConnection.addEventListener("open", (e) => this._logger.log("RTCPeerConnection open"));
      this._peerConnection.addEventListener("tonechange", (e) => this._logger.log("RTCPeerConnection tonechange"));
      this._peerConnection.addEventListener("identityresult", (e) => this._logger.log("RTCPeerConnection identityresult"));
      this._peerConnection.addEventListener("peeridentity", (e) => this._logger.log("RTCPeerConnection peeridentity"));
      this._peerConnection.addEventListener("isolationchange", (e) => this._logger.log("RTCPeerConnection isolationchange"));

      this._peerConnection.addEventListener("close", (e) => this._logger.log("RTCPeerConnection close"));
      this._peerConnection.addEventListener("error", (e) => this._logger.log("RTCPeerConnection error", e));
      this._peerConnection.addEventListener("idpassertionerror", (e) => this._logger.log("RTCPeerConnection idpassertionerror", e));
      this._peerConnection.addEventListener("idpvalidationerror", (e) => this._logger.log("RTCPeerConnection idpvalidationerror", e));

      this._peerConnection.addEventListener("icecandidate", async (evt) => {
        this._logger.log("RTCPeerConnection icecandidate");

        if (!this._peerConnection) {
          this._logger.log("icecandidate: this._peerConnection is undefined");
          return;
        }

        if (!this._peerConnection.localDescription) {
          this._logger.log("icecandidate: this._peerConnection.localDescription is undefined");
          return;
        }

        let candidate = evt.candidate;
        // Chrome sends a null candidate once the ICE gathering phase completes.
        // In this case, it makes sense to send one copy-paste blob.
        if (null == candidate) {
          this._logger.log("Finished gathering ICE candidates.");
          const sdp = this._peerConnection.localDescription.sdp;

          // let sdpList = localDescription.sdp.split('\n');
          // const sdp = sdpList
          //   .filter(function (row) {
          //     return !/^a=candidate:.*tcp.*/i.test(row);
          //   })
          //  .join('\n');
          // this._logger.log(sdp);

          let receive = async (sdp: string) => {
            if (!this._peerConnection) {
              this._logger.log("icecandidate, receive: this._peerConnection is undefined");
              return;
            }

            try {
              let sessionDescription = new RTCSessionDescription({
                sdp: sdp,
                type: "answer"
              });
              await this._peerConnection.setRemoteDescription(sessionDescription);
              this._logger.log("setRemoteDescription done");
            }
            catch (e: any) {
              // TODO: add check
              this._logger.error("Invalid SDP message.", e.message);
            }
          };

          try {
            const response = await this._api.getSDP({
              obj: this._parameters.obj,
              sdp: sdp
            });
            await receive(response.sdp);
          }
          catch (e) {
            reject(e);

            if (e instanceof AjaxError) {
              console.error(e.message);
            }
          }
        }
      });

      this._peerConnection.addEventListener("negotiationneeded", () => {
        this._logger.log("RTCPeerConnection negotiationneeded");
        this.sendOffer();
      });

      // Creating the first data channel triggers ICE negotiation.
      // set channel name as objid;live/archive;coma separated ip addresses from sdp
      const channelName = `${this._parameters.obj};${this._parameters.isArchive ? "archive" : "live"};`;
      this._channel = this._peerConnection.createDataChannel(channelName);
      this._channel.binaryType = "arraybuffer";
      this._channel.addEventListener("open", () => {
        this._logger.log('RTCDataChannel open');

        resolve();

        this._parameters.open && this._parameters.open();
        this._heartbeat();
      });
      this._channel.addEventListener("close", (e) => {
        this._logger.log('RTCDataChannel close');

        this.close();
      });
      this._channel.addEventListener("error", (e) => {
        // @ts-ignore
        this._logger.log('RTCDataChannel error', e.error.message);

        this._parameters.error && this._parameters.error();
      });
      this._channel.addEventListener("message", (message) => {
        isUseFileTransport && this._fileTransport.message(message);

        if (message.data instanceof ArrayBuffer) {
          this._debugData && this._logger.log("chunk> " + message.data.byteLength + " bytes");
        } else {
          this._debugData && this._logger.log("onmessage> ", message.data);
        }

        this._parameters.message && this._parameters.message(message);
      });
      this._channel.addEventListener("bufferedamountlow", (e) => {
        this._logger.log('RTCDataChannel bufferedamountlow');
      });
    });
  }

  _heartbeat(): void {
    this._heartBeatInterval = window.setInterval(() => this.send({}), 5000);
  }
}
