import React, {useEffect, useMemo, useRef} from "react";
import {CameraCellParameters, PlayerCell} from "@solid/player-cell/cell";
import { WidgetId, useRealmForPlayerQuery } from "@solid/client/src/generated/graphql";
import type {PlayerCommandArgs, PlayerEventArgs, PlayerProps} from "@solid/client/src/core/playerTypes";
import {PlayerCommand, PlayerEvent, toPlayerMetadataDrawType} from "@solid/client/src/core/playerTypes";
import type {WidgetEventArgs} from "@solid/client/src/components/Widgets";
import  { useAccessability } from "@solid/client/src/core/store/actions/accessability";
import { getTimelineStore } from "@solid/client/src/components/Timeline/timelineStore";
import { isElectron } from "@solid/libs/utils";

import "vis-timeline/styles/vis-timeline-graph2d.css";
import "@solid/timeline/vistimeline.css";
import "@solid/player/mediaplayer.css";
import "@solid/player/mediaplayercontrols.css";
import "@solid/joystick/ptz.css";
import "@solid/player-cell/digitalzoom.css";
import "@bevacqua/rome/dist/rome.css";
import "@solid/player-cell/cell.css";
import "./PlayerCellLegacy.css";

type PlayerCellLegacyProps = PlayerProps;

const PlayerCellLegacy = React.forwardRef((props: PlayerCellLegacyProps, ref: React.Ref<HTMLDivElement>) => {
  const { data: realmData, loading: realmLoading, error: realmError } = useRealmForPlayerQuery({fetchPolicy: "network-only"});
  const { config: { limitedSpeedList } } = useAccessability({ widgetId: WidgetId.Timeline });
  const playerCellNodeRef = !ref || typeof ref === "function" ? useRef<HTMLDivElement>(null) : ref;
  const cameraCellRef = useRef<PlayerCell | null>(null);

  const idleTimeout: number | undefined = useMemo(() => {
    return realmData?.realm?.playerIdleTimeout ?? 0;
  }, [realmData]);

  const jitterBufferLength: number | undefined = useMemo(() => {
    if (!isElectron()) return undefined;
    return realmData?.realm?.jitterBufferLength ?? undefined;
  }, [realmData]);

  const initCell = () => {
    if (typeof ref === "function") {
      ref(playerCellNodeRef.current);
    }

    if (!playerCellNodeRef.current || !props.obj || realmLoading) {
      return;
    }

    cameraCellRef.current = new PlayerCell();

    const { time: timeLineWidgetTime, isPlay: timeLineWidgetIsPlay, connectionType: preferredConnectionType } = getTimelineStore();

    const parameters: CameraCellParameters = {
      isShowHeader: !props.hideHeader,
      isTimeLineVisible: props.stepBackControl,
      isControlsHoverable: !props.isArchive,
      isPlayOnStart: props.playOnStart, // TODO: check
      isPTZVisible: props.ptzControls,
      isDigitalZoomVisible: props.digitalZoom, // TODO: check
      isFullScreenControlVisible: props.fullScreenControl,
      metadataType: props.metadataType ? toPlayerMetadataDrawType(props.metadataType) : undefined,
      connectionType: preferredConnectionType,
      isShowPauseButton: props.showPauseButton,
      disableLocalTransport: props.disableLocalTransport,
      idleTimeout,
      limitedSpeedList: !!limitedSpeedList,
      onChangeMode: (mode, time) => {
        props.onChangeMode?.(mode);
        publishEvent(PlayerEvent.Mode, { mode, time })
      },
      onAddDownloadJob: props.onAddDownloadJob,
      onArchiveLengthChange: props.onArchiveLengthChange,
      onGeoPosition: (geo) => publishEvent(PlayerEvent.GeoPosition, { geo }),
      muted: props.muted,
      jitterBufferLength
    };

    let time: number | undefined;
    if (props.isArchive && props.time) {
      time = props.time.getTime();
    } else
    if (props.isArchive && timeLineWidgetTime) {
      time = timeLineWidgetTime;
      parameters.isPlayOnStart = timeLineWidgetIsPlay;
    } else
    if (props.isArchive && !props.time) {
      time = Date.now() - 10 * 60 * 1000; // 5 min
    }

    cameraCellRef.current.init(
      playerCellNodeRef.current,
      props.obj,
      props.header,
      time,
      parameters,
      {
        onplay: () => {
          publishEvent(PlayerEvent.Play);
        },
        onstop: () => {
          publishEvent(PlayerEvent.Stop)
        },
        onpause: (byUser) => {
          publishEvent(byUser ? PlayerEvent.PausedByUser : PlayerEvent.Pause)
        },
        onframe: (time) => {
          publishEvent(PlayerEvent.Frame, { time })
        },
        onclose: () => props.onClose && props.onClose(),
        onaoddone : () => publishEvent(PlayerEvent.AODDone),
        onadddownloadjob: (start) => props.onAddDownloadJob && props.onAddDownloadJob(start),
        onarchivelengthchange: (hour) => props.onArchiveLengthChange && props.onArchiveLengthChange(hour)
      }
    );
  };

  const destroyCell = () => {
    if (!cameraCellRef.current) {
      return;
    }

    cameraCellRef.current && cameraCellRef.current.destroy();
    cameraCellRef.current = null;
  };

  useEffect(() => {
    if (!playerCellNodeRef.current) {
      return;
    }

    initCell();

    return () => {
      destroyCell();
    };
  }, [props.obj, props.disableLocalTransport, props.time, idleTimeout, jitterBufferLength, realmLoading]);

  useEffect(() => {
    realmError && console.error("Can not get realm for player", realmError);
  }, [realmError])

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    props.widgetEvent?.subscribe(id, onWidgetEvent);

    return function cleanup() {
      props.widgetEvent?.unsubscribe(id);
    }
  }, [props.widgetEvent]);

  const onWidgetEvent = (args: WidgetEventArgs): void => {
    const cameraCell = cameraCellRef.current;
    if (!cameraCell) {
      return;
    }

    const cmdArgs = args.args as PlayerCommandArgs;
    if (!cmdArgs.widgetIndices?.includes(props.widgetIndex ?? -1) || !cmdArgs.objs?.includes(props.obj)) {
      return;
    }

    switch (args.event) {
      case PlayerCommand.Play:
        const time = cmdArgs.time ? cmdArgs.time.getTime() : undefined;
        const speed = cmdArgs.speed;
        if (typeof speed !== "undefined") {
          cameraCell.getModel().setSpeed(speed);
        }
        cameraCell.getModel().play(props.obj, time);
        break;

      case PlayerCommand.Pause:
        const byUser = cmdArgs.byUser ?? false;
        cameraCell.getModel().pause(byUser);
        break;

      case PlayerCommand.FullScreen:
        playerCellNodeRef.current?.requestFullscreen();
        break;

      case PlayerCommand.PreferredConnectionType:
        if (cmdArgs.preferredConnectionType) {
          cameraCell.getModel().setPreferredConnectionType(cmdArgs.preferredConnectionType);
        }
        break;

      case PlayerCommand.SetTime:
        if (cmdArgs.time) {
          console.log(cmdArgs.time, cmdArgs.byUser, cameraCell.getModel().isPlay);
          cameraCell.getModel().setTime(cmdArgs.time.getTime(), cmdArgs.byUser, cameraCell.getModel().isPlay);
        }
        break;

      case PlayerCommand.SetSpeed:
        if (cmdArgs.speed) {
          cameraCell.getModel().setSpeed(cmdArgs.speed, cameraCell.getModel().isPlay);
          cameraCell.getModel().timeline.updateData();
        }
        break;

      case PlayerCommand.GetAOD:
        const { deviceId, start, duration } = cmdArgs;
        if (deviceId && start && duration && props.obj === deviceId) {
          cameraCell.getModel().getAOD(deviceId, start, duration);
        }
        break;
      case PlayerCommand.MuteAudio:
        const { muted } = cmdArgs;
        if (typeof muted === "boolean") {
          cameraCell.getPlayer().muteAudio(muted);
        }
        break;
    }
  }

  const publishEvent = (event: PlayerEvent, extraArgs: Partial<PlayerEventArgs> = {}): void => {
    const args: PlayerEventArgs = { obj: props.obj, widgetIndex: props.widgetIndex ?? -1, ...extraArgs };
    props.widgetEvent?.publish({ event, args });
  };

  useEffect(() => {
    if (props.metadataType) {
      const player = cameraCellRef.current?.getPlayer();
      if (player) {
        player.setMetadataDrawType(toPlayerMetadataDrawType(props.metadataType));
      }
    }
  }, [props.metadataType]);

  return (
    <div ref={playerCellNodeRef} className="cell_wrapper"/>
  );
});

export default PlayerCellLegacy;
