// Copyright (c) 2015-2017 David M. Lee, II

/**
 * Exception indicating that the timeout expired.
 */
export class TimeoutError extends Error {
  constructor(message?: string) {
    message = message ?? "Timeout";
    super(message); // 'Error' breaks prototype chain here
    this.name = 'TimeoutError';
    Object.setPrototypeOf(this, new.target.prototype); // restore prototype chain
  }
}

/**
 * Rejects a promise with a {@link TimeoutError} if it does not settle within
 * the specified timeout.
 *
 * @param {Promise} promise The promise.
 * @param {number} timeoutMillis Number of milliseconds to wait on settling.
 */
export const timeout = function <T>(promise: Promise<T>, timeoutMillis: number): Promise<T> {
  let timeout: NodeJS.Timeout;

  return Promise.race([
    promise,
    new Promise<T>(function (resolve, reject) {
      timeout = setTimeout(function () {
        reject(new TimeoutError());
      }, timeoutMillis);
    }),
  ]).then(function (v) {
    clearTimeout(timeout);
    return v;
  }, function (err) {
    clearTimeout(timeout);
    throw err;
  });
};
