import {AjaxError, alignBorder, alignTimestampToChunk, getOrigin, ptsToTimestamp, Utils} from "../utils";
import {AODAPI, AODAPIGetRequest, AODAPIGetResponse, AODStatus} from "./aod";
import {AlertLogAPI} from "../elogapi";
import {API} from "../api";
import {Log} from "../log";
import $ from "jquery";
import type {StreamObjectList, TimeLineData, TimeLineDataList, TimeStamp, UUID} from "@solid/types";
import {CHUNK_SIZE, Granularity, MSEMediaPlayerStream} from "@solid/types";
import fs from "fs";
import path from "path";
import {isFileExists} from "../file";

type CoverageAPIResult = {
  code: number,
  error: string,
  avatar_coverage: { [avatarId in UUID]: StreamObjectList<TimeStamp[]> },
  partial_coverage: { [avatarId in UUID]: StreamObjectList<TimeStamp[]> },
  coverage: { [avatarId in UUID]: StreamObjectList<TimeStamp[]> },
  ttl_hours: { [avatarId in UUID]: number }
}

type CoverageResult = {
  list: TimeLineData[],
  avatar_coverage: {},
  ttl_hours?: CoverageAPIResult["ttl_hours"]
}

type EventLegacy = {
  eventId: number,
  from: TimeStamp,
  to: TimeStamp,
  when: TimeStamp,
  object: {
    objId: UUID,
    name: string,
    type: string,
    subtype: string
  },
  propertyList: [],
  witnessList: {
    objId: UUID,
    name: string,
    type: string,
    subtype: string
  }[],
  actionList: [],
  mediasize: {
    exportclip: number,
    storage: number
  },
  source: number,
  state: number,
  priority: number,
  message: string,
  eventtype: number,
  note: [],
  tagList: number[],
  history: string,
  lastModified: number
}

const elogapi = new AlertLogAPI();

let roleId: UUID | undefined;
const isGetOfflineCoverage = false;

/**
 *
 * @param obj
 * @param startTime unix timestamp ms
 * @param endTime unix timestamp ms
 * @param granularity
 * @param streamNumber
 * @param dataDir
 * @param withLow
 */
export async function getCoverage(obj: UUID, startTime: TimeStamp, endTime: TimeStamp, granularity: Granularity = Granularity.chunk, streamNumber: number = -1, dataDir: string = "", withLow: boolean = false): Promise<CoverageResult> {
  endTime = Math.min(endTime, Date.now());

  let origin = getOrigin();

  let list: TimeLineData[] = [];
  let api = new API();
  let coverageResponse: CoverageAPIResult | undefined;
  let offlineCoverage: TimeLineDataList = {};
  let getAODResponse: AODAPIGetResponse | undefined;
  let eventResponse: EventLegacy[] | undefined;
  try {
    if (!roleId) {
      const response = await api.getInfo();
      roleId = response.info.role.obj;
    }

    [coverageResponse, eventResponse, getAODResponse, offlineCoverage] = await Promise.all([
      Utils.promisifyAjax($.ajax({
        url: `${origin}/api/cm/coverage`,
        type: "GET",
        dataType: "json",
        cache: false,
        data: {
          obj: obj,
          startTime: Math.round(startTime / 1000),
          endTime: Math.round(endTime / 1000),
          granularity: granularity,
          streamNumber: streamNumber
        }
      })),
      /*(new Promise((resolve) => {resolve({})})),*/
      Utils.promisifyAjax(elogapi.getEventList({
        roleid: roleId,
        filter: {
          filter: {
            objid: [obj],
            started: {
              minValue: startTime,
              maxValue: endTime
            }
          },
          output: {timeOnly: true}
        }
      })),
      AODAPI.get({ objid: obj, startTs: Math.round(startTime / 1000), endTs: Math.round(endTime / 1000) }),
      getOfflineCoverage(obj, startTime, endTime, dataDir)
    ].map(p => p.catch(e => e)));
  }
  catch (e) {
    if (e instanceof AjaxError) {
      Log.error(e.message);
    } else {
      throw e;
    }
    return {
      list: list,
      avatar_coverage: {},
      ttl_hours: coverageResponse?.ttl_hours
    };
  }

  let aodList = getAODResponse?.requests ?? [];

  let avatarCoverage = coverageResponse?.avatar_coverage ?? {};
  let partialCoverage = coverageResponse?.partial_coverage ?? {};
  let coverage = coverageResponse?.coverage ?? {};

  let timestampList = obj === Object.keys(coverage)[0] ? coverage[obj] : [];
  let partialTimestampList = obj === Object.keys(partialCoverage)[0] ? partialCoverage[obj] : [];

  let eventList = Array.isArray(eventResponse) ? eventResponse : [];

  if (granularity === Granularity.chunk) {
    const splitAodByResolution = (aodList: AODAPIGetRequest[]): StreamObjectList<AODAPIGetRequest[]> => {
      let result: StreamObjectList<AODAPIGetRequest[]> = {};
      aodList.forEach(aodReq => {
        const {streamNum} = aodReq;
        if (result[streamNum] === undefined) {
          result[streamNum] = [aodReq];
        } else {
          result[streamNum].push(aodReq);
        }
      });
      return result;
    };
    const filteredAodList = splitAodByResolution(aodList);

    let split = function (start: TimeStamp, end: TimeStamp, step: number): TimeStamp[] {
      let list: number[] = [];

      for (let timeStamp = start; timeStamp < end - step; timeStamp += step) {
        if (timeStamp % step !== 0) {
          continue;
        }

        list.push(timeStamp);
      }

      return list;
    };

    let getRow = function (timeStampList: TimeStamp[], className: string, color?: number, borderColor?: number, partialTimestampList?: TimeStamp[], partialGradient?: string): TimeLineData {
      let row: TimeLineData = {
        list: {},
        granularity: granularity
      };

      if (!timeStampList || timeStampList.length === 0) {
        return row;
      }

      let min, max;
      min = max = timeStampList[0];
      for (let i = 0; i < timeStampList.length; i++) {
        let timestamp = timeStampList[i];
        row.list[timestamp + '000'] = {
          isChunk: true,
          percentage: 100,
          className,
          color: color,
          borderColor: borderColor
        };

        if (timestamp < min) {
          min = timestamp;
        }
        if (timestamp > max) {
          max = timestamp;
        }
      }
      if (Array.isArray(partialTimestampList) && !!partialGradient) {
        for (let i = 0; i < partialTimestampList.length; i++) {
          let timestamp = partialTimestampList[i];
          row.list[timestamp + '000'] = {
            isPartialChunk: true,
            percentage: 100,
            className: "partial-chunk",
            gradient: partialGradient,
            borderColor: borderColor
          };

          if (timestamp < min) {
            min = timestamp;
          }
          if (timestamp > max) {
            max = timestamp;
          }
        }
      }

      let start = new Date();
      start.setTime(min * 1000);
      let end = new Date();
      end.setTime(max * 1000);
      // console.log("set data " + start.getHours() + ":" + start.getMinutes() + ":" +
      // start.getSeconds() + " " + end.getHours() + ":" + end.getMinutes() + ":" + end.getSeconds());

      return row;
    };

    let getEventRow = function (timeStampList: EventLegacy[], className: string, color?: number, borderColor?: number): TimeLineData {
      let row: TimeLineData = {
        list: {},
        granularity: Granularity.second,
        isEvent: true
      };

      if (timeStampList.length === 0) {
        return row;
      }

      let min: TimeStamp;
      let max: TimeStamp;
      min = max = timeStampList[0].when;
      for (let i = 0; i < timeStampList.length; i++) {
        let timestamp = timeStampList[i].when;
        row.list[timestamp] = {
          isEvent: true,
          percentage: 100,
          className,
          color: color,
          borderColor: borderColor
        };

        if (timestamp < min) {
          min = timestamp;
        }
        if (timestamp > max) {
          max = timestamp;
        }
      }

      let start = new Date();
      start.setTime(min * 1000);
      let end = new Date();
      end.setTime(max * 1000);
      // console.log("set data " + start.getHours() + ":" + start.getMinutes() + ":" +
      // start.getSeconds() + " " + end.getHours() + ":" + end.getMinutes() + ":" + end.getSeconds());

      return row;
    };

    let aodCombine = (aodList: TimeStamp[], row: TimeLineData): TimeLineData => {
      Object.values(aodList).forEach((timestamp) => {
        if (row.list[timestamp + '000'] === undefined) {
          row.list[timestamp + '000'] = {
            isAOD: true,
            percentage: 100,
            className: "aod-chunk"
          };
        }
      });

      return row;
    };

    let offlineCombine = (offlineList: TimeLineDataList, row: TimeLineData): TimeLineData => {
      Object.keys(offlineList).forEach((timestamp) => {
        row.list[timestamp] = offlineList[timestamp];
      });

      return row;
    };

    /*
    0 - black, nothing
    1 - light grey, content POTENTIALLY present (avatar, date-from)
    2 - orange, active download
    3 - blue, low-res
    4 - green, high-res
    5 - red, event lines
    */

    let step = 30; // sec
    const genAodTimestampList = (aodList: AODAPIGetRequest[]): TimeStamp[] => {
      if (aodList) {
        return aodList
          // .filter((row) => row.status !== AODStatus.COMPLETED)
          .filter((row) => ![AODStatus.CANCELED, AODStatus.FAILED].includes(row.status))
          .reduceRight<TimeStamp[]>((prevRow, row) => {
            let endDate = new Date(row.endTs * 1000);
            endDate.setUTCSeconds(endDate.getUTCSeconds() <= 29 ? 29 : 59, 0);

            let startTs = Math.round(alignTimestampToChunk(row.startTs * 1000) / 1000);
            let endtTs = Math.round(endDate.getTime() / 1000);
            return prevRow.concat(split(startTs, endtTs, step));
          }, []);
      }
      return [];
    };

    // low resolution, Blue
    if (withLow) {
      list.push(aodCombine(
        genAodTimestampList(filteredAodList["0"]),
        getRow(timestampList[0], "low-resolution-chunk"))
      );
    } else {
      list.push({list: {}, granularity: Granularity.chunk});
    }

    /*
    const localCoverage = isElectron() ? await FileTransport.getCoverage(obj) : {};
    const localCoverageList = Object.keys(localCoverage).map((row) => { return Number(row) / 1000; });

    list.push(aodCombine(
      [],
      getRow(localCoverageList, "low-resolution-chunk"))
    );
    */

    // high resolution, Green
    // partial delivered chunks high resolution, Green with Yellow stripes
    list.push(offlineCombine(offlineCoverage ?? {}, aodCombine(
      genAodTimestampList(filteredAodList[1]),
      getRow(timestampList[1], "high-resolution-chunk", undefined, undefined, partialTimestampList[1]),
    )));

    // events
    list.push(getEventRow(eventList, "event"));
  } else {
    let row = {
      list: {},
      granularity
    };
    for (let timestamp in timestampList) {
      row.list[timestamp + '000'] = {
        percentage: timestampList[timestamp],
        color: 0x83382f,
        borderColor: 0xcd2813
      };
    }
    list.push(row);
  }

  return {
    // 0 - low resolutions
    // 1 - high resolution
    // 2 - events
    list: list,
    avatar_coverage: avatarCoverage,
    ttl_hours: coverageResponse?.ttl_hours
  };
}

/**
 *
 * @param obj
 * @param startTime unix timestamp ms
 * @param endTime unix timestamp ms
 * @param dataDir
 */
export async function getOfflineCoverage(obj: UUID, startTime: TimeStamp, endTime: TimeStamp, dataDir: string): Promise<TimeLineDataList> {
  if (!isGetOfflineCoverage) {
    return {};
  }

  endTime = Math.min(endTime, Date.now());

  // const time = performance.now();

  const list: TimeLineDataList = {};

  if (!dataDir) {
    return list;
  }

  try {
    let cameraIdList = await fs.promises.readdir(dataDir, {encoding: "utf8"});
    cameraIdList = cameraIdList.filter((cameraId) => {
      return cameraId === obj;
    });
    if (cameraIdList.length === 0) {
      throw new Error(`obj=${obj} not found in ${dataDir}`);
    }

    const {start, end} = alignBorder({startTime, endTime});
    const startDate = new Date(start);
    const endDate = new Date(end);

    const cameraId = cameraIdList[0];

    const yearMonthDateList = await fs.promises.readdir(path.join(dataDir, cameraId), {encoding: "utf8"});
    for (const yearMonthDate of yearMonthDateList) {
      // 211119
      const match = yearMonthDate.match(/^(\d{2})(\d{2})(\d{2})$/);
      if (!match) {
        continue;
      }

      const [, year, month, date] = match;
      const storageDate = new Date();
      storageDate.setUTCFullYear(Number(`20${year}`), Number(month) - 1, Number(date));
      storageDate.setUTCHours(0, 0, 0, 0);

      const yearMonthStart = new Date(startDate);
      yearMonthStart.setUTCHours(0, 0, 0, 0);
      const yearMonthEnd = new Date(endDate);
      yearMonthEnd.setUTCHours(0, 0, 0, 0);

      if (yearMonthStart.getTime() <= storageDate.getTime() && storageDate.getTime() <= yearMonthEnd.getTime()) {
        const hourStreamList = await fs.promises.readdir(path.join(dataDir, cameraId, yearMonthDate), {encoding: "utf8"});
        for (const hourStream of hourStreamList) {
          // 09-01
          const match = hourStream.match(/^(\d{2})-(\d{2})$/);
          if (!match) {
            continue;
          }

          const [, hour, stream] = match;
          storageDate.setUTCHours(Number(hour));

          const hourStreamStart = new Date(startDate);
          hourStreamStart.setUTCMinutes(0, 0, 0);
          const hourStreamEnd = new Date(endDate);
          hourStreamEnd.setUTCMinutes(0, 0, 0);

          if (stream === "01"
              && hourStreamStart.getTime() <= storageDate.getTime() && storageDate.getTime() <= hourStreamEnd.getTime()
          ) {
            const chunkList = await fs.promises.readdir(path.join(dataDir, cameraId, yearMonthDate, hourStream), {encoding: "utf8"});
            for (const chunk of chunkList) {
              // 211119095230.mov or 211119095230.mov.tmp
              const match = chunk.match(/^(\d+)\.mov(\.tmp)?$/);
              if (match) {
                const [, pts, tmp] = match;
                const timestamp = ptsToTimestamp(`20${pts}`);

                if (!(start <= timestamp && timestamp <= end)) {
                  continue;
                }

                if (tmp) {
                  /*
                  list[timestamp] = {
                    className: "offline-partial-chunk",
                    percentage: 100,
                    isPartialChunk: true
                  };
                  */
                } else {
                  list[timestamp] = {
                    className: "offline-chunk",
                    percentage: 100,
                    isOffline: true
                  };
                }
              }
            }
          }
        }
      }
    }

    // console.log(`getCoverage ${performance.now() - time}ms`);
  }
  catch (e: any) {
    console.warn(`getCoverage ${e.message}`);
  }

  return list;
}

/**
 * check if timestamp in chunk list
 */
export function isChunkExist(timestamp: TimeStamp, list: TimeLineData[], streamId: MSEMediaPlayerStream, isOffline?: boolean): boolean {
  const streamIndex = streamId === MSEMediaPlayerStream.DIGEST ? 0 : 1;

  if (!list || !list[streamIndex]) {
    return false;
  }

  const stream = list[streamIndex];

  const alignedTimestamp = alignTimestampToChunk(timestamp);
  return alignedTimestamp in stream.list
         && (!isOffline || !!stream.list[alignedTimestamp].isOffline);
}

/**
 * check if timestamp is exists in offline store
 */
export async function isOfflineChunkExist(cameraId: UUID, timestamp: TimeStamp/*, direction: 1 | -1*/, dataDir: string): Promise<boolean> {
  if (!dataDir) {
    return false;
  }

  const date = new Date(alignTimestampToChunk(timestamp));
  const yearMonthDate = `${date.getUTCFullYear() % 100}${String(date.getUTCMonth() + 1).padStart(2, "0")}${String(date.getUTCDate()).padStart(2, "0")}`;
  const hourStream = `${String(date.getUTCHours()).padStart(2, "0")}-01`;
  const chunkFileName = `${yearMonthDate}${String(date.getUTCHours()).padStart(2, "0")}${String(date.getUTCMinutes()).padStart(2, "0")}${String(date.getUTCSeconds()).padStart(2, "0")}.mov`;
  // g:\solid_data\b77112fc-2528-11ec-9173-00155dd9270b\211209\17-01\211209173230.mov
  const chunkPath = path.join(dataDir, cameraId, yearMonthDate, hourStream, chunkFileName);
  const isExists = await isFileExists(chunkPath);
  console.log(isExists, chunkPath);
  return isExists;
}

export function searchTimestamp(timestamp: TimeStamp, list: TimeStamp[]): TimeStamp {
  if (list.length > 0) {
    for (let i = 0; i < list.length; i++) {
      const time = list[i];
      if (timestamp < time) {
        return -1;
      }
      if (time <= timestamp && timestamp < time + CHUNK_SIZE) {
        return i;
      }
    }
  }

  return -1;
}
