/**
 * based on virtualjoystick.js
 *
 * MIT License
 *
 * https://github.com/jeromeetienne/virtualjoystick.js
 *
 * opts.container is the dom element on which we display joystick
 * opts.stickElement is the dom element which is display for the stick of the joystick.
 * opts.baseElement is the dom element which is display for its base.
 * Both elements are optional with sensible default
 * you may set opts.mouseSupport to true during debug.
 * you may set opts.stationaryBase to true for a permanent Stationary joystick base.
 * if you do use a stationary base, you must also set opts.baseX to the desired X-coordinate on the webpage and opts.baseY to the desired Y-coordinate. The joystick base will now be fixed at this location.
 * you may set opts.limitStickTravel to true in order to limit the distance that the stick can travel from its base. This will create an invisible circle barrier that the stick cannot leave.
 * if you do use opts.limitStickTravel , you can also set opts.stickRadius to the desired radius (in pixels). The stick will now be confined to stickRadius. If you do not set opts.stickRadius , it will default to 100 pixels radius.
 *
 * Example:
 VirtualJoystick({
		  container	: document.getElementById("container"),
		  mouseSupport: true,
		  stationaryBase: true,
		  baseX: 200,
		  baseY: 200,
		  limitStickTravel: true,
		  stickRadius: 50
		});
 *
 */

import { Utils } from "@solid/libs";
import { MicroEvent } from "./microevent";

type TransformStyleList =
  "webkitTransform"
  | "MozTransform"
  | "msTransform"
  | "OTransform"
  | "transform";

export class VirtualJoystick extends MicroEvent {
  private _container: Element;
  private _strokeStyle: string;
  private _stickEl: HTMLCanvasElement;
  private _baseEl: HTMLCanvasElement;
  private _stationaryBase: boolean;
  private _mouseSupport: boolean;
  private _baseX: number;
  private _baseY: number;
  private _limitStickTravel: boolean;
  private _stickRadius: number;
  private _useCssTransform: boolean;
  private _stickX: number;
  private _stickY: number;
  private _pressed: boolean;
  private _touchIdx: number | null;
  private _transform: TransformStyleList | undefined;
  private _has3d: boolean = false;
  private _$onTouchStart?: EventListenerOrEventListenerObject;
  private _$onTouchEnd?: EventListenerOrEventListenerObject;
  private _$onTouchMove?: EventListenerOrEventListenerObject;
  private _$onMouseDown?: EventListenerOrEventListenerObject;
  private _$onMouseUp?: EventListenerOrEventListenerObject;
  private _$onMouseMove?: EventListenerOrEventListenerObject;
  private _$resizeObserver?: ResizeObserver;

  constructor(opts: {
    useCssTransform?: boolean,
    baseElement?: HTMLCanvasElement,
    stickElement?: HTMLCanvasElement,
    container?: Element,
    strokeStyle?: string,
    limitStickTravel?: boolean,
    baseX?: number,
    stickRadius?: number,
    mouseSupport?: boolean,
    baseY?: number,
    stationaryBase?: boolean
  }) {
    super();

    opts = opts || {};
    this._container = opts.container || document.body;
    this._strokeStyle = opts.strokeStyle || "cyan";
    this._stickEl = opts.stickElement || this._buildJoystickStick();
    this._baseEl = opts.baseElement || this._buildJoystickBase();
    this._mouseSupport = opts.mouseSupport !== undefined ? opts.mouseSupport : false;
    this._stationaryBase = opts.stationaryBase || false;
    this._baseX = this._stickX = opts.baseX || 0;
    this._baseY = this._stickY = opts.baseY || 0;
    this._limitStickTravel = opts.limitStickTravel || false;
    this._stickRadius = opts.stickRadius !== undefined ? opts.stickRadius : 100;
    this._useCssTransform = opts.useCssTransform !== undefined ? opts.useCssTransform : false;

    //this._container.style.position = "relative"

    this._container.appendChild(this._baseEl);
    this._baseEl.style.position = "absolute";
    this._baseEl.style.display = "none";

    this._container.appendChild(this._stickEl);
    this._stickEl.style.position = "absolute";
    this._stickEl.style.display = "none";
    this._stickEl.style.left = "50%";
    this._stickEl.style.top = "50%";

    this._pressed = false;
    this._touchIdx = null;

    this._$resizeObserver = new ResizeObserver(Utils.throttleToDraw(() => this._resize()));
  }

  init() {
    if (this._stationaryBase) {
      this._baseEl.style.display = "";
      this._baseEl.style.left = (this._baseX - this._baseEl.width / 2) + "px";
      this._baseEl.style.top = (this._baseY - this._baseEl.height / 2) + "px";
    }

    this._transform = this._useCssTransform ? this._getTransformProperty() : undefined;
    this._has3d = this._check3D();

    const __bind = function (fn: Function, me: any) {
      return function () {
        return fn.apply(me, arguments);
      };
    };
    this._$onTouchStart = __bind(this._onTouchStart, this);
    this._$onTouchEnd = __bind(this._onTouchEnd, this);
    this._$onTouchMove = __bind(this._onTouchMove, this);
    this._container.addEventListener("touchstart", this._$onTouchStart, false);
    this._container.addEventListener("touchend", this._$onTouchEnd, false);
    this._container.addEventListener("touchmove", this._$onTouchMove, false);
    if (this._mouseSupport) {
      this._$onMouseDown = __bind(this._onMouseDown, this);
      this._$onMouseUp = __bind(this._onMouseUp, this);
      this._$onMouseMove = __bind(this._onMouseMove, this);
      this._container.addEventListener("mousedown", this._$onMouseDown, false);
      this._container.addEventListener("mouseup", this._$onMouseUp, false);
      this._container.addEventListener("mouseleave", this._$onMouseUp, false);
      this._container.addEventListener("mousemove", this._$onMouseMove, false);
      this._$resizeObserver?.observe(this._container);
    }
  }

  destroy() {
    this._container.removeChild(this._baseEl);
    this._container.removeChild(this._stickEl);

    this._$onTouchStart && this._container.removeEventListener("touchstart", this._$onTouchStart, false);
    this._$onTouchEnd && this._container.removeEventListener("touchend", this._$onTouchEnd, false);
    this._$onTouchMove && this._container.removeEventListener("touchmove", this._$onTouchMove, false);
    if (this._mouseSupport) {
      this._$onMouseUp && this._container.removeEventListener("mouseup", this._$onMouseUp, false);
      this._$onMouseDown && this._container.removeEventListener("mousedown", this._$onMouseDown, false);
      this._$onMouseMove && this._container.removeEventListener("mousemove", this._$onMouseMove, false);
      this._$resizeObserver && this._$resizeObserver.unobserve(this._container);
    }
  }

  /**
   * @returns {Boolean} true if touchscreen is currently available, false otherwise
   */
  static touchScreenAvailable = function () {
    return "createTouch" in document;
  }

  //////////////////////////////////////////////////////////////////////////////////
  //										//
  //////////////////////////////////////////////////////////////////////////////////

  deltaX() {
    return this._stickX - this._baseX;
  }

  deltaY() {
    return this._stickY - this._baseY;
  }

  up() {
    if (!this._pressed) return false;
    const deltaX = this.deltaX();
    const deltaY = this.deltaY();
    if (deltaY >= 0) return false;
    if (Math.abs(deltaX) > 2 * Math.abs(deltaY)) return false;
    return true;
  }

  down() {
    if (!this._pressed) return false;
    const deltaX = this.deltaX();
    const deltaY = this.deltaY();
    if (deltaY <= 0) return false;
    if (Math.abs(deltaX) > 2 * Math.abs(deltaY)) return false;
    return true;
  }

  right() {
    if (!this._pressed) return false;
    const deltaX = this.deltaX();
    const deltaY = this.deltaY();
    if (deltaX <= 0) return false;
    if (Math.abs(deltaY) > 2 * Math.abs(deltaX)) return false;
    return true;
  }

  left() {
    if (!this._pressed) return false;
    const deltaX = this.deltaX();
    const deltaY = this.deltaY();
    if (deltaX >= 0) return false;
    if (Math.abs(deltaY) > 2 * Math.abs(deltaX)) return false;
    return true;
  }

  //////////////////////////////////////////////////////////////////////////////////
  //										//
  //////////////////////////////////////////////////////////////////////////////////

  _resize() {
    if (this._stationaryBase) {
      const baseX = this._container.scrollWidth / 2;
      const baseY = this._container.scrollHeight / 2;
      this._baseX = baseX;
      this._baseY = baseY;
      this._stickX = baseX;
      this._stickY = baseY;
      this._baseEl.style.left = (baseX - this._baseEl.width / 2) + "px";
      this._baseEl.style.top = (baseY - this._baseEl.height / 2) + "px";
    }
  }

  _onUp() {
    this._pressed = false;
    this._stickEl.style.display = "none";

    if (this._stationaryBase) {
      this._stickX = this._baseX;
      this._stickY = this._baseY;
    } else {
      this._baseEl.style.display = "none";

      this._baseX = this._baseY = 0;
      this._stickX = this._stickY = 0;
    }
  }

  _onDown(x: number, y: number) {
    this._pressed = true;
    if (!this._stationaryBase) {
      this._baseX = x;
      this._baseY = y;
      this._baseEl.style.display = "";
      this._move(this._baseEl.style, (this._baseX - this._baseEl.width / 2), (this._baseY - this._baseEl.height / 2));
    }

    this._stickX = x;
    this._stickY = y;

    if (this._limitStickTravel) {
      const deltaX = this.deltaX();
      const deltaY = this.deltaY();
      const stickDistance = Math.sqrt((deltaX * deltaX) + (deltaY * deltaY));
      if (stickDistance > this._stickRadius) {
        const stickNormalizedX = deltaX / stickDistance;
        const stickNormalizedY = deltaY / stickDistance;

        this._stickX = stickNormalizedX * this._stickRadius + this._baseX;
        this._stickY = stickNormalizedY * this._stickRadius + this._baseY;
      }
    }

    this._stickEl.style.display = "";
    this._move(this._stickEl.style, (this._stickX - this._stickEl.width / 2), (this._stickY - this._stickEl.height / 2));
  }

  _onMove(x: number, y: number) {
    if (this._pressed) {
      this._stickX = x;
      this._stickY = y;

      if (this._limitStickTravel) {
        const deltaX = this.deltaX();
        const deltaY = this.deltaY();
        const stickDistance = Math.sqrt((deltaX * deltaX) + (deltaY * deltaY));
        if (stickDistance > this._stickRadius) {
          const stickNormalizedX = deltaX / stickDistance;
          const stickNormalizedY = deltaY / stickDistance;

          this._stickX = stickNormalizedX * this._stickRadius + this._baseX;
          this._stickY = stickNormalizedY * this._stickRadius + this._baseY;
        }
      }

      this._move(this._stickEl.style, (this._stickX - this._stickEl.width / 2), (this._stickY - this._stickEl.height / 2));
    }
  }

  //////////////////////////////////////////////////////////////////////////////////
  //		bind touch events (and mouse events for debug)			//
  //////////////////////////////////////////////////////////////////////////////////

  _onMouseUp(event: MouseEvent) {
    return this._onUp();
  }

  _onMouseDown(event: MouseEvent) {
    event.preventDefault();

    const rect = this._container.getBoundingClientRect();
    const x = event.clientX - rect.left;
    const y = event.clientY - rect.top;

    return this._onDown(x, y);
  }

  _onMouseMove(event: MouseEvent) {
    const rect = this._container.getBoundingClientRect();
    const x = event.clientX - rect.left;
    const y = event.clientY - rect.top;
    return this._onMove(x, y);
  }

  //////////////////////////////////////////////////////////////////////////////////
  //		comment								//
  //////////////////////////////////////////////////////////////////////////////////

  _onTouchStart(event: TouchEvent) {
    // if there is already a touch in progress do nothing
    if (this._touchIdx !== null) return;

    // notify event for validation
    const isValid = this.dispatchEvent("touchStartValidation", event);
    if (isValid === false) return;

    // dispatch touchStart
    this.dispatchEvent("touchStart", event);

    event.preventDefault();
    // get the first who changed
    const touch = event.changedTouches[0];
    // set the touchIdx of this joystick
    this._touchIdx = touch.identifier;

    // forward the action
    const x = touch.pageX;
    const y = touch.pageY;
    return this._onDown(x, y);
  }

  _onTouchEnd(event: TouchEvent) {
    // if there is no touch in progress, do nothing
    if (this._touchIdx === null) return;

    // dispatch touchEnd
    this.dispatchEvent("touchEnd", event);

    // try to find our touch event
    const touchList = event.changedTouches;
    for (var i = 0; i < touchList.length && touchList[i].identifier !== this._touchIdx; i++) {
    }
    // if touch event isn"t found,
    if (i === touchList.length) return;

    // reset touchIdx - mark it as no-touch-in-progress
    this._touchIdx = null;

    //??????
    // no preventDefault to get click event on ios
    event.preventDefault();

    return this._onUp();
  }

  _onTouchMove(event: TouchEvent) {
    // if there is no touch in progress, do nothing
    if (this._touchIdx === null) return;

    // try to find our touch event
    const touchList = event.changedTouches;
    for (var i = 0; i < touchList.length && touchList[i].identifier !== this._touchIdx; i++) {
    }
    // if touch event with the proper identifier isn"t found, do nothing
    if (i === touchList.length) return;
    const touch = touchList[i];

    event.preventDefault();

    const x = touch.pageX;
    const y = touch.pageY;
    return this._onMove(x, y);
  }

  //////////////////////////////////////////////////////////////////////////////////
  //		build default stickEl and baseEl				//
  //////////////////////////////////////////////////////////////////////////////////

  /**
   * build the canvas for joystick base
   */
  _buildJoystickBase() {
    const canvas = document.createElement("canvas");
    canvas.width = 70;
    canvas.height = 70;

    const ctx = canvas.getContext("2d");
    if (ctx) {
      ctx.beginPath();
      ctx.strokeStyle = this._strokeStyle;
      ctx.lineWidth = 6;
      ctx.arc(canvas.width / 2, canvas.width / 2, 20, 0, Math.PI * 2, true);
      ctx.stroke();

      ctx.beginPath();
      ctx.strokeStyle = this._strokeStyle;
      ctx.lineWidth = 2;
      ctx.arc(canvas.width / 2, canvas.width / 2, 40, 0, Math.PI * 2, true);
      ctx.stroke();
    }
    return canvas;
  }

  /**
   * build the canvas for joystick stick
   */
  _buildJoystickStick() {
    const canvas = document.createElement("canvas");
    canvas.width = 70;
    canvas.height = 70;
    const ctx = canvas.getContext("2d");
    if (ctx) {
      ctx.beginPath();
      ctx.strokeStyle = this._strokeStyle;
      ctx.lineWidth = 6;
      ctx.arc(canvas.width / 2, canvas.width / 2, 20, 0, Math.PI * 2, true);
      ctx.stroke();
    }
    return canvas;
  }

  //////////////////////////////////////////////////////////////////////////////////
  //		move using translate3d method with fallback to translate > "top" and "left"
  //      modified from https://github.com/component/translate and dependents
  //////////////////////////////////////////////////////////////////////////////////

  _move(style: CSSStyleDeclaration, x: number, y: number) {
    if (this._transform) {
      if (this._has3d) {
        style[this._transform] = "translate3d(" + x + "px," + y + "px, 0)";
      } else {
        style[this._transform] = "translate(" + x + "px," + y + "px)";
      }
    } else {
      style.left = x + "px";
      style.top = y + "px";
    }
  }

  _getTransformProperty(): TransformStyleList | undefined {
    const styles: TransformStyleList[] = [
      "webkitTransform",
      "MozTransform",
      "msTransform",
      "OTransform",
      "transform"
    ];

    const el = document.createElement("p");
    let style;

    for (let i = 0; i < styles.length; i++) {
      style = styles[i];
      if (null != el.style[style]) {
        return style;
      }
    }

    return undefined;
  }

  _check3D() {
    const prop = this._getTransformProperty();

    // IE8<= doesn"t have `window.getComputedStyle`
    if (!prop || !window.getComputedStyle) {
      return module.exports = false;
    }

    const map = {
      webkitTransform: "-webkit-transform",
      OTransform: "-o-transform",
      msTransform: "-ms-transform",
      MozTransform: "-moz-transform",
      transform: "transform"
    };

    // from: https://gist.github.com/lorenzopolidori/3794226
    const el = document.createElement("div");
    el.style[prop] = "translate3d(1px,1px,1px)";
    document.body.insertBefore(el, null);
    const val = window.getComputedStyle(el).getPropertyValue(map[prop]);
    document.body.removeChild(el);
    const exports = null != val && !!val.length && "none" != val;
    return exports;
  }
}
