/**
 * @version $Id$
 * ------------------------------------------------------------------------------
 * Logic for communicate with hardware joystick
 * Joystick - object with methods for creating applet and communication with him
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @author Sergey Koloney
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 *
 */

import { __ } from "@solid/libs/i18n";
import { Log } from "@solid/libs/log";
import { API } from "@solid/libs/api";
import { UUID } from "@solid/types";
import type {
  Ptz_PresetInput,
  Ptz_SpeedInput,
  Ptz_StepFocus,
  Ptz_StepGain,
  Ptz_StepInput,
  Ptz_StepIris,
  Ptz_StepMove,
  Ptz_StepZoom,
  PtzPresetMutationVariables,
  PtzSpeedMutationVariables,
  PtzStepMutationVariables,
  PtzRelMutationVariables
} from "@solid/graphql";
import { clone, getOrigin, Utils } from "@solid/libs/utils";

type JoystickParameters = {
  name: string,
  axis: {[axisName: number]: string},
  button: {[buttonName: string]: string}
}

type AxisState = {
  [axisName: number]: number
}

type AxisDeadZoneInformation = {
  [axisName: number]: number
}

export enum Ptz_Mode {
  STEP = "ptzStep",
  SPEED = "ptzSpeed",
  PRESET = "ptzPreset",
  REL = "ptzRel"
}

enum Ptz_Input {
  STEP = "PTZ_StepInput!",
  SPEED = "PTZ_SpeedInput!",
  PRESET = "PTZ_PresetInput!",
  REL = "PTZ_RelInput!"
}

type SpeedCommand = {
  name: keyof Ptz_SpeedInput,
  value: number | number[]
} | undefined

type StepCommand = {
  name : keyof Ptz_StepInput,
  value: Ptz_StepMove | Ptz_StepFocus | Ptz_StepGain | Ptz_StepIris | Ptz_StepZoom
}

 type PresetCommand = {
  name: keyof Ptz_PresetInput,
  value: number
 }

 type PtzMutationVariables = PtzStepMutationVariables | PtzSpeedMutationVariables | PtzPresetMutationVariables | PtzRelMutationVariables;

class Joystick {
  obj?: UUID;
  priority?: number;
  lock_incheck = false;
  userid = null;
  commandPreset: any = null;
  speed = 1; // current move speed, can be changed by the sensitivity slider
  axisState: AxisState = { "0": 0, "1": 0 };
  zoomAxis = 3;  // Zoom axis; needed to quickly find value of zoom axis when moving x or y to provide simultaneous zooming and moving

  /* Speed commands support */
  speedCommandDelay = 500;   // Delay between two consecutive calls of sendModeSpeedCommand
  minDelayBetween2Command = 100;   // minimum delay between two consecutive calls of sendModeSpeedCommand
  isCommandPending = false; // pending speed command flag
  lastCommandSent = 0;       // time when last ajax command was sent to ptz server
  timerSendSpeedCommand: NodeJS.Timeout | null = null;

  isDebug = false;

  defaultPreset = {
    button: {
      "0": "mode=step&zoom=in",
      "1": "mode=step&zoom=out",
      "2": "mode=preset&goto=1",
      "3": "mode=preset&goto=2",
      "4": "mode=preset&goto=3",
      "5": "mode=preset&goto=4",
      "6": "mode=preset&goto=5",
      "7": "mode=preset&goto=6",
      "8": "mode=preset&goto=7",
      "9": "mode=preset&goto=8",
      "10": "mode=preset&goto=9",
      "11": "",
      "12": "",
      "99": "" // do not remove
    },
    axis: {
      "1": "-tilt",
      "3": "z"
    },
    deadzone: {
      "default": 5
    }
  };

  /**
   * list of all command presets available
   * @var {Object}
   */
  commandPresets = {};

  /**
   * HWJ parameters
   * @var {Object}
   */
  parameters: JoystickParameters = { name: "", axis: {}, button: {} };

  controller?: AbortController;

  throttlePtz = Utils.throttle((mode: Ptz_Mode, command: SpeedCommand | StepCommand | PresetCommand) => {this.ptz(mode, command)}, this.speedCommandDelay)


  /**
   * initialization of HWJ
   */
  init() {
    // "hwjload", self.onHWJLoad
    // "axis", self.onAxis
    // "hat", self.onHat
    // "button", self.onButton
    // "enddeadzonecalibrate", self.onEndDeadZoneCalibrate
    // "log", self.onLog

    this.onHWJLoad(true, this.parameters);
  }

  /**
   * load command presets from db, check current joystick name and set appropriate command preset
   */
  async loadCommandPresets() {
    try {
      const api = new API();
      const response = await api.getIdentityAttribute({
        attribute: 'JOYSTICK_PRESETS'
      })
      if (response.value) {
        this.commandPresets = JSON.parse(response.value);
      }
    }
    finally {
      // get joystick name and search it in command presets
      const info = this.getInfo();
      let preset = this.defaultPreset;
      if (this.commandPresets[info.name]) {
        preset = this.commandPresets[info.name];
      }
      this.setCommandPreset(preset);
    }
  }

  /**
   * Set current preset for joystick commands
   */
  setCommandPreset(commandPreset: {}) {
    // Clone object
    this.commandPreset = clone(commandPreset);

    const info = this.getInfo();
    for (const axisName in info.axis) {
      const axis = Number(axisName);
      if (!info.axis.hasOwnProperty(axis)) {
        continue;
      }

      // const value = 0; // info.axis[i].value

      // set speed value accordingly to current axis position if sensitivity assigned for some axis
      if (this.commandPreset.axis[axis] && this.commandPreset.axis[axis].indexOf('sensitivity') >= 0) {
        this.setSpeed(this.commandPreset.axis[axis], this.axisState[axis]);
      }

      // Find zooming axis
      if (this.commandPreset.axis[axis] && this.commandPreset.axis[axis].indexOf('mode=speed&zoom=') >= 0) {
        this.zoomAxis = axis;
        this.axisState[this.zoomAxis] = 0;
      }
    }

    // if preset contains DeadZone settings, send them to applet
    if (this.commandPreset.deadzone) {
      this.setDeadZone(this.commandPreset.deadzone);
    }
  }

  /**
   * remove HWJ from page
   */
  destroy() {
  }

  /**
   * get joystick info
   */
  getInfo(): JoystickParameters {
    return this.parameters;
  }

  /**
   * set obj of PTZ device in system
   */
  async setObj(obj: UUID): Promise<void> {
    this.obj = obj;
    if (obj) {
      this.priority = await this.getPTZPriority();
    }
  }

  /**
   *
   * @returns {Promise<number>}
   */
  async getPTZPriority() {
    let priority = 1;

    let api = new API();
    const getObjectListResponse = await api.getObjectList({
      type: 'role',
      withAttributes: true
    });

    for (let i = 0; i < getObjectListResponse.list.length; i++) {
      const { value } = await api.getAttribute({
        obj: getObjectListResponse.list[i].obj,
        attribute: 'PTZ_PRIORITY'
      });
      if (value > priority) {
        priority = value;
      }
    }

    return priority;
  }

  /**
   * Send info about deadzone to applet
   *
   * @param axis data for dead zone, ex. {0: 0.1, 1: 0.05, 2: 0}
   */
  setDeadZone(axis: AxisDeadZoneInformation) {
    /*
    if (this.hwj) {
      this.hwj.setDeadZone(axis);
    }
    */
  }

  /**
   * Set speed of joystick accordingly to value of sensitivity axis
   *
   * @param sens sensitivity|-sensitivity, how to compute speed
   * @param value value of axis
   */
  setSpeed(sens: string, value: number) {
    // Reversed sensitivity
    if (sens == "-sensitivity") {
      value = -value;
    }

    // axis returns values -100..100, we need values 0.1 .. 1
    value = ((value + 100) / 200);
    if (value < 0.1) value = 0.1;
    this.speed = value;
  }

  /**
   * send PTZ command to server, and call callback function after server response or error
   */
  async ptz(mode: Ptz_Mode, command: SpeedCommand | StepCommand | PresetCommand, priority?: number) {
    if (!this.obj) {
      throw new Error("obj should be defined");
    }
    if (!command) {
      throw new Error("command should be defined");
    }

    const input: PtzSpeedMutationVariables | PtzPresetMutationVariables | PtzStepMutationVariables  = {
      id: this.obj,
      command: {
        [command.name]: command.value
      }
    }
    if (priority) {
      input["priority"] = priority;
    }

    try {
      const response = await this.ptzMutation(mode, input);
      if (response) {
        const data = await response.json();
        // check priority lock
        if (data.lockinfo && data.lockinfo.allow_override != undefined) {
          this.checkLock(data.lockinfo.allow_override);
        }
      }
    } catch (e) {
      throw e;
    } finally {
      this.isCommandPending = false;
    }
  }

  async gotoXY(x: number, y: number, width: number, height: number) {
    if (!this.obj) {
      throw new Error("obj should be defined");
    }

    const input: PtzRelMutationVariables = {
      id: this.obj,
      command: {
        xy: [x, y],
        size: { width, height }
      }
    }

    if (this.priority) {
      input["priority"] = this.priority;
    }

    try {
      const response = await this.ptzMutation(Ptz_Mode.REL, input);
      if (response) {
        const data = await response.json();
        // check priority lock
        if (data.lockinfo && data.lockinfo.allow_override != undefined) {
          this.checkLock(data.lockinfo.allow_override);
        }
      }
    } catch (e) {
      throw e;
    } finally {
      this.isCommandPending = false;
    }
    // await this.sendCommand("mode=rel&size=" + width + 'x' + height + "&xy=" + x + ',' + y);
  }

  async sendCommand(command: string, addDev?: boolean, priority?: number) {
    addDev = typeof addDev == "undefined" ? true : !!addDev;
    priority = priority || this.priority;

    if (addDev && !(this.obj && command)) {
      throw new Error("obj and command should be defined");
    }

    // mode=speed&pt=0,0
    this.isCommandPending = true;
    this.lastCommandSent = Date.now();

    const devString = addDev ? "cam_objid=" + this.obj + "&" : "";
    command = devString + command + "&priority=" + priority;
    command = command.replace(/--/g, ''); // delete '--' to provide "reverse" behaviour

    if (this.isDebug) {
      console.log("Send: " + command);
    }

    /*
    console.log(command);
    Joystick.isCommandPending = false;
    */

    let origin = getOrigin();
    let url = `${origin}/api/ptz?${command}`;

    if (this.controller) {
      this.controller.abort();
      delete this.controller;
    }

    try {
      this.controller = new AbortController();
      const response = await fetch(new Request(url), {signal: this.controller.signal});
      const data = await response.json();
      // check priority lock
      if (data.lockinfo && data.lockinfo.allow_override != undefined) {
        this.checkLock(data.lockinfo.allow_override);
      }
    }
    catch (e: any) {
      if (e.name !== 'AbortError') {
        console.error(`${e.message()}`);
      }
    }
    finally {
      this.isCommandPending = false;
    }
  }



  /**
   * check lock
   *
   * @param lockOverride lock override state
   */
  checkLock(lockOverride: string) {
    if (this.lock_incheck) {
      return;
    }

    this.lock_incheck = true;

    switch (lockOverride) {
      case "sameuser":
        this.lockDevice();
        break;
      case "no":
        Log.info(__("Another lock is higher priority. You have to wait until controls are released. Please try again later."));
        break;
      case "yes":
        const getLock = confirm(__("Another lock is in place. Do you want to override?"));
        if (getLock) {
          this.overrideLock();
          this.lockDevice();
        }
        break;
    }

    this.lock_incheck = false;
  }

  lockDevice() {

  }

  /**
   * override lock
   */
  async overrideLock() {
    await this.sendCommand("mode=lock&cmd=override");
  }

  /**
   * Send speed command; when user use axis, this function calls every speedCommandDelay milliseconds
   * until user releases joystick.
   * Every 5 iterations it checks joystick position to prevent endless movement if we skipped "stop" command for some reason.
   *
   * Next speed command can be sent only if:
   * 1) speedCommandDelay milliseconds passed since last command was sent;
   * 2) last ajax command was completed
   *
   * @param command    speed command
   * @param iteration  number of function iteration
   */
  sendModeSpeedCommand(speedCommand: SpeedCommand, iteration: number) {
    if (!this.obj) {
      return;
    }

    /*
    // Once in a 10 iterations check current position of axis to prevent endless movement
    // (if last event from joystick with axis=0 was lost)
    // TODO: check code and remove it
    if (iteration > 5)
    {
      const info = Joystick.getInfo();
      for (const i in info.axis)
      {
        if (!info.axis.hasOwnProperty(i))
          continue;

        // If axis state changed send event
        if (Joystick.axisState[info.axis[i].name] && Joystick.axisState[info.axis[i].name] != info.axis[i].value)
        {
          Joystick.onAxis(info.axis[i].name, info.axis[i].value);
        }
      }
      return;
    }
    */

    // if previous command completed,
    // and
    // delay time passed or this new command
    // we can send another command
    if (!this.isCommandPending
      //&&
      //(Date.now() - Joystick.lastCommandSent > Joystick.minDelayBetween2Command)
    ) {
      // this.ptz(Ptz_Mode.SPEED, speedCommand);
      this.throttlePtz(Ptz_Mode.SPEED, speedCommand);
    }

    this.stopSendSpeedCommand();
    this.timerSendSpeedCommand = setTimeout(
      () => {
        this.sendModeSpeedCommand(speedCommand, iteration + 1 );
      },
      this.speedCommandDelay
    );
  }

  /**
   * Clear sendModeSpeedCommand timeout
   */
  stopSendSpeedCommand() {
    if (this.timerSendSpeedCommand !== null) {
      clearTimeout(this.timerSendSpeedCommand);
    }
  }

  /**
   * Function is called when user use joystick axis
   *
   * @param axis   name of axis
   * @param value  value of axis
   */
  onAxis(axis: number, value: number) {
    if (this.isDebug) {
      console.log(axis, value);
    }
    this.axisState[axis] = value;

    // Check for reversed Y behaviour
    if (axis == 1 && this.commandPreset.axis[1] && this.commandPreset.axis[1] == '-tilt') {
      this.axisState[axis] = -value;
    }

    let isStop = false;
    let speedCommand: SpeedCommand;

    if (axis == 0 || axis == 1) { // for ptz command: || axis === Joystick.zoomAxis
      speedCommand = {
        name: "pt",
        value: [Math.round(this.axisState["0"] * this.speed), Math.round(this.axisState["1"] * this.speed)]
      };
      // for ptz command: Math.round(Joystick.axisState[Joystick.zoomAxis] * Joystick.speed);

      if (this.axisState["0"] == 0 && this.axisState["1"] == 0) // for ptz command: && Joystick.axisState[Joystick.zoomAxis] === 0
      {
        isStop = true;
      }
    } else
    if (this.commandPreset.axis[axis]) {
      // Sensitivity command
      if (this.commandPreset.axis[axis].indexOf("sensitivity") != -1) {
        this.setSpeed(this.commandPreset.axis[axis], value);
        return;
      }

      speedCommand = {
        name: this.commandPreset.axis[axis],
        value: Math.round(this.axisState[axis] * this.speed)
      }

      if (this.axisState[axis] == 0) {
        isStop = true;
      }
    }

    if (isStop) {
      this.stopSendSpeedCommand();
      // this.ptz(Ptz_Mode.SPEED, speedCommand);
      this.throttlePtz(Ptz_Mode.SPEED, speedCommand);
    } else {
      this.sendModeSpeedCommand(speedCommand, 1);
    }
  }

  onHat(hat: string, value: number) {
  }

  /**
   * Function is called when user click on joystick button
   *
   * @param button joystick button name
   * @param value
   */
  onButton(button: string, value: number) {
    if (value == 1 && this.commandPreset.button[button]) {
      this.sendCommand(this.commandPreset.button[button]);
    }
  }

  onEndDeadZoneCalibrate(parameters: {}) {
  }

  onLog(code: number, message: string) {
    switch (code) {
      case 0:
        Log.info(message);
        break;
      case 1:
        Log.warning(message);
        break;
    }
  }

  onLoad(success: boolean) {

  }

  async onHWJLoad(success: boolean, parameters: JoystickParameters) {
    if (success) {
      // Set default command preset
      this.setCommandPreset(this.defaultPreset);

      this.parameters = parameters;

      // Load presets
      await this.loadCommandPresets()
      this.onLoad(success);
    }
  }
  private async ptzMutation(name: Ptz_Mode, variable: PtzMutationVariables) {
    const ptzInput = this.getPtzInput(name);

    if(!ptzInput) {
      Log.error("Control command is undefined.");
      return;
    }

    this.isCommandPending = true;
    this.lastCommandSent = Date.now();

    if (this.controller) {
      this.controller.abort();
      delete this.controller;
    }

    if (this.isDebug) {
      console.log("Send: " + variable.command);
    }

    const query = {
      operationName: null,
      variables: {
        id: variable.id,
        command: variable.command,
      },
      query: `mutation ($id: ID!, $command: ${ptzInput}, $priority: Int) {${name}(id: $id, input: $command, priority: $priority)}`
    }

    this.controller = new AbortController();

    let origin = getOrigin();
    const response = await fetch(`${origin}/api/graphql`, {
      headers: { "Content-Type": "application/json" },
      method: 'POST',
      body: JSON.stringify(query),
      signal: this.controller.signal
    });

    return response;
  }

  private getPtzInput(name: Ptz_Mode) {
    switch (name) {
      case Ptz_Mode.PRESET:
        return Ptz_Input.PRESET;
      case Ptz_Mode.SPEED:
        return Ptz_Input.SPEED;
      case Ptz_Mode.STEP:
        return Ptz_Input.STEP;
      case Ptz_Mode.REL:
        return Ptz_Input.REL
      default:
        return undefined;
    }
  }
}

export default Joystick;
