type DigitalZoomInitParameters = {
  node: HTMLCanvasElement | HTMLVideoElement,
  metaDataNode: HTMLDivElement,
  onzoomin?: () => void,
  onzoomout?: () => void,
  width: number,
  height: number
}

export class DigitalZoom {
  controls: DigitalZoomImplementation | null = null;
  enabled = false;
  disable = false;

  switchZoom() {
    this.enabled = !this.enabled;
    if (this.enabled) {
      if (this.controls !== null) {
        this.controls.enable();
      }
    } else {
      this.destroy();
    }
  }

  destroy() {
    if (this.controls !== null) {
      this.controls.returnToInitialState();
      this.controls.removeHandlers();
      this.controls.removeStyle();
      this.controls.eventElement?.remove();
    }
    this.enabled = false;
  }

  init(params: DigitalZoomInitParameters): boolean {
    if (!params.node) {
      this.destroy();
      return false;
    }
    if (this.controls && this.controls.element) {
      if (params.node.isSameNode(this.controls.element)) {
        return false;
      } else {
        this.destroy();
      }
    }

    let digitalZoom = new DigitalZoomImplementation();

    digitalZoom.metaDataNode = params.metaDataNode;

    digitalZoom.element = params.node;
    if (typeof params.onzoomin === 'function') {
      digitalZoom.onzoomin = params.onzoomin;
    }
    if (typeof params.onzoomout === 'function') {
      digitalZoom.onzoomout = params.onzoomout;
    }

    digitalZoom.streamWidth = params.width;
    digitalZoom.streamHeight = params.height;

    digitalZoom.maxWidth = digitalZoom.streamWidth * digitalZoom.maxZoomFactor;
    digitalZoom.maxHeight = digitalZoom.streamHeight * digitalZoom.maxZoomFactor;

    this.disable = digitalZoom.calculateRect();

    if (this.enabled && !this.disable) {
      digitalZoom.enable();
    }

    if (!this.disable) {
      this.controls = digitalZoom;
    } else {
      this.destroy();
      return false;
    }

    return true;
  }

  changeSize() {
    if (this.controls !== null) {
      this.controls.returnToInitialState();
      this.disable = this.controls.calculateRect();
    }
  }

  /**
   * show and draw mini map
   */
  showMiniMap() {
    // TODO: add implementation
  }

  /**
   * hide mini map
   */
  hideMiniMap() {
    // TODO: add implementation
  }
}

class DigitalZoomImplementation {
  deltaWidth = 0;
  deltaHeight = 0;

  metaDataNode: HTMLDivElement | null = null;
  eventElement: HTMLElement | null = null;
  element: HTMLElement | null = null;
  elementRect: DOMRect | null = null;

  parentRect: DOMRect | null = null;

  isMouseDown = false;

  translateX = 0;
  translateY = 0;

  lastTranslateX = 0;
  lastTranslateY = 0;

  clickX = 0;
  clickY = 0;

  zoomStep = 0.1;
  zoomFactor = 1;

  maxZoomFactor = 2;
  maxWidth = 0;
  maxHeight = 0;

  maxTranslateX = 0;
  maxTranslateY = 0;
  minTranslateX = 0;
  minTranslateY = 0;

  bodyEvents = false;

  streamWidth = 0;
  streamHeight = 0;

  //handlers
  onMouseMove = (e: MouseEvent) => {
    if (this.zoomFactor !== 1 && this.isMouseDown) {
      this.translateX = this.lastTranslateX + e.x - this.clickX;
      this.translateY = this.lastTranslateY + e.y - this.clickY;

      this.applyTransform && this.applyTransform();
    }
  }

  onMouseLeaveOrUp = () => {
    if (this.element) {
      this.element.removeEventListener('mousemove', this.onMouseMove);
    }
    if (this.eventElement) {
      this.eventElement.removeEventListener('mousemove', this.onMouseMove);
    }
    document.body.removeEventListener('mousemove', this.onMouseMove);
    document.body.removeEventListener('mouseup', this.onMouseLeaveOrUp);
    document.body.removeEventListener('mouseleave', this.onMouseLeaveOrUp);

    this.lastTranslateX = this.translateX;
    this.lastTranslateY = this.translateY;

    this.bodyEvents = false;
  }

  onMouseUp = () => {
    this.isMouseDown = false;

    if (this.element) {
      this.element.removeEventListener('mousemove', this.onMouseMove);
    }
    if (this.eventElement) {
      this.eventElement.removeEventListener('mousemove', this.onMouseMove);
    }
  }

  onMouseDown = (e: MouseEvent) => {
    this.clickX = e.x;
    this.clickY = e.y;
    this.isMouseDown = true;

    if (this.element) {
      this.element.addEventListener('mousemove', this.onMouseMove);
    }

    if (this.eventElement) {
      this.eventElement.addEventListener('mousemove', this.onMouseMove);
    }

    document.body.addEventListener('mouseup', this.onMouseLeaveOrUp);
    document.body.addEventListener('mouseleave', this.onMouseLeaveOrUp);

    this.bodyEvents = true;
  }

  onMouseWheel = (e: WheelEvent | Event) => {
    e.preventDefault();
    e.stopPropagation();

    let step = (e instanceof WheelEvent) && (e.detail < 0 || e.deltaY < 0) ? this.zoomStep : -this.zoomStep;
    if (this.zoomFactor === 1 && step < 0) {
      return;
    }

    if (typeof this.onzoomin === 'function' && this.zoomFactor === 1 && step > 0) {
      this.onzoomin();
    }
    if (typeof this.onzoomout === 'function' && this.zoomFactor + step === 1) {
      this.onzoomout();
    }

    if (this.elementRect
        && this.parentRect
        && this.elementRect.width * (this.zoomFactor + step) <= this.maxWidth && this.elementRect.height * (this.zoomFactor + step) <= this.maxHeight
    ) {
      this.zoomFactor += step;

      if (this.zoomFactor > 1) {
        this.minTranslateX = -(((this.parentRect.width - this.deltaWidth) * this.zoomFactor - this.parentRect.width) / 2);
        this.minTranslateY = -(((this.parentRect.height - this.deltaHeight) * this.zoomFactor - this.parentRect.height) / 2);

        this.maxTranslateX = (((this.parentRect.width - this.deltaWidth) * this.zoomFactor - this.parentRect.width) / 2);
        this.maxTranslateY = (((this.parentRect.height - this.deltaHeight) * this.zoomFactor - this.parentRect.height) / 2);

        this.maxTranslateX = this.maxTranslateX < 0 ? 0 : this.maxTranslateX;
        this.maxTranslateY = this.maxTranslateY < 0 ? 0 : this.maxTranslateY;

        this.applyTransform();
      } else {
        this.returnToInitialState();
      }
    }
  }

  applyTransform() {
    if (this.translateX < this.minTranslateX) {
      this.translateX = this.minTranslateX;
    }
    if (this.translateX > this.maxTranslateX) {
      this.translateX = this.maxTranslateX;
    }
    if (this.translateY < this.minTranslateY) {
      this.translateY = this.minTranslateY;
    }
    if (this.translateY > this.maxTranslateY) {
      this.translateY = this.maxTranslateY;
    }

    const transform = 'translate(' + this.translateX + 'px, ' + this.translateY + 'px) scale(' + this.zoomFactor + ')';

    if (this.element) {
      this.element.style.transform = transform;
    }
    if (this.eventElement) {
      this.eventElement.style.transform = transform;
    }
    if (this.metaDataNode) {
      this.metaDataNode.style.transform = transform;
    }
  }

  removeHandlers() {
    if (this.element) {
      this.element.removeEventListener('mousedown', this.onMouseDown);
      this.element.removeEventListener('mouseup', this.onMouseUp);
      this.element.removeEventListener('DOMMouseScroll', this.onMouseWheel);
      this.element.removeEventListener('wheel', this.onMouseWheel);
    }

    if (this.eventElement) {
      this.eventElement.removeEventListener('mousedown', this.onMouseDown);
      this.eventElement.removeEventListener('mouseup', this.onMouseUp);
      this.eventElement.removeEventListener('DOMMouseScroll', this.onMouseWheel);
      this.eventElement.removeEventListener('wheel', this.onMouseWheel);
    }

    if (this.bodyEvents) {
      document.body.removeEventListener('mousemove', this.onMouseMove);
      document.body.removeEventListener('mouseup', this.onMouseLeaveOrUp);
      document.body.removeEventListener('mouseleave', this.onMouseLeaveOrUp);
    }
  }

  removeStyle() {
    if (this.element) {
      this.element.style.removeProperty('transformOrigin');
      this.element.style.removeProperty('transform');
      this.element.style.removeProperty('cursor');
      this.element.style.removeProperty('userSelect');
      this.element.style.removeProperty('-moz-user-select');
    }

    if (this.eventElement) {
      this.eventElement.style.removeProperty('transformOrigin');
      this.eventElement.style.removeProperty('transform');
      this.eventElement.style.removeProperty('cursor');
      this.eventElement.style.removeProperty('userSelect');
      this.eventElement.style.removeProperty('-moz-user-select');
    }
  }

  //callbacks
  onzoomin?: () => void;
  onzoomout?: () => void;

  returnToInitialState() {
    this.zoomFactor = 1;

    this.translateX = 0;
    this.translateY = 0;

    this.lastTranslateX = 0;
    this.lastTranslateY = 0;

    this.minTranslateX = 0;
    this.minTranslateY = 0;
    this.maxTranslateX = 0;
    this.maxTranslateY = 0;

    this.applyTransform();
  }

  enable() {
    if (this.element) {
      this.element.addEventListener('DOMMouseScroll', this.onMouseWheel);
      this.element.addEventListener('mousedown', this.onMouseDown);
      this.element.addEventListener('mouseup', this.onMouseUp);
      this.element.addEventListener('wheel', this.onMouseWheel);

      this.element.style.cursor = 'move';
      this.element.style.userSelect = 'none';
      this.element.style['-moz-user-select'] = 'none';
    }

    if (this.metaDataNode) {
      const eventElement = document.createElement("div");
      eventElement.setAttribute("class", "digitalzoom-eventlayer");
      eventElement.addEventListener('DOMMouseScroll', this.onMouseWheel);
      eventElement.addEventListener('mousedown', this.onMouseDown);
      eventElement.addEventListener('mouseup', this.onMouseUp);
      eventElement.addEventListener('wheel', this.onMouseWheel);

      eventElement.style.cursor = 'move';
      eventElement.style.userSelect = 'none';
      eventElement.style['-moz-user-select'] = 'none';

      this.eventElement = eventElement;
      this.metaDataNode.after(eventElement);
    }
  }

  calculateRect() {
    if (!this.element || !this.metaDataNode) {
      return false;
    }

    this.elementRect = this.element.getBoundingClientRect();
    this.parentRect = this.metaDataNode.getBoundingClientRect();
    this.deltaWidth = this.parentRect.width - this.elementRect.width;
    this.deltaHeight = this.parentRect.height - this.elementRect.height;

    return this.elementRect.width * (1 + this.zoomStep) >= this.maxWidth || this.elementRect.height * (1 + this.zoomStep) >= this.maxHeight;
  }
}
