import {
  ChannelStatus,
  DeviceFunctionalAspectType,
  DsConfiguredState,
  DsRunningState,
  HealthStatus,
  useDevicesByAspectTypesShortQuery,
  DevicesByAspectTypesShortQuery,
  StatTime,
  BaseDeviceType,
} from "@generated/graphql";
import {Device, DeviceReport, DeviceShort} from "./deviceActions";
import { formatDateTime } from "@core/utils";
import {__} from "@solid/libs/i18n";
import { clone } from "@solid/libs";

type StorageUsageValue = null | undefined | number;

const maxAllowRatio = 0.1;

export function getStorageUsageStateText(value: StorageUsageValue): string {
  return typeof value === "number" ? `${value} GB` : "N/A";
}

export function checkStorageUsageShortage(requiredVolume: StorageUsageValue, storageValue: StorageUsageValue): boolean {
  if (typeof requiredVolume !== "number" || typeof storageValue !== "number") {
    return false;
  }

  return requiredVolume > storageValue + (storageValue * maxAllowRatio);
}

export function getLinkChannelStatusText(device: Device | DeviceShort): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || device.healthStatus !== HealthStatus.Normal) {
    return "";
  }
  switch (aspect.channelStatus) {
    case ChannelStatus.Ontime: return __("ONTIME");
    case ChannelStatus.DelayedDegraded: return __("DELAYED/DEGRADED");
    case ChannelStatus.DelayedCatchup: return __("DELAYED/CATCHUP");
    default: return "";
  }
}

export function getLinkDsRunningState(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.dsRunningState) {
    return "N/A";
  }

  return aspect.dsRunningState;
}

export function getLinkDsConfiguredState(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.dsConfiguredState) {
    return "N/A";
  }

  return aspect.dsConfiguredState;
}

export function getLinkLoadState(device: Device): number | undefined {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.avatarLoad) {
    return undefined;
  }
  return aspect.avatarLoad;
}

export function getLinkDeliveryQueue(device: Device): number | undefined {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.deliveryQueue) {
    return undefined;
  }
  return aspect.deliveryQueue;
}

export function getLinkDeliveryQueueUploadTime(device: Device): number | undefined {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.deliveryQueueUploadTime) {
    return undefined;
  }
  return aspect.deliveryQueueUploadTime;
}

export function getLinkStatusUpdateAge(device: Device | DeviceShort): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.lastStatusUpdate) {
    return __("Not updated");
  }
  const interval = aspect.statusUpdateInterval || 5;
  const age = Math.round((Date.now() - new Date(aspect.lastStatusUpdate).getTime()) / 1000);
  if (age <= interval * 2) {
    return "";
  }
  if (age < 60) {
    return __("{{number}} second(s) ago", {number: age});
  }
  if (age < 60 * 60) {
    return __("{{number}} minute(s) ago", {number: Math.round(age / 60)});
  }
  if (age < 60 * 60 * 24) {
    return __("{{number}} hour(s) ago", {number: Math.round(age / (60 * 60))});
  }
  return __("{{number}} day(s) ago", {number: Math.round(age / (60 * 60 * 24))});
}

export function getLinkDsStateText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.dsConfiguredState) {
    return __("N/A");
  }
  if (aspect.dsConfiguredState === DsConfiguredState.On) {
    return __("On") + (aspect.dsRunningState ? " / " + (aspect.dsRunningState === DsRunningState.Running ? __("Running") : __("Stopped")) : "");
  }
  return __("Off");
}

export function getLinkStatusUpdateText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.lastStatusUpdate) {
    return __("N/A");
  }
  let timeText = formatDateTime(new Date(aspect.lastStatusUpdate));
  if (aspect.statusUpdateInterval) {
    timeText += " (" + __("Interval") + ":";
    const min = Math.floor(aspect.statusUpdateInterval / 60);
    const sec = Math.floor(aspect.statusUpdateInterval) - min * 60;
    if (min) {
      timeText += __("{{number}} min", {number: min});
    }
    if (sec) {
      timeText += __("{{number}} s", {number: sec});
    }
    timeText += ")";
  }
  return timeText;
}

export function getLinkSoftwareVersionText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.software?.version) {
    return __("N/A");
  }
  return `${aspect.software?.updateAvailable ? __("OUTDATED") + " " : ""}${aspect.software.version}`;
}

const defaultCountByStatus: Record<HealthStatus, number> = {
  [HealthStatus.Normal]: 0,
  [HealthStatus.Broken]: 0,
  [HealthStatus.Degraded]: 0,
  [HealthStatus.Pending]: 0,
  [HealthStatus.Off]: 0
};

const defaultDeviceInfo = { total: 0, countByStatus: defaultCountByStatus };

type DevicesInfo = { total: number, countByStatus: Record<HealthStatus, number> };

type AvatarDevicesInfo = {
  camera: DevicesInfo;
  sensor: DevicesInfo;
  gateway: DevicesInfo;
  total: number;
};

type LinkDeviceInfoResult = {
  loading: boolean;
  error?: Error;
  data?: DevicesByAspectTypesShortQuery;
  getLinkDevicesCounts: (link: DeviceShort | DeviceReport) => AvatarDevicesInfo;
  getLinkDevicesInfo: (link: DeviceShort | DeviceReport) => string;
};

export function useLinkDeviceInfo(): LinkDeviceInfoResult {
  const { data, loading, error } = useDevicesByAspectTypesShortQuery({ variables: { types: [
    { type: DeviceFunctionalAspectType.Media },
    { type: DeviceFunctionalAspectType.Sensor },
    { type: DeviceFunctionalAspectType.Gateway }
  ] } });

  function getLinkDevicesCounts(link: DeviceShort | DeviceReport): AvatarDevicesInfo {
    const devicesCounts: AvatarDevicesInfo = {
      camera: clone(defaultDeviceInfo),
      sensor: clone(defaultDeviceInfo),
      gateway: clone(defaultDeviceInfo),
      total: 0
    };

    if (data) {
      for (const device of data.devicesByAspectTypes) {
        if (device.platform?.id === link.id) {
          if (device.deviceType === BaseDeviceType.Camera) {
            devicesCounts.camera.countByStatus[device.healthStatus]++;
            devicesCounts.camera.total++;
          }
          else if (device.deviceType === BaseDeviceType.Sensor) {
            devicesCounts.sensor.countByStatus[device.healthStatus]++;
            devicesCounts.sensor.total++;
          }
          else if (device.deviceType === BaseDeviceType.Gateway) {
            devicesCounts.gateway.countByStatus[device.healthStatus]++;
            devicesCounts.gateway.total++;
          }
        }
      }
    }

    devicesCounts.total = devicesCounts.sensor.total + devicesCounts.camera.total + devicesCounts.gateway.total;

    return devicesCounts;
  }

  function getLinkDevicesInfo(link: DeviceShort | DeviceReport): string {
    if (!data) {
      return "";
    }

    const { camera, sensor, gateway, total } = getLinkDevicesCounts(link);
    const countByStatus = { ...defaultCountByStatus };

    for (const status in defaultCountByStatus) {
      countByStatus[status] = camera.countByStatus[status] + sensor.countByStatus[status] + gateway.countByStatus[status];
    }

    let text = "";
    if (countByStatus[HealthStatus.Normal]) {
      text += __("{{status}} operational", {status: countByStatus[HealthStatus.Normal]});
    }
    if (countByStatus[HealthStatus.Broken]) {
      text += (text ? ", " : "") + __("{{status}} broken", {status: countByStatus[HealthStatus.Broken]});
    }
    if (countByStatus[HealthStatus.Degraded]) {
      text += (text ? ", " : "") + __("{{status}} degraded", {status: countByStatus[HealthStatus.Degraded]});
    }
    if (countByStatus[HealthStatus.Pending]) {
      text += (text ? ", " : "") + __("{{status}} pending", {status: countByStatus[HealthStatus.Pending]});
    }
    if (countByStatus[HealthStatus.Off]) {
      text += (text ? ", " : "") + __("{{status}} off", {status: countByStatus[HealthStatus.Off]});
    }
    return `${total} (${text})`;
  }

  return { loading, error, data, getLinkDevicesCounts, getLinkDevicesInfo };
}

export function getLinkChannelBenchmarkText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.channelBenchmark) {
    return __("N/A");
  }
  const { error, startedAt, tcpUp, tcpDown } = aspect.channelBenchmark;
  if (error) {
    return error;
  }
  const timeStr = startedAt ? formatDateTime(new Date(startedAt)) : __("N/A");
  const up = tcpUp?.bitsPerSecond ? Math.round(tcpUp.bitsPerSecond / 1024 / 1024 * 1000) / 1000 : undefined;
  const down = tcpDown?.bitsPerSecond ? Math.round(tcpDown.bitsPerSecond / 1024 / 1024 * 1000) / 1000 : undefined;
  return `${timeStr} - ${__("Up")}: ${up ? up + " " + __("MBit/s") : __("unavailable")}; ${__("Down")}: ${down ? down + " " + __("MBit/s") : __("unavailable")}`;
}

export function getLinkEncryptionKeyText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.encryptionKey) {
    return __("N/A");
  }
  const { isValid, issuedAt } = aspect.encryptionKey;
  if (!isValid) {
    return __("INVALID");
  }
  return issuedAt ? __("Generated {{date}}", {date: formatDateTime(new Date(issuedAt))}) : __("Generation time is not known");
}

export function getLinkUptimeText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect.upTime) {
    return __("N/A");
  }
  let uptime = aspect.upTime;
  const days = Math.trunc(uptime / (24 * 60 * 60));
  uptime %= 24 * 60 * 60;
  const hours = Math.trunc(uptime / (60 * 60));
  uptime %= 60 * 60;
  const minutes = Math.trunc(uptime / 60);
  const seconds = uptime % 60;
  let text = `${hours.toString().padStart(2, "0")}:${minutes.toString().padStart(2, "0")}:${seconds.toString().padStart(2, "0")}`;
  if (days) {
    text = `${days} day(s), ` + text;
  }
  return text;
}

export function getLinkUpdateHistoryText(device: Device): string {
  const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  if (!aspect || aspect.__typename !== "DFA_Avatar" || !aspect?.software?.lastUpdateTime) {
    return __("N/A");
  }
  const { lastUpdateTime, lastUpdateSucceeded } = aspect.software;
  return `${formatDateTime(new Date(lastUpdateTime))} - ${lastUpdateSucceeded ? __("SUCCESS") : __("FAILED")}`;
}

export function getStatTimeText(statTime?: StatTime | null): string {
  if (!statTime) {
    return "N/A";
  }
  const { avg, min, max } = statTime;
  return avg + " ms " + __("average") + ", " + min  + " ms " + __("minimum") + ", " + max  + " ms " + __("maximum");
}
