/* eslint-disable import/no-extraneous-dependencies */
import { useEffect, useRef } from "react";
import produce from "immer";
import { IconProp } from "@fortawesome/fontawesome-svg-core";
import { ApolloClient, DataProxy, defaultDataIdFromObject, DocumentNode, useApolloClient } from "@apollo/client";
import {
  DeviceDocument,
  DeviceFunctionalAspectInput,
  DeviceFunctionalAspectType,
  DeviceInput,
  DeviceListByAspectTypesDocument,
  DeviceListByAspectTypesQuery,
  DeviceListByAspectTypesQueryVariables,
  DeviceNameDocument,
  DeviceNameQuery,
  DeviceNameQueryVariables,
  DeviceQuery,
  DeviceQueryVariables,
  DevicesByAspectTypesQuery,
  DevicesByAspectTypesReportQuery,
  DevicesByAspectTypesShortDocument,
  DevicesByAspectTypesShortQuery,
  DevicesByAspectTypesShortQueryVariables,
  DevicesByIdsDocument,
  DevicesByIdsQuery,
  DevicesByIdsQueryVariables,
  DeviceShortFieldsFragment,
  DeviceShortFieldsFragmentDoc,
  DeviceStorageConfigInput,
  HealthStatus,
  LabelObject,
  LabelsDocument,
  LabelsQuery,
  LabelType,
  ObjectAction,
  ObjectDescriptor,
  PoliciesDocument,
  PoliciesQuery,
  RealmObjectType,
  SetsDocument,
  SetsQuery,
  useDeviceLazyQuery,
  useDevicesByAspectTypesShortLazyQuery,
  ZonesDocument,
  ZonesQuery,
  SensorListByAspectTypeQuery,
  SensorListByAspectTypeQueryVariables,
  SensorListByAspectTypeDocument,
  SensorListFieldsFragmentDoc,
  SensorListFieldsFragment,
  BaseDeviceType,
  DeviceWitnessesByAspectTypeQuery,
  DeviceWitnessesByAspectTypeQueryVariables,
  DeviceWitnessesByAspectTypeDocument, GatewayListByAspectTypeQuery, GatewayListByAspectTypeQueryVariables, GatewayListByAspectTypeDocument, GatewayListFieldsFragment, GatewayListFieldsFragmentDoc, useGatewayListByAspectTypeLazyQuery
} from "@generated/graphql";
import { readQuery } from "@core/utils";
import { PropType } from "utils";
import { Log } from "@solid/libs/log";
import { __ } from "@solid/libs/i18n";
import { Type, UUID } from "@solid/types";
import { useObjectUpdate } from "core/actions";

type DeviceActionsProps = {
  skipSubscription?: boolean;
};

type DeviceActionsResult = {
  updateCachedDevice: (id: UUID) => Promise<void>;
  deleteCachedDevice: (id: UUID) => void;
  updateCachedDevices: (reconnectTimestamp: number) => Promise<void>;
};

const waitBeforeAddTimeout = 10 * 1000; // ms
const waitBeforeUpdateTimeout = 5 * 1000; // ms

const witnessesAspectTypes = {
  "sensor": DeviceFunctionalAspectType.Sensor,
  "camera": DeviceFunctionalAspectType.Media,
  "avatar": DeviceFunctionalAspectType.Avatar,
};

// types what are used in DevicesByAspectTypesShortQuery, DeviceListByAspectTypesDocument
const deviceShortListByAspectTypes = [
  [{ type: DeviceFunctionalAspectType.Media }],
  [{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }]
];

// types what are used in DeviceWitnessesByAspectType
const deviceWitnessesByAspectTypes = [
  [{ type: DeviceFunctionalAspectType.Media }],
  [{ type: DeviceFunctionalAspectType.Sensor }],
  [{ type: DeviceFunctionalAspectType.Avatar }]
];

export function useDeviceActions({ skipSubscription = false }: DeviceActionsProps): DeviceActionsResult {
  const client = useApolloClient();
  const { data: subData, error: subError } = useObjectUpdate();
  const [getDevice, { data: devData, error: devError }] = useDeviceLazyQuery({ fetchPolicy: "network-only" });
  const [getShortDevices, { data: devShortData, error: devShortError }] = useDevicesByAspectTypesShortLazyQuery({ fetchPolicy: "network-only" });
  const [getGatewayList, { data: gatewayListData, error: gatewayListError }] = useGatewayListByAspectTypeLazyQuery({ fetchPolicy: "network-only", variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] } });
  const deviceTimeoutMap = useRef(new Map<UUID, NodeJS.Timeout>());
  const lastUpdateDevicesCacheTs = useRef<number>();

  useEffect(() => {
    subError && console.error("Object actions subscription error:", subError);

    if (!subData) {
      return;
    }

    const { objectUpdate: { id, action, type, healthStatus } } = subData;
    switch (action) {
      case ObjectAction.Create:
        console.info("ObjectAction.Create", id, isDeviceInCache(id));

        // TODO: device will be available between 1sec to 5sec after creating, find more correct way for LIVE adding new device to list

        globalThis.setTimeout(() => {
          getDevice({ variables: { id } });
        }, waitBeforeAddTimeout);
        break;

      case ObjectAction.Update:
        console.info("ObjectAction.Update", id, isDeviceInCache(id));

        const timeout = deviceTimeoutMap.current.get(id);
        if (timeout) {
          clearTimeout(timeout);
        }
        deviceTimeoutMap.current.set(
          id,
          globalThis.setTimeout(() => {
            getDevice({variables: {id}});
          }, waitBeforeUpdateTimeout)
        );
        break;

      case ObjectAction.Delete:
        console.info("ObjectAction.Delete", id, isDeviceInCache(id));

        deleteCachedDevice(id);
        break;

      case ObjectAction.Status:
        console.info("ObjectAction.Status", id, isDeviceInCache(id), type, healthStatus);

        if (isDeviceInCache(id)) {
          updateStatusInCache(id, type as Type, {health: healthStatus ?? undefined});
        }
        break;
    }
  }, [subData, subError]);

  useEffect(() => {
    devError && console.error("Object actions read device error:", devError);

    if (devData?.device) {
      updateDeviceInCache(devData.device);
    }
  }, [devData, devError]);

  useEffect(() => {
    devShortError && console.error("Cache updating error:", devShortError);
    gatewayListError && console.error("Cache updating error:", gatewayListError);
  }, [devShortData, devShortError, gatewayListError, gatewayListData]);

  async function updateCachedDevices(reconnectTimestamp: number) {
    if (lastUpdateDevicesCacheTs.current && lastUpdateDevicesCacheTs.current === reconnectTimestamp) return;
    console.log(("Updating devices cache..."));

    lastUpdateDevicesCacheTs.current = reconnectTimestamp;

    const cameras = await getShortDevices({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] }});
    const sensors = await getShortDevices({ variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] }});
    const gateways = await getShortDevices({ variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] }});

    const newCameras = cameras.data?.devicesByAspectTypes;
    const newSensors = sensors.data?.devicesByAspectTypes;
    const newGateways = gateways.data?.devicesByAspectTypes;
    const newSensorsAndCameras = [...(newCameras ?? []), ...(newSensors ?? [])];
    const newSensorsCamerasAndGateways = [...newSensorsAndCameras, ...(newGateways ?? [])];

    // update cache list of sensors, cameras and gateway
    if (newSensorsCamerasAndGateways.length > 0) {
      const devicesQueryTypes = [
        { type: DeviceFunctionalAspectType.Media },
        { type: DeviceFunctionalAspectType.Sensor },
      ];

      updateDeviceCacheIsExist({
        query: DevicesByAspectTypesShortDocument,
        variables: { types: devicesQueryTypes },
        data: { devicesByAspectTypes: [...newSensorsCamerasAndGateways] }
      });
    }

    // update sensors and cameras lists in cache
    if (newSensorsAndCameras.length > 0) {
      const devicesQueryTypes = [
        { type: DeviceFunctionalAspectType.Media },
        { type: DeviceFunctionalAspectType.Sensor }
      ];

      updateDeviceCacheIsExist({
        query: DevicesByAspectTypesShortDocument,
        variables: { types: devicesQueryTypes },
        data: { devicesByAspectTypes: [...newSensorsAndCameras] }
      });
      updateDeviceCacheIsExist({
        query: DeviceListByAspectTypesDocument,
        variables: { types: devicesQueryTypes },
        data: { devicesByAspectTypes: [...newSensorsAndCameras] }
      });
    }

    // update cameras lists in cache
    if (newCameras) {
      updateDeviceCacheIsExist({
        query: DevicesByAspectTypesShortDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Media }] },
        data: { devicesByAspectTypes: [...newCameras] }
      });
      updateDeviceCacheIsExist({
        query: DeviceListByAspectTypesDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Media }] },
        data: { devicesByAspectTypes: [...newCameras] }
      });
    }

    // update sensors lists in cache
    if (newSensors) {
      updateDeviceCacheIsExist({
        query: SensorListByAspectTypeDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] },
        data: { devicesByAspectTypes: [...newSensors] }
      });
    }

    // update gateway list in cache
    const gatewayListByAspectTypeData = client.readQuery<GatewayListByAspectTypeQuery, GatewayListByAspectTypeQueryVariables>({
      query: GatewayListByAspectTypeDocument,
      variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] }
    });
    if (gatewayListByAspectTypeData?.devicesByAspectTypes) {
      const gatewayList = await getGatewayList();
      if (gatewayList.data?.devicesByAspectTypes) {
        client.writeQuery<GatewayListByAspectTypeQuery>({
          query: GatewayListByAspectTypeDocument,
          variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] },
          data: { devicesByAspectTypes: [...gatewayList.data.devicesByAspectTypes] }
        });
      }
    }
  }

  function updateDeviceCacheIsExist<TQuery, TVariables, TData>(options: DataProxy.WriteQueryOptions<TData, TVariables>) {
    const cacheData = client.readQuery<TQuery, TVariables>({
      query: options.query,
      variables: options.variables
    });
    cacheData && client.writeQuery(options);
  }

  function updateDeviceInCache(device: DeviceShort): void {
    // update gateway
    if (device.aspects.some(aspect => aspect.type === DeviceFunctionalAspectType.Gateway)) {
      // update in GatewayListByAspectTypeDocument
      const gatewayListByAspectTypeData = readQuery<GatewayListByAspectTypeQuery, GatewayListByAspectTypeQueryVariables>(client, {
        query: GatewayListByAspectTypeDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] }
      });
      if (gatewayListByAspectTypeData && !gatewayListByAspectTypeData.devicesByAspectTypes.some(dev => dev.id === device.id)) {
        client.writeQuery<GatewayListByAspectTypeQuery>({
          query: GatewayListByAspectTypeDocument,
          variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] },
          data: { devicesByAspectTypes: [...gatewayListByAspectTypeData.devicesByAspectTypes, device as GatewayShort] }
        });
      }
    }

    // update cameras and sensor
    if (device.aspects.some(aspect => [DeviceFunctionalAspectType.Media, DeviceFunctionalAspectType.Sensor].includes(aspect.type))) {
      for (const deviceTypes of deviceShortListByAspectTypes) {
        if (device.aspects.some(aspect => deviceTypes.map(aspect => aspect.type).includes(aspect.type))) {
          // update in DevicesByAspectTypesShortDocument
          const devicesByAspectTypesShortData = readQuery<DevicesByAspectTypesShortQuery, DevicesByAspectTypesShortQueryVariables>(client, {
            query: DevicesByAspectTypesShortDocument,
            variables: { types: deviceTypes }
          });
          if (devicesByAspectTypesShortData && !devicesByAspectTypesShortData.devicesByAspectTypes.some(dev => dev.id === device.id)) {
            client.writeQuery<DevicesByAspectTypesShortQuery>({
              query: DevicesByAspectTypesShortDocument,
              variables: { types: deviceTypes },
              data: { devicesByAspectTypes: [...devicesByAspectTypesShortData.devicesByAspectTypes, device] }
            });
          }

          // update in DeviceListByAspectTypesDocument
          const deviceListByAspectTypesData = readQuery<DeviceListByAspectTypesQuery, DeviceListByAspectTypesQueryVariables>(client, {
            query: DeviceListByAspectTypesDocument,
            variables: { types: deviceTypes }
          });
          if (deviceListByAspectTypesData && !deviceListByAspectTypesData.devicesByAspectTypes.some(dev => dev.id === device.id)) {
            client.writeQuery<DeviceListByAspectTypesQuery>({
              query: DeviceListByAspectTypesDocument,
              variables: { types: deviceTypes },
              data: { devicesByAspectTypes: [...deviceListByAspectTypesData.devicesByAspectTypes, device] }
            });
          }
        }
      }
    }

    // update sensor
    if (device.aspects.some(aspect => aspect.type === DeviceFunctionalAspectType.Sensor)) {
      // update in SensorListByAspectTypeDocument
      const sensorListByAspectTypeData = readQuery<SensorListByAspectTypeQuery, SensorListByAspectTypeQueryVariables>(client, {
        query: SensorListByAspectTypeDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] }
      });
      if (sensorListByAspectTypeData && !sensorListByAspectTypeData.devicesByAspectTypes.some(dev => dev.id === device.id)) {
        client.writeQuery<SensorListByAspectTypeQuery>({
          query: SensorListByAspectTypeDocument,
          variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] },
          data: { devicesByAspectTypes: [...sensorListByAspectTypeData.devicesByAspectTypes, device] }
        });
      }
    }

    // update camera
    if (device.aspects.some(aspect => aspect.type === DeviceFunctionalAspectType.Media)) {
      // update in DevicesByIdsDocument
      const devicesByIdsData = readQuery<DevicesByIdsQuery, DevicesByIdsQueryVariables>(client, {
        query: DevicesByIdsDocument,
        variables: { ids: [device.id] }
      });
      if (devicesByIdsData && !devicesByIdsData.devicesByIds.some(dev => dev.id === device.id)) {
        client.writeQuery<DeviceListByAspectTypesQuery>({
          query: DevicesByIdsDocument,
          variables: { types: [{ type: DeviceFunctionalAspectType.Media }] },
          data: { devicesByAspectTypes: [...devicesByIdsData.devicesByIds, device] }
        });
      }

      // update in DeviceNameDocument
      const deviceNameData = readQuery<DeviceNameQuery, DeviceNameQueryVariables>(client, {
        query: DeviceNameDocument,
        variables: { id: device.id }
      });
      /* eslint-disable @typescript-eslint/naming-convention */
      if (deviceNameData) {
        const { id, name, __typename } = device;
        client.writeQuery<DeviceNameQuery>({
          query: DeviceNameDocument,
          variables: { id },
          data: { device: { id, name, __typename } }
        });
      }
      /* eslint-enable @typescript-eslint/naming-convention */
    }

    // update avatar
    if (device.aspects.some(aspect => aspect.type === DeviceFunctionalAspectType.Avatar)) {
      const data = readQuery<DevicesByAspectTypesShortQuery, DevicesByAspectTypesShortQueryVariables>(client, {
        query: DevicesByAspectTypesShortDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] }
      });
      if (data && !data.devicesByAspectTypes.some(dev => dev.id === device.id)) {
        client.writeQuery<DevicesByAspectTypesShortQuery>({
          query: DevicesByAspectTypesShortDocument,
          variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] },
          data: { devicesByAspectTypes: [...data.devicesByAspectTypes, device] }
        });
      }
    }

    // update DeviceWitnessesByAspectTypeDocument
    if (device.aspects.some(aspect => [DeviceFunctionalAspectType.Media, DeviceFunctionalAspectType.Sensor, DeviceFunctionalAspectType.Avatar].includes(aspect.type))) {
      for (const deviceType of deviceWitnessesByAspectTypes) {
        const deviceWitnessesByAspectTypeData = readQuery<DeviceWitnessesByAspectTypeQuery, DeviceWitnessesByAspectTypeQueryVariables>(client, {
          query: DeviceWitnessesByAspectTypeDocument,
          variables: { types: deviceType }
        });

        if (deviceWitnessesByAspectTypeData && device.aspects.some(aspect => aspect.type === deviceType[0].type)) {
          let devices = [...deviceWitnessesByAspectTypeData.devicesByAspectTypes];
          const existedDeviceIndex = devices.findIndex(dev => dev.id === device.id);

          if (existedDeviceIndex >= 0) {
            devices[existedDeviceIndex] = device;
          }
          else {
            devices = [...deviceWitnessesByAspectTypeData.devicesByAspectTypes, device];
          }

          client.writeQuery<DeviceWitnessesByAspectTypeQuery>({
            query: DeviceWitnessesByAspectTypeDocument,
            variables: { types: deviceType },
            data: { devicesByAspectTypes: devices }
          });
        }
      }
    }
  }

  function updateStatusInCache(id: UUID, type: Type, status: { health?: HealthStatus }) {
    if (!status.health) return;

    const device = { id, healthStatus: status.health };

    if (type === "camera") {
      // DevicesByAspectTypesShortDocument
      const cameraQueryByAspectTypesVariables = { types: [{type: DeviceFunctionalAspectType.Media}] };

      const deviceShortQueryOptions = {
        query: DevicesByAspectTypesShortDocument,
        variables: cameraQueryByAspectTypesVariables
      };
      updateDeviceStatusInDeviceByAspectTypesQuery({ device, queryOptions: deviceShortQueryOptions });

      // DeviceListByAspectTypesDocument
      const deviceListQueryOptions = {
        query: DeviceListByAspectTypesDocument,
        variables: cameraQueryByAspectTypesVariables
      };
      updateDeviceStatusInDeviceByAspectTypesQuery({ device, queryOptions: deviceListQueryOptions });
    }

    if (type === "sensor") {
      // SensorListByAspectTypeDocument
      const options = {
        query: SensorListByAspectTypeDocument,
        variables: { types: [{type: DeviceFunctionalAspectType.Sensor}]}
      };
      updateDeviceStatusInDeviceByAspectTypesQuery({ device, queryOptions: options });
    }

    if (type === "avatar") {
      // DevicesByAspectTypesShortDocument
      const options = {
        query: DevicesByAspectTypesShortDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }]}
      };
      updateDeviceStatusInDeviceByAspectTypesQuery({ device, queryOptions: options });
    }

    if (["camera", "sensor"].includes(type)) {
      // DeviceListByAspectTypesDocument
      const deviceListQueryOptions = {
        query: DeviceListByAspectTypesDocument,
        variables: {
          types: [
            { type: DeviceFunctionalAspectType.Media },
            { type: DeviceFunctionalAspectType.Sensor },
          ]
        }
      };
      updateDeviceStatusInDeviceByAspectTypesQuery({ device, queryOptions: deviceListQueryOptions });
    }

    if (["camera", "sensor", "avatar"].includes(type)) {
      // DeviceWitnessesByAspectTypeDocument
      const aspectType: DeviceFunctionalAspectType = witnessesAspectTypes[type];
      const options = {
        query: DeviceWitnessesByAspectTypeDocument,
        variables: {types: [{ type: aspectType }]}
      };
      updateDeviceStatusInDeviceByAspectTypesQuery({ device, queryOptions: options });
    }
  }

  type UpdateStatusProps = {
    device: {
      healthStatus: HealthStatus,
      id: string,
    }
    queryOptions: {
      query: DocumentNode,
      variables: { types: { type: DeviceFunctionalAspectType }[] }
    }
  };

  function updateDeviceStatusInDeviceByAspectTypesQuery({device, queryOptions}: UpdateStatusProps) {
    const { id, healthStatus } = device;
    const deviceData = client.readQuery<DevicesByAspectTypesShortQuery, DevicesByAspectTypesShortQueryVariables>(queryOptions);
    const deviceIndex = deviceData?.devicesByAspectTypes.findIndex(dev => dev.id === id);

    if (typeof deviceIndex === "number" && deviceIndex >= 0) {
      const devicesByAspectTypes = produce(deviceData?.devicesByAspectTypes ?? [], (draft) => {
        draft[deviceIndex].healthStatus = healthStatus;
      });
      client.writeQuery<DevicesByAspectTypesShortQuery, DevicesByAspectTypesShortQueryVariables>({
        ...queryOptions,
        data: { devicesByAspectTypes }
      });
    }
  }

  function isDeviceInCache(id: UUID): boolean {
    return !!readDeviceShortFromCache(id)  || !!readSensorShortFromCache(id) || !!readGatewayShortFromCache(id);
  }

  function readDeviceShortFromCache(id: UUID): DeviceShort | null {
    try {
      const deviceShortFields = client.readFragment<DeviceShortFieldsFragment, DeviceQueryVariables>({
        id: defaultDataIdFromObject({ __typename: "Device", id }) ?? id,
        fragment: DeviceShortFieldsFragmentDoc,
        fragmentName: "deviceShortFields"
      });

      return deviceShortFields;
    }
    catch {
      return null;
    }
  }

  function readSensorShortFromCache(id: UUID): SensorShort | null {
    try {
      const sensorShortFields = client.readFragment<SensorListFieldsFragment, DeviceQueryVariables>({
        id: defaultDataIdFromObject({ __typename: "Device", id }) ?? id,
        fragment: SensorListFieldsFragmentDoc,
        fragmentName: "sensorListFields"
      });

      return sensorShortFields;
    }
    catch {
      return null;
    }
  }

  function readGatewayShortFromCache(id: UUID): GatewayShort | null {
    try {
      const gatewayShortFields = client.readFragment<GatewayListFieldsFragment, DeviceQueryVariables>({
        id: defaultDataIdFromObject({ __typename: "Device", id }) ?? id,
        fragment: GatewayListFieldsFragmentDoc,
        fragmentName: "gatewayListFields"
      });

      return gatewayShortFields;
    }
    catch {
      return null;
    }
  }

  async function readDevice(id: UUID): Promise<Device | undefined> {
    try {
      const { data, errors } = await client.query<DeviceQuery, DeviceQueryVariables>({
        query: DeviceDocument,
        variables: { id },
        fetchPolicy: "network-only",
        errorPolicy: "all"
      });
      if (data) {
        return data.device ?? undefined;
      }
      if (errors && errors.length > 0) {
        Log.error(__("Could not read device {{id}}: {{message}}", {id, message: errors[0].message}));
      }
    }
    catch (e: any) {
      Log.error(__("Could not read device {{id}}: {{message}}", {id, message: e.message}));
    }
    return undefined;
  }

  async function updateCachedDevice(id: UUID): Promise<void> {
    if (!id) {
      return;
    }
    const device = await readDevice(id);
    if (device) {
      updateDeviceInCache(device);

      // update SetsDocument
      if (device.set) {
        const setsCacheData = client.readQuery<SetsQuery>({
          query: SetsDocument
        });
        const setsCache = setsCacheData?.sets ?? [];
        if (setsCache.length > 0) {
          const sets = produce(setsCache, (draft) => {
            if (device.set) {
              for (const set of draft) {
                let inDevice = false;
                for (const deviceSet of device.set) {
                  if (deviceSet.id === set.id) {
                    if (!set.devices.some(dev => dev.id === device.id)) {
                      const newDeviceInSet: ObjectDescriptor = {
                        __typename: "ObjectDescriptor",
                        id: device.id,
                        name: device.name,
                        type: RealmObjectType.Device
                      };
                      set.devices.push(newDeviceInSet);
                    }
                    inDevice = true;
                  }
                }
                if (!inDevice && set.devices.some(dev => dev.id === device.id)) {
                  const newDevices = set.devices.filter(dev => dev.id !== device.id);
                  set.devices = newDevices;
                }
              }
            }
          });

          client.writeQuery<SetsQuery>({
            query: SetsDocument,
            data: {
              sets
            }
          });
        }

        // update LabelsDocument
        const labelsCacheData = client.readQuery<LabelsQuery>({
          query: LabelsDocument
        });
        const labelsCache = labelsCacheData?.labels ?? [];
        if (labelsCache.length > 0) {
          const newLabelObject: LabelObject = { __typename: "LabelObject", objectId: device.id, type: RealmObjectType.Device };
          const labels = produce(labelsCache, (draft) => {
            if (device.set) {
              for (const label of draft) {
                //add new camera to 'All Cameras' label
                if (label.name === "All Cameras" && label.type === LabelType.Set && !label.objects.some(obj => obj.objectId === device.id)) {
                  label.objects.push(newLabelObject);
                }

                let inLabel = false;
                //add/update camera to label assigned in set
                for (const deviceSet of device.set) {
                  if (deviceSet.id === label.id && label.type === LabelType.Set) {
                    const isLabelHasCamera = label.objects.some(obj => obj.objectId === device.id);

                    if (isLabelHasCamera) {
                      const labelIndex = label.objects.findIndex(obj => obj.objectId === device.id);

                      if (labelIndex >= 0) {
                        label.objects[labelIndex] = newLabelObject;
                      }

                    } else {
                      label.objects.push(newLabelObject);
                    }

                    inLabel = true;
                  }
                }

                //remove camera from label which unassigned in set
                if (!inLabel && label.objects.some(obj => obj.objectId === device.id) && label.type === LabelType.Set && device.set.some(set => set.id !== label.id)) {
                  const newObjects = label.objects.filter(obj => obj.objectId !== device.id);
                  label.objects = newObjects;
                }
              }
            }
          });

          client.writeQuery<LabelsQuery>({
            query: LabelsDocument,
            data: {
              labels
            }
          });
        }
      }

      // update ZonesDocument
      const zonesCacheDocument = client.readQuery<ZonesQuery>({
        query: ZonesDocument
      });

      const zonesCache = zonesCacheDocument?.zones ?? [];
      if (zonesCache.length > 0) {
        const zones = produce(zonesCache, (draft) => {
          for (const zone of draft) {
            let inZone = false;
            if (zone.id === device.zone?.id) {
              zone.devices.push({
                __typename: "ObjectDescriptor",
                name: device.name,
                id: device.id
              });
              inZone = true;
            }
            if (!inZone && zone.devices.some(dev => dev.id === device.id)) {
              const newDevices = zone.devices.filter(dev => dev.id !== device.id);
              zone.devices = newDevices;
            }
          }
        });
        client.writeQuery<ZonesQuery>({
          query: ZonesDocument,
          data: {
            zones
          }
        });
      }
    }
  }

  function deleteCachedDevice(id: UUID): void {
    if (!id) {
      return;
    }

    // remove from GatewayListByAspectTypeDocument
    const gatewayListByAspectTypeData = readQuery<GatewayListByAspectTypeQuery, GatewayListByAspectTypeQueryVariables>(client, {
      query: GatewayListByAspectTypeDocument,
      variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] }
    });
    if (gatewayListByAspectTypeData && gatewayListByAspectTypeData.devicesByAspectTypes.some(dev => dev.id === id)) {
      client.writeQuery<GatewayListByAspectTypeQuery>({
        query: GatewayListByAspectTypeDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Gateway  }] },
        data: { devicesByAspectTypes: [...gatewayListByAspectTypeData.devicesByAspectTypes.filter(dev => dev.id !== id)] }
      });
    }

    // remove from SensorListByAspectTypeDocument
    const sensorListByAspectTypeData = readQuery<SensorListByAspectTypeQuery, SensorListByAspectTypeQueryVariables>(client, {
      query: SensorListByAspectTypeDocument,
      variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] }
    });
    if (sensorListByAspectTypeData && sensorListByAspectTypeData.devicesByAspectTypes.some(dev => dev.id === id)) {
      client.writeQuery<SensorListByAspectTypeQuery>({
        query: SensorListByAspectTypeDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Sensor  }] },
        data: { devicesByAspectTypes: [...sensorListByAspectTypeData.devicesByAspectTypes.filter(dev => dev.id !== id)] }
      });
    }

    for (const aspectTypes of deviceShortListByAspectTypes) {
      // remove from DevicesByAspectTypesShortDocument
      const devicesByAspectTypesShortData = readQuery<DevicesByAspectTypesShortQuery, DevicesByAspectTypesShortQueryVariables>(client, {
        query: DevicesByAspectTypesShortDocument,
        variables: { types: aspectTypes }
      });
      if (devicesByAspectTypesShortData && devicesByAspectTypesShortData.devicesByAspectTypes.some(dev => dev.id === id)) {
        client.writeQuery<DevicesByAspectTypesShortQuery>({
          query: DevicesByAspectTypesShortDocument,
          variables: { types: aspectTypes },
          data: { devicesByAspectTypes: [...devicesByAspectTypesShortData.devicesByAspectTypes.filter(dev => dev.id !== id)] }
        });
      }

      // remove from DeviceListByAspectTypesDocument
      const deviceListByAspectTypesData = readQuery<DeviceListByAspectTypesQuery, DeviceListByAspectTypesQueryVariables>(client, {
        query: DeviceListByAspectTypesDocument,
        variables: { types: aspectTypes }
      });
      if (deviceListByAspectTypesData && deviceListByAspectTypesData.devicesByAspectTypes.some(dev => dev.id === id)) {
        client.writeQuery<DeviceListByAspectTypesQuery>({
          query: DeviceListByAspectTypesDocument,
          variables: { types: aspectTypes },
          data: { devicesByAspectTypes: [...deviceListByAspectTypesData.devicesByAspectTypes.filter(dev => dev.id !== id)] }
        });
      }

      // remove from DeviceWitnessesByAspectTypeDocument
      for (const aspectTypes of deviceWitnessesByAspectTypes) {
        const deviceWitnessesByAspectTypeData = readQuery<DeviceWitnessesByAspectTypeQuery, DeviceWitnessesByAspectTypeQueryVariables>(client, {
          query: DeviceWitnessesByAspectTypeDocument,
          variables: { types: aspectTypes }
        });
        if (deviceWitnessesByAspectTypeData && !deviceWitnessesByAspectTypeData.devicesByAspectTypes.some(dev => dev.id === id)) {
          client.writeQuery<DeviceWitnessesByAspectTypeQuery>({
            query: DeviceWitnessesByAspectTypeDocument,
            variables: { types: aspectTypes },
            data: { devicesByAspectTypes: [...deviceWitnessesByAspectTypeData.devicesByAspectTypes.filter(dev => dev.id !== id)] }
          });
        }
      }
    }

    // remove from DevicesByIdsDocument
    const devicesByIdsData = readQuery<DevicesByIdsQuery, DevicesByIdsQueryVariables>(client, {
      query: DevicesByIdsDocument,
      variables: { ids: [id] }
    });
    if (devicesByIdsData && devicesByIdsData.devicesByIds.some(dev => dev.id === id)) {
      client.writeQuery<DevicesByIdsQuery>({
        query: DevicesByIdsDocument,
        variables: { ids: [id] },
        data: { devicesByIds: [...devicesByIdsData.devicesByIds.filter(dev => dev.id !== id)] }
      });
    }

    // remove from DeviceNameDocument
    const deviceNameData = readQuery<DeviceNameQuery, DeviceNameQueryVariables>(client, {
      query: DeviceNameDocument,
      variables: { id }
    });
    if (deviceNameData) {
      client.writeQuery<DeviceNameQuery>({
        query: DeviceNameDocument,
        variables: { id },
        data: { device: null }
      });
    }

    // remove from SetsDocument
    const setsCacheData = client.readQuery<SetsQuery>({
      query: SetsDocument
    });
    const setsCache = setsCacheData?.sets ?? [];
    if (setsCache.length > 0) {
      const sets = produce(setsCache, (draft) => {
        for (const set of draft) {
          if (set.devices.some(dev => dev.id === id)) {
            set.devices = set.devices.filter(dev => dev.id !== id);
          }
        }
      });
      client.writeQuery<SetsQuery>({
        query: SetsDocument,
        data: {
          sets
        }
      });
    }

    // remove from LabelsDocument
    const labelsCacheData = client.readQuery<LabelsQuery>({
      query: LabelsDocument
    });

    const labelsCache = labelsCacheData?.labels ?? [];
    if (labelsCache.length > 0) {
      const labels = produce(labelsCache, (draft) => {
        for (const label of draft) {
          if (label.objects.some(obj => obj.objectId === id) && label.type === LabelType.Set) {
            const newLabelObject = label.objects.filter(obj => obj.objectId !== id);
            label.objects = newLabelObject;
          }
        }
      });

      client.writeQuery<LabelsQuery>({
        query: LabelsDocument,
        data: {
          labels
        }
      });
    }

    // remove from PoliciesDocument
    const policiesCacheData = client.readQuery<PoliciesQuery>({
      query: PoliciesDocument
    });

    const policiesCache = policiesCacheData?.policies ?? [];
    if (policiesCache.length > 0) {
      const policies = produce(policiesCache, (draft) => {
        for (const policy of draft) {
          for (const statement of policy.statements) {
            for (let i = 0; i < statement.resources.length; i++) {
              const resource = statement.resources[i];
              if (resource.id === id) {
                statement.resources.splice(i, 1);
                break;
              }
            }
          }
        }
      });
      client.writeQuery<PoliciesQuery>({
        query: PoliciesDocument,
        data: {
          policies
        }
      });
    }

    // remove from ZonesDocument
    const zonesCacheDocument = client.readQuery<ZonesQuery>({
      query: ZonesDocument
    });

    const zonesCache = zonesCacheDocument?.zones ?? [];
    if (zonesCache.length > 0) {
      const zones = produce(zonesCache, (draft) => {
        for (const zone of draft) {
          if (zone.devices.some(dev => dev.id === id)) {
            const newDevices = zone.devices.filter(dev => dev.id !== id);
            zone.devices = newDevices;
          }
        }
      });
      client.writeQuery<ZonesQuery>({
        query: ZonesDocument,
        data: {
          zones
        }
      });
    }
  }

  return { updateCachedDevice, deleteCachedDevice, updateCachedDevices };
}

export type Device = PropType<DevicesByAspectTypesQuery, "devicesByAspectTypes">[0];
export type Aspect = PropType<Device, "aspects">[0];
export type DeviceShort = PropType<DevicesByAspectTypesShortQuery, "devicesByAspectTypes">[0];
export type DeviceReport = PropType<DevicesByAspectTypesReportQuery, "devicesByAspectTypes">[0];
export type DeviceList = PropType<DeviceListByAspectTypesQuery, "devicesByAspectTypes">[0];
export type SensorShort = PropType<SensorListByAspectTypeQuery, "devicesByAspectTypes">[0];
export type GatewayShort = PropType<GatewayListByAspectTypeQuery, "devicesByAspectTypes">[0];
export type DeviceWitness = PropType<DeviceWitnessesByAspectTypeQuery, "devicesByAspectTypes">[0];
type EntityWithIdT = { id: string, [x: string]: any };

export function deviceToDeviceInput(device: Device): DeviceInput {
  /* eslint-disable @typescript-eslint/naming-convention */
  const {
    name,
    enabled,
    location,
    zone,
    eventType,
    platform,
    config: {
      __typename: __1,
      connect: {
        __typename: __2,
        ...connect
      },
      ...config
    },
    storageConfig,
    deliveryPriority,
    aspects,
    acknowledgeRequired
  } = device;
  /* eslint-enable @typescript-eslint/naming-convention */

  let storageConfigInput: DeviceStorageConfigInput | undefined;
  if (storageConfig) {
    const { storagePool } = storageConfig;
    storageConfigInput = { storagePoolId: storagePool.id};
  }

  const input: DeviceInput = {
    name,
    enabled,
    location,
    eventType,
    zoneId: zone?.id,
    platformId: platform?.id,
    config: { connect: { ...connect }, ...config },
    storageConfig: storageConfigInput,
    deliveryPriority,
    aspects: aspects.map(aspect => aspectToAspectInput(aspect)),
    acknowledgeRequired
  };
  return input;
}

export function aspectToAspectInput(aspect: Aspect): DeviceFunctionalAspectInput {
  const { id, name, template, enabled, inputAspects } = aspect;
  let input: DeviceFunctionalAspectInput = {
    id,
    name,
    templateId: template.id,
    enabled,
    inputAspectIDs: inputAspects.map(aspect => aspect.id)
  };
  if (aspect.__typename === "DFA_Media") {
    const { streamType, URL, videoWidth, videoHeight, codec, frameRate, onvifProfile, onvifProfiles, edgeArchive, cloudArchive } = aspect;
    input = {
      ...input,
      streamType,
      URL,
      videoWidth,
      videoHeight,
      codec,
      frameRate,
      onvifProfile,
      onvifProfiles,
      edgeArchive,
      cloudArchive
    };
  }
  if (aspect.__typename === "DFA_VAE") {
    const { vaeConfig, vaeFeatures } = aspect;
    input = {
      ...input,
      vaeConfigName: vaeConfig?.name || "",
      vaeFeatures
    };
  }
  if (aspect.__typename === "DFA_PTZ") {
    const { ptzConfig } = aspect;
    input = {
      ...input,
      ptzConfig
    };
  }
  if (aspect.__typename === "DFA_Avatar") {
    const { statusUpdateInterval, webRtcEnabled, uploadBandwidthCap } = aspect;
    input = {
      ...input,
      statusUpdateInterval,
      webRtcEnabled,
      uploadBandwidthCap
    };
  }

  if (aspect.__typename === "DFA_Sensor") {
    const { model, serial, sensorEvents, category, associatedDevices, userEvents } = aspect;
    const sensorEventsInput = sensorEvents?.map(event => (event && { description: event.description, type: event.type }));

    input = {
      ...input,
      model,
      serial,
      sensorEvents: sensorEventsInput,
      category,
      associatedDevices,
      userEvents
    };
  }

  if (aspect.__typename === "DFA_Gateway") {
    const { lenelVersion, directory, associatedGatewayDevices, subscribedEvents } = aspect;

    //temporary solution
    let formattedSubscribedEvents, formattedAssociatedDevice;

    if (subscribedEvents) {
      formattedSubscribedEvents = subscribedEvents.map(event => {
        return {
          id: event.id,
          type: event.type
        };
      });
    }
    if (associatedGatewayDevices) {
      formattedAssociatedDevice = associatedGatewayDevices.map(device => {
        return {
          id: device.id,
          associatedDevices: device.associatedDevices
        };
      });
    }
    input = {
      ...input,
      lenelVersion,
      directoryId: directory.id,
      associatedGatewayDevices: formattedAssociatedDevice,
      subscribedEvents: formattedSubscribedEvents
    };
  }

  return input;
}

export function updateZoneInDeviceCache(zoneObj: { id: string, name: string } | null, deviceIds: string[], deviceType: DeviceFunctionalAspectType, document: DocumentNode, client: ApolloClient<any>): void {
  const camerasCacheData = client.readQuery({
    query: document,
    variables: { types: [{ type: deviceType }] }
  });

  const camerasCache = camerasCacheData?.devicesByAspectTypes ?? [];
  if (camerasCache.length > 0) {
    const devicesByAspectTypes = produce<DeviceShort[]>(camerasCache, (draft) => {
      for (const deviceId of deviceIds) {
        const devIndex = draft.findIndex(dev => dev.id === deviceId);
        if (devIndex >= 0) {
          draft[devIndex].zone = zoneObj;
        }
      }
    });

    client.writeQuery({
      query: document,
      data: { devicesByAspectTypes }
    });
  }
}

export function getEntityWithoutExceptions<EntityType>(
  exceptionIds: string[],
  nextState: EntityWithIdT[],
  prevState: EntityWithIdT[] = [],
): EntityType {
  if (exceptionIds.length === 0) {
    return nextState as EntityType;
  }

  const newEntities: EntityWithIdT[] = [...new Set([...prevState, ...nextState])];
  const willUnassignedEntities = prevState.reduce<EntityWithIdT[]>((acc, entity) => {
    !nextState.includes(entity) && acc.push(entity);
    return acc;
  }, []);
  const entityToAssign = nextState.reduce<EntityWithIdT[]>((acc, entity) => {
    !prevState.includes(entity) && acc.push(entity);
    return acc;
  }, []);

  willUnassignedEntities.forEach(entity => {
    if (!exceptionIds.includes(entity.id)) {
      const setIndex = newEntities.findIndex(ne => ne.id === entity.id);
      newEntities.splice(setIndex, 1);
    }
  });
  entityToAssign.forEach(entity => {
    if (exceptionIds.includes(entity.id)) {
      const setIndex = newEntities.findIndex(ne => ne.id === entity.id);
      newEntities.splice(setIndex, 1);
    }
  });

  return newEntities as EntityType;
}

const disabledIconsMap: Record<BaseDeviceType, IconProp> = {
  [BaseDeviceType.Camera]: "video-slash",
  [BaseDeviceType.Sensor]: "bell-slash",
  [BaseDeviceType.Gateway]: "server",
  [BaseDeviceType.Avatar]: "hdd"
};

const activeIconsMap: Record<BaseDeviceType, IconProp> = {
  [BaseDeviceType.Camera]: "video",
  [BaseDeviceType.Sensor]: "bell",
  [BaseDeviceType.Gateway]: "server",
  [BaseDeviceType.Avatar]: "hdd"
};

export function getListItemIcon(deviceType: BaseDeviceType, disabled: boolean): IconProp {
  return !disabled ? activeIconsMap[deviceType] : disabledIconsMap[deviceType];
}

export const healthStatusText: Record<HealthStatus, string> = {
  [HealthStatus.Off]: __("Off"),
  [HealthStatus.Pending]: __("Pending"),
  [HealthStatus.Normal]: __("Normal"),
  [HealthStatus.Degraded]: __("Degraded"),
  [HealthStatus.Broken]: __("Broken"),
};

export const linkHealthStatusText: Record<HealthStatus, string> = {
  [HealthStatus.Off]: __("OFFLINE"),
  [HealthStatus.Pending]: __("PENDING"),
  [HealthStatus.Normal]: __("ONLINE"),
  [HealthStatus.Degraded]: __("DEGRADED"),
  [HealthStatus.Broken]: __("BROKEN"),
};
