import React from "react";
import { DragObjectWithType } from "react-dnd";
import { Widget, ControlType, CameraWidget, DropdownItem, WidgetWarning } from "./lib";
import { ModuleInfo, DragObjectType } from "@core/types";
import { RWidgetLoader, RTabPanel, AddRowColEvent, ViewIdEvent } from "components/Layout";
import { EventPubSub, parseJSON } from "utils";
import { PlayerStartFrom, PlayerOnClipEnd, MetadataDrawType } from "@core/playerTypes";
import { CameraCellProps } from "components/CellContainer/CameraCell";
import { TimelineControlsPosition } from "components/Timeline";
import { getImageMapConfig, getHierarchies } from "@core/actions";
import HierarchyEditorDialog from "components/LabelsAndHierarchies/HierarchyEditorDialog";
import { EventView, ResetToLiveInterval, EventPreviewPlacement, EventListWarning } from "components/EventList";
import { EventInfo, ViewId, WidgetId } from "@generated/graphql";
import {__} from "@solid/libs/i18n";
import {VideoTouringWidget} from "./lib/VideoTouringWidget";
import {GeoMapWidget} from "./lib/GeoMapWidget";

import CameraLiveIcon from "./images/panorama_48px.svg";
import EventListIcon from "./images/format_list_numbered_48px.svg";
import EventInfoIcon from "./images/announcement_48px.svg";
import EventVideoIcon from "./images/video_library_48px.svg";
import DeviceListIcon from "./images/view_list_48px.svg";
import ArchiveViewerIcon from "./images/movie-roll.svg";
import TimelineIcon from "./images/timeline_48px.svg";
import ImageMapIcon from "./images/map_48px.svg";
import LinkProvisionIcon from "./images/link_48px.svg";
import UsersIcon from "./images/group_48px.svg";
import GroupsIcon from "./images/group_work_48px.svg";
import PoliciesIcon from "./images/verified_user_48px.svg";
import ZonesIcon from "./images/th-large-solid.svg";
import LinksIcon from "./images/devices_48px.svg";
import VMXIcon from "./images/vmx_48px.svg";
import GEOMapIcon from "./images/geo-map_48px.svg";
import HierarchiesIcon from "./images/hierarchies_48px.svg";
import LabelsIcon from "./images/labels_48px.svg";
import DownloadManagerIcon from "./images/download-manager_48px.svg";
import NotificationPolicies from "./images/envelope-regular_48px.svg";
import RealmSettings from "./images/cog-solid_48px.svg";
import Watchlist from "./images/eye_48px.svg";
import SetsIcon from "./images/circle-regular_48px.svg";
import VideoTouringIcon from "./images/retweet-solid_48px.svg";
import ReportIcon from "./images/file-contract_48px.svg";
import VideoWalls from "./images/video_walls_48px.svg";
import About from "./images/file-signature_48px.svg";
import SensorIcon from "./images/bell-solid_48px.svg";

export enum EventInfoVisual {
  Image = "Image",
  Player = "Player"
}

export enum HierarchyEditPropName {
  allowLabelHierarchyEdit = "allowLabelHierarchyEdit",
  hierarchySelection = "hierarchySelection",
  openMultipleBranches = "openMultipleBranches",
  hierarchyEditor = "hierarchyEditor",
  hierarchyId = "hierarchyId"
}

export const EventListAssociatedWidgets = [WidgetId.Timeline, WidgetId.ImageMap, WidgetId.CameraLive, WidgetId.DeviceList];

export const Widgets: Widget[] = [
  new CameraWidget(WidgetId.CameraLive, __("Camera"), __("Camera Live Viewer"), "CellContainer/CameraCell", CameraLiveIcon, {
    props: [
      { name: "canReplaceSource", label: __("Can replace source"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "metadataType", label: __("Analytics markup"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: MetadataDrawType.Alert, text: __("Alert") },
        { value: MetadataDrawType.All, text: __("All") },
        { value: MetadataDrawType.None, text: __("None") },
      ] },
      { name: "ptzControls", label: __("PTZ controls"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "digitalZoom", label: __("Digital zoom"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "fullScreenControl", label: __("Fullscreen control"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "stepBackControl", label: __("Step back control"), controlType: ControlType.Checkbox, defaultValue: true },
    ],
    acceptsConstructorProps: true,
    headerDragObjectType: DragObjectType.Camera,
    minHeight: 220,
    //disableCondition: widgets =>  widgets.some(w => w.widgetId === WidgetId.ArchiveViewer)
  }),

  new CameraWidget(WidgetId.ArchiveViewer, __("Archive Viewer"), __("Archive Viewer"), "CellContainer/CameraCell", ArchiveViewerIcon, {
    props: [
      { name: "canReplaceSource", label: __("Can replace source"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "metadataType", label: __("Analytics markup"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: MetadataDrawType.Alert, text: __("Alert") },
        { value: MetadataDrawType.All, text: __("All") },
        { value: MetadataDrawType.None, text: __("None") },
      ] },
      { name: "digitalZoom", label: __("Digital zoom"), controlType: ControlType.Checkbox, defaultValue: true,
        hideCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "fullScreenControl", label: __("Fullscreen control"), controlType: ControlType.Checkbox, defaultValue: true,
        hideCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "calendar", label: __("Calendar"), controlType: ControlType.Checkbox, defaultValue: true,
        hideCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "allowExport", label: __("Allow export"), controlType: ControlType.Checkbox, defaultValue: true,
        hideCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "userDefinedEvent", label: __("User defined event"), controlType: ControlType.Checkbox, defaultValue: true },
    ],
    acceptsConstructorProps: true,
    countLimit: widgets => widgets.some(w => w.widgetId === WidgetId.Timeline) ? 6 : 1,
    minHeight: 220,
    //disableCondition: widgets => widgets.some(w => w.widgetId === WidgetId.CameraLive)
  }),

  new Widget(WidgetId.EventList, __("Events"), __("Events"), "EventList", EventListIcon, {
    props: [
      { name: "showPreview", label: __("Show event preview"), controlType: ControlType.Checkbox },
      { name: "previewPlacement", label: __("Preview placement"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: EventPreviewPlacement.Top, text: __("Top") },
        { value: EventPreviewPlacement.Bottom, text: __("Bottom") },
        { value: EventPreviewPlacement.Left, text: __("Left") },
        { value: EventPreviewPlacement.Right, text: __("Right") },
      ], disableCondition: (widgets, props) => !props["showPreview"] },
      { name: "eventView", label: __("Event view"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: EventView.Content, text: __("Content") },
        { value: EventView.Tiles, text: __("Tiles") },
      ]},
      { name: "liveOnly", label: __("Real-time events only"), controlType: ControlType.Checkbox,
        getDefaultValue: (widgets, props) => {
          return !widgets.some(w => w.widgetId === WidgetId.Timeline);
        },
        disableCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "associatedWidgets", label: __("Associated widgets"), controlType: ControlType.Dropdown, multiple: true, minWidth: 190,
        getDropdownItems: async (client, widgets, props) => {
          const items: DropdownItem[] = [];
          for (const widget of widgets) {
            if (EventListAssociatedWidgets.includes(widget.widgetId as WidgetId) && !items.some(item => item.value === widget.widgetId)) {
              items.push({ value: widget.widgetId, text: Widgets.find(w => w.id === widget.widgetId)?.title ?? "" });
            }
          }
          return items;
        },
        getDefaultValue: (widgets, props) => {
          const timeline = widgets.find(w => w.widgetId === WidgetId.Timeline);
          return timeline ? [timeline.widgetId] : [];
        },
        disableCondition: (widgets, props) =>
          props["liveOnly"] === undefined || props["liveOnly"] === true || widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "resetToLiveAfter", label: __("Reset to real-time after"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: ResetToLiveInterval.Never, text: __("Never") },
        { value: ResetToLiveInterval.FiveMin, text: __("5 min") },
        { value: ResetToLiveInterval.TenMin, text: __("10 min") },
        { value: ResetToLiveInterval.FifteenMin, text: __("15 min") },
        { value: ResetToLiveInterval.ThirtyMin, text: __("30 min") },
      ],
      disableCondition: (widgets, props) =>
        props["liveOnly"] === undefined || props["liveOnly"] === true || widgets.some(w => w.widgetId === WidgetId.Timeline) },
      { name: "showSnapshots", label: __("Display event image"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "userInteraction", label: __("Enable user interaction"), controlType: ControlType.Checkbox, defaultValue: true,
        disableCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
    ],
    countLimit: widgets => widgets.some(w => w.widgetId === WidgetId.Timeline) ? 1 : 2,
    minWidth: 400,
    minHeight: 200,
    getWidgetWarning: (widget, props, size) => {
      const showPreview: boolean = props["showPreview"] ?? false;
      const previewPlacement: EventPreviewPlacement = props["previewPlacement"] ?? EventPreviewPlacement.Top;
      if (!showPreview) {
        return undefined;
      }
      const warning: WidgetWarning = { code: EventListWarning.TooSmallForPreview, text: __("'{{name}}' widget size is too small. Event preview disabled.", {name: widget.title})};
      if ((previewPlacement === EventPreviewPlacement.Top || previewPlacement === EventPreviewPlacement.Bottom) &&
          (size.width < 520 || size.height < 510)) {
        return warning;
      }
      if ((previewPlacement === EventPreviewPlacement.Left || previewPlacement === EventPreviewPlacement.Right) &&
          (size.width < 720 || size.height < 452)) {
        return warning;
      }
      return undefined;
    },
  }),

  new Widget(WidgetId.ArchiveEventList, __("Events"), __(" Archive Events"), "ArchiveEvents", EventListIcon, {
    props: [
      { name: "showPreview", label: __("Show event preview"), controlType: ControlType.Checkbox },
      { name: "previewPlacement", label: __("Preview placement"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: EventPreviewPlacement.Top, text: __("Top") },
        { value: EventPreviewPlacement.Bottom, text: __("Bottom") },
        { value: EventPreviewPlacement.Left, text: __("Left") },
        { value: EventPreviewPlacement.Right, text: __("Right") },
      ], disableCondition: (widgets, props) => !props["showPreview"] },
      { name: "eventView", label: __("Event view"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: EventView.Content, text: __("Content") },
        { value: EventView.Tiles, text: __("Tiles") },
      ]},
      { name: "showSnapshots", label: __("Display event image"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "userInteraction", label: __("Enable user interaction"), controlType: ControlType.Checkbox, defaultValue: true,
        disableCondition: (widgets, props) => widgets.some(w => w.widgetId === WidgetId.Timeline) },
    ],
    countLimit: widgets => widgets.some(w => w.widgetId === WidgetId.Timeline) ? 1 : 2,
    minWidth: 400,
    minHeight: 200,
    getWidgetWarning: (widget, props, size) => {
      const showPreview: boolean = props["showPreview"] ?? false;
      const previewPlacement: EventPreviewPlacement = props["previewPlacement"] ?? EventPreviewPlacement.Top;
      if (!showPreview) {
        return undefined;
      }
      const warning: WidgetWarning = { code: EventListWarning.TooSmallForPreview, text: __("'{{name}}' widget size is too small. Event preview disabled.", {name: widget.title})};
      if ((previewPlacement === EventPreviewPlacement.Top || previewPlacement === EventPreviewPlacement.Bottom) &&
          (size.width < 520 || size.height < 510)) {
        return warning;
      }
      if ((previewPlacement === EventPreviewPlacement.Left || previewPlacement === EventPreviewPlacement.Right) &&
          (size.width < 720 || size.height < 452)) {
        return warning;
      }
      return undefined;
    },
  }),

  new Widget(WidgetId.EventDetails, __("Event Details"), __("Event Details"), "EventInfo", EventInfoIcon, {
    props: [
      { name: "visual", label: __("Visual"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: EventInfoVisual.Image, text: __("Image") },
        { value: EventInfoVisual.Player, text: __("Video") },
      ] },
      { name: "metadataType", label: __("Analytics markup"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: MetadataDrawType.Alert, text: __("Alert") },
        { value: MetadataDrawType.All, text: __("All") },
        { value: MetadataDrawType.None, text: __("None") },
      ], disableCondition: (widgets, props) => props["visual"] !== EventInfoVisual.Player },
      { name: "displayAcknowledgeForm", label: __("Display acknowledge form"), controlType: ControlType.Checkbox },
      { name: "requireAcknowledgeDetails", label: __("Require acknowledge details"), controlType: ControlType.Checkbox,
        disableCondition: (widgets, props) => !props["displayAcknowledgeForm"] },
      { name: "autoUpdate", label: __("Auto-update on new event"), controlType: ControlType.Checkbox, defaultValue: true },
    ],
    countLimit: 1,
  }),

  new Widget(WidgetId.EventVideo, __("Event Video"), __("Event Video"), "CellContainer/EventVideo", EventVideoIcon, {
    props: [
      { name: "startFrom", label: __("Start from"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: PlayerStartFrom.Event, text: __("Event") },
        { value: PlayerStartFrom.Live, text: __("Live") },
      ] },
      { name: "metadataType", label: __("Analytics markup"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: MetadataDrawType.Alert, text: __("Alert") },
        { value: MetadataDrawType.All, text: __("All") },
        { value: MetadataDrawType.None, text: __("None") },
      ] },
      { name: "onClipEnd", label: __("On clip end"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: PlayerOnClipEnd.Clear, text: __("Clear tile") },
        { value: PlayerOnClipEnd.Stop, text: __("Stop") },
        { value: PlayerOnClipEnd.Loop, text: __("Loop") },
      ], disableCondition: (widgets, props) => props["startFrom"] === PlayerStartFrom.Live },
      { name: "autoUpdate", label: __("Auto-update on new event"), controlType: ControlType.Checkbox, defaultValue: true },
    ],
    countLimit: 1
  }),

  new Widget(WidgetId.DeviceList, __("Device List"), __("Device List"), "CameraList", DeviceListIcon, {
    props: [
      { name: HierarchyEditPropName.hierarchyId, label: __("Hierarchy"), controlType: ControlType.Dropdown, minWidth: 150,
        getDropdownItems: async (client, widgets, props) => {
          const items: DropdownItem[] = (await getHierarchies(client)).map(({ id, name }) => ({ value: id, text: name }));
          items.unshift({ value: "", text: "" });
          return items;
        }},
      { name: HierarchyEditPropName.hierarchyEditor, label: __("Hierarchy editor"), controlType: ControlType.Button,
        buttonLabel: __("Open"), buttonModal: HierarchyEditorDialog },
      { name: HierarchyEditPropName.openMultipleBranches, label: __("Open multiple branches"), controlType: ControlType.Checkbox,
        disableCondition: (widgets, props) => !props[HierarchyEditPropName.hierarchyId] },
      { name: HierarchyEditPropName.hierarchySelection, label: __("Hierarchy selection"), controlType: ControlType.Checkbox,
        disableCondition: (widgets, props) => !props[HierarchyEditPropName.hierarchyId] },
      { name: "nameFilter", label: __("Name filter"), controlType: ControlType.Checkbox, defaultValue: true },
      { name: "labelsFilter", label: __("Label(s) filter"), controlType: ControlType.Checkbox, defaultValue: true,
        disableCondition: (widgets, props) => !!props[HierarchyEditPropName.hierarchyId] },
      { name: HierarchyEditPropName.allowLabelHierarchyEdit, label: __("Labels/hierarchies editing"), controlType: ControlType.Checkbox },
    ],
    countLimit: 1
  }),

  new Widget(WidgetId.Timeline, __("Timeline"), __("Timeline"), "Timeline", TimelineIcon, {
    props: [
      { name: "controlsPosition", label: __("Controls position"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: TimelineControlsPosition.Top, text: __("Top") },
        { value: TimelineControlsPosition.Bottom, text: __("Bottom") },
      ], defaultValue: TimelineControlsPosition.Bottom },
    ],
    countLimit: 1,
    compatibleWidgets: [WidgetId.DeviceList, WidgetId.ArchiveViewer, WidgetId.EventList],
    getViewValidationError: widgets => {
      if (!widgets.some(w => w.widgetId === WidgetId.ArchiveViewer)) {
        return __("At least one Archive Viewer must be added.");
      }
      if (!widgets.some(w => {
        if (w.widgetId !== WidgetId.ArchiveViewer) {
          return false;
        }
        const props = parseJSON<CameraCellProps>(w.propsJSON);
        if (!props) {
          return false;
        }
        return !!props.object?.obj && !!props.object?.obj["data-id"];
      }) && !widgets.some(w => w.widgetId === WidgetId.DeviceList)) {
        return __("Device must be assigned to an Archive Viewer or Device List widget must be added.");
      }
      return undefined;
    }
  }),

  new Widget(WidgetId.ImageMap, __("Image Map"), __("Image Map"), "ImageMap", ImageMapIcon, {
    props: [
      { name: "editMode", label: __("Editor mode"), controlType: ControlType.Checkbox },
      { name: "maxEventCount", label: __("Max device event count"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: "1", text: "1" },
        { value: "5", text: "5" },
        { value: "10", text: "10" },
        { value: "20", text: "20" },
      ], defaultValue: "10",
      disableCondition: (widgets, props) => !!props["editMode"] },
      { name: "showEventsInInvisibleLayers", label: __("Show events in invisible layers"), controlType: ControlType.Checkbox,
        disableCondition: (widgets, props) => !!props["editMode"] },
      { name: "mapSetId", label: __("Image map set"), controlType: ControlType.Dropdown,
        getDropdownItems: async (client, widgets, props) => {
          const config = await getImageMapConfig(client);
          const items: DropdownItem[] = config.sets.map(({ id, name }) => ({ value: id, text: name }));
          items.unshift({ value: "", text: "" });
          return items;
        }},
      { name: "showMapSetList", label: __("Show map set list"), controlType: ControlType.Checkbox, defaultValue: true },
    ]
  }),

  new Widget(WidgetId.Vmx, __("VMX"), __("VMX"), "VMX", VMXIcon, {
    minWidth: 400,
    minHeight: 400,
  }),

  new VideoTouringWidget(WidgetId.VideoTouring, __("Video Touring"), __("Video Touring"), "VideoTouring", VideoTouringIcon, {
    props: [
      { name: "editMode", label: __("Editor mode"), controlType: ControlType.Checkbox },
      { name: "delay", label: __("Delay Time"), controlType: ControlType.Dropdown, dropdownItems: [
        { value: "5", text: "5 sec" },
        { value: "15", text: "15 sec" },
        { value: "30", text: "30 sec" },
        { value: "60", text: "1 min" },
      ], defaultValue: "5"}
    ],
    acceptsConstructorProps: true,
    headerDragObjectType: DragObjectType.Camera,
    countLimit: 1
  }),

  new Widget(WidgetId.DownloadManager, __("Video Delivery Manager"), __("Video Delivery Manager"), "Video Delivery Manager", DownloadManagerIcon, {
    hideCondition: () => true
  }),

  new GeoMapWidget(WidgetId.GeoMap, __("GEO Map"), __("GEO Map"), "GEOMap", GEOMapIcon, {
    props: [
      { name: "defaultPosition", label: __("Default position and zoom"), controlType: ControlType.Checkbox },
    ]
  }),

  new Widget(WidgetId.LinkProvision, __("Avatar Network Discovery"), __("Avatar Network Discovery"), "Admin/LinkProvision", LinkProvisionIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.AvatarProvision, __("Avatar Activation"), __("Avatar Activation"), "Admin/AvatarActivation", LinkProvisionIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.HierarchyEdit, __("Hierarchy"), __("Hierarchy"), "LabelsAndHierarchies/HierarchyEditor", HierarchiesIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.LabelEdit, __("Labels"), __("Labels"), "LabelsAndHierarchies/LabelEditor", LabelsIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Users, __("Users"), __("Users"), "Admin/Users", UsersIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Groups, __("Groups"), __("Groups"), "Admin/Groups", GroupsIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Zones, __("Zones"), __("Zones"), "Admin/Zones", ZonesIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Sets, __("Sets"), __("Sets"), "Admin/Sets", SetsIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Policies, __("Authorization Policies"), __("Authorization Policies"), "Admin/Policies", PoliciesIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.AdminDevices, __("Devices"), __("Devices"), "Admin/AdminCameraList", DeviceListIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Reports, __("Reports"), __("Reports"), "Admin/ConfigurationStatusReport", ReportIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.AdminAvatars, __("Avatars"), __("Avatars"), "Admin/AdminLinkList", LinksIcon, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.NotificationPolicies, __("Notification Policies"), __("Notification Policies"), "NotificationPolicies", NotificationPolicies, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.RealmSettings, __("Realm Settings"), __("Realm Settings"), "RealmSettings", RealmSettings, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.ManageAvatars, __("Manage Avatars"), __("Manage Avatars"), "ManageAvatars", RealmSettings, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.WatchlistLpr, __("Watchlist: License Plate Recognition"), __("Watchlist: License Plate Recognition"), "Watchlist", Watchlist, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.WatchlistFacial, __("Watchlist: Facial"), __("Watchlist: Face Recognition"), "Watchlist", Watchlist, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.VideoWalls, __("Video Walls"), __("Video Walls"), "VideoWalls", VideoWalls, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.Subscription, __("Subscription"), __("Subscription"), "Subscription", About, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.QuickStart, __("Quick Start"), __("Quick Start"), "QuickStart", About, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.TrainingVideos, __("Training Videos"), __("Training Videos"), "TrainingVideos", About, {
    hideCondition: () => true
  }),

  new Widget(WidgetId.AdminSensors, __("Sensors"), __("Sensors"), "Sensors", SensorIcon, {
    hideCondition: () => true
  }),
  new Widget(WidgetId.AdminGateways, __("Gateways"), __("Gateways"), "Gateways", SensorIcon, {
    hideCondition: () => true
  }),
];

export const WidgetDragObjectType = "Widget";

export interface WidgetDragObject extends DragObjectWithType {
  id: WidgetId;
}

export type WidgetProps = {
  viewId?: ViewId;                                     // View unique ID
  index?: number;                                      // Index of widget within the view
  isViewActive?: boolean;                              // True if currently view is active (on active tab)
  widgetId?: WidgetId;                                 // ID of the widget
  cellProps?: Required<CellProps>;                     // Widget cell properties
  setCellProps?: (cellProps: CellProps) => void;       // call for setting widget cell properties
  // TODO: if this is event, it's should not be in props
  cellCloseEventPubSub?: CellCloseEventPubSub;         // Event on click close button in widget header
  headerClickEventPubSub?: HeaderClickEventPubSub;           // Widget header click event
}
& ViewConstructorProps
& ViewWidgetProps;

export type ViewConstructorProps = {
  inConstructor?: boolean;         // True in edit mode
  selectedInConstructor?: boolean; // True if selected in edit mode
};

export type ViewWidgetProps = {
  widgets?: ModuleInfo[];                   // List of view widgets
  loader?: RWidgetLoader;                   // Widget loader instance
  tabPanelRef?: React.RefObject<RTabPanel>; // Reference to main tab panel instance
  nextActiveViewId?: string;                // ID of view which will become active next
  alwaysUpdateWidgetContent?: boolean;      // Turn off widget redraw optimization (used in edit mode)
  addRowColEvent?: AddRowColEvent;          // Add grid row or column event (in edit mode)
  paddingClickEvent?: ViewIdEvent;          // Padding click event
  widgetEvent?: WidgetEvent;                // Widget event for interaction between widgets
  aspectRatio?: number;                     // View aspect ratio
};

export type CellProps = {
  title?: React.ReactNode;
  closable?: boolean;
  hideHeader?: boolean;
  hideIcon?: boolean;
  icon?: string;
  dragObject?: DragObjectWithType;
};

export type CellCloseArgs = {};
export class CellCloseEventPubSub extends EventPubSub<CellCloseArgs> {}

export type HeaderClickArgs = {};
export class HeaderClickEventPubSub extends EventPubSub<HeaderClickArgs> {}

export type WidgetEventArgs = {
  event: string;
  args: object;
};

export class WidgetEvent extends EventPubSub<WidgetEventArgs> {}

export enum CommonWidgetEvent {
  GetName = "GetName",
  Progress = "Progress",
  CancelProgress = "CancelProgress",
  EventClick = "EventClick",
  WidgetWarning = "WidgetWarning"
}

export type CommonWidgetEventArgs = {
  widgetIndex: number;
  name?: string;
};

export enum ProgressAction {
  Start = "Start",
  Progress = "Progress",
  End = "End"
}

export type ProgressEventArgs = {
  action: ProgressAction;
  percent: number;
  value?: number | string;
  label?: string;
  cancelable?: boolean;
};

export type EventClickEventArgs = {
  widgetIndex: number;
  deviceId: string;
  deviceName: string;
  startTime?: Date;
  endTime?: Date;
  categories?: string[];
  event?: EventInfo;
};

export type WidgetWarningEventArgs = {
  widgetIndex: number;
  warning?: WidgetWarning;
};

export function getEditModeWidgets(viewId: string, widgetCount: number): ModuleInfo[] {
  const widgets = new Array<ModuleInfo>(widgetCount);
  for (let i = 0; i < widgetCount; i++) {
    widgets[i] = {
      id: `${viewId}_widget${i}`,
      name: "",
      module: "Widgets/WidgetPlaceholder",
      props: { creatingViewId: viewId, creatingWidgetIndex: i } };
  }
  return widgets;
}

export function getPreviewWidgets(view: ModuleInfo): ModuleInfo[] {
  const { widgets = [] } = view;
  return widgets.map(w => ({ ...w, module: "Widgets/WidgetPreview", props: { ...w.props, module: w } }));
}

export * from "./lib";
export * from "./widgetPropGroups";
