import React, { useCallback } from "react";
import { withStore, WithStoreProps } from "@core/store";
import { Widgets, Widget } from "components/Widgets";
import WidgetListItem from "./WidgetListItem";

import "./style.css";

type WidgetListProps = {
  viewId: string;
} & WithStoreProps;

const WidgetList = ({ viewId, store: { workspace: { creatingViews }, session: { info } } }: WidgetListProps) => {
  const getInstance = useCallback((widget: Widget): Widget | undefined => {
    const view = creatingViews?.find(v => v.viewId + "_selectWidgets" === viewId);
    if (!view || !view.widgets) {
      return undefined;
    }
    return widget.createInstance(viewId, 0, view.widgets);
  }, [viewId, creatingViews]);

  const isDisabled = useCallback((widget: Widget): boolean => {
    const w = getInstance(widget);
    return !w || !w.isEnabled();
  }, [getInstance]);

  const isVisible = useCallback((widget: Widget): boolean => {
    const w = getInstance(widget);
    let isAllowed = true;
    if (info) {
      const { user } = info;
      const allowedWidgets = user.solidConfiguration.widgets.map(widgetConf => widgetConf.widgetId.toLowerCase());
      if (allowedWidgets && allowedWidgets?.length > 0) {
        isAllowed = allowedWidgets.includes(widget.id.toLowerCase());
      }
    }

    return !!w && w.isVisible() && isAllowed;
  }, [getInstance, info]);

  return (
    <div className="WidgetList">
      { Widgets.filter(item => isVisible(item)).map(item => <WidgetListItem key={item.id} widget={item} disabled={isDisabled(item)} />) }
    </div>
  );
};

export default withStore(WidgetList);
