import React from "react";
import { CreatingView } from "@generated/graphql";
import { withStore, WithStoreProps } from "@core/store";
import { RLayoutRender, LayoutType } from "components/Layout";
import { ViewProps } from "components/ViewLayouts";
import { copyObject, parseJSON } from "utils";
import { ViewLayoutItem } from "@core/types";
import { getEditModeWidgets } from "components/Widgets";
import { copyToSuffixedLayout } from "@core/store/actions";

type ViewCustomProps = {
  creatingView?: CreatingView;
} & ViewProps;

const ViewCustom = ({ viewId, name, index, addWidget, widgets: viewWidgets, layout: viewLayout, layoutRef, creatingView, ...restProps }: ViewCustomProps) => {
  const layout: ViewLayoutItem[] = parseJSON<ViewLayoutItem[]>(creatingView?.layoutJSON) ?? viewLayout ?? [];

  let widgets = viewWidgets;
  if (creatingView && creatingView.widgets) {
    widgets = getEditModeWidgets(creatingView.viewId, creatingView.widgets.length);
  }

  const widgetProps = { viewId, widgets: viewWidgets, ...restProps };

  function onGetLayout(): LayoutType | undefined {
    const layout = layoutRef?.current;
    if (!layout) {
      return undefined;
    }
    if (layout[viewId]) {
      return layout[viewId];
    }
    for (const suffix of ["_selectWidgets", "_preview"]) {
      if (viewId.endsWith(suffix)) {
        const id = viewId.replace(suffix, "");
        if (layout[id]) {
          return copyToSuffixedLayout(id, layout[id], suffix);
        }
      }
    }
    return undefined;
  }

  return (
    <RLayoutRender
      id={viewId}
      layout={layout}
      widgets={widgets}
      widgetProps={widgetProps}
      index={index}
      name={name}
      addWidget={addWidget}
      onGetLayout={onGetLayout}
      loader={widgetProps.loader} />
  );
};

type ViewWrapperProps = ViewProps & WithStoreProps;
type ViewWrapperState = {
  creatingView?: CreatingView;
};

class ViewWrapper extends React.Component<ViewWrapperProps, ViewWrapperState> {
  constructor(props: ViewWrapperProps) {
    super(props);
    this.state = {};
  }

  static getDerivedStateFromProps(props: ViewWrapperProps, state: ViewWrapperState): Partial<ViewWrapperState> | null {
    return { creatingView: props.store.workspace.creatingViews?.find(v => v.viewId + "_selectWidgets" === props.viewId) };
  }

  override shouldComponentUpdate(nextProps: ViewWrapperProps, nextState: ViewWrapperState): boolean {
    const compProps = copyObject(this.props, undefined, ["store", "setStore"]);
    const nextCompProps = copyObject(nextProps, undefined, ["store", "setStore"]);
    return JSON.stringify(compProps) !== JSON.stringify(nextCompProps) ||
      JSON.stringify(this.state.creatingView) !== JSON.stringify(nextState.creatingView);
  }

  override render() {
    const { store, setStore, ...props } = this.props;
    return (
      <ViewCustom {...props} creatingView={this.state.creatingView}/>
    );
  }
}

export default withStore(ViewWrapper);
