import React, { useState, useEffect, useCallback } from "react";
import { Button, Popup } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { ModuleInfo } from "@core/types";
import { useDisplays } from "@core/actions";
import { useConfig } from "@core/store/actions";
import {__} from "@solid/libs/i18n";

import "./style.css";
import { DisplayInfo } from "electron/screen";

type ViewEditFooterProps = {
  view: ModuleInfo;
  onClick?: () => void;
};

const ViewEditFooter = ({ view, onClick }: ViewEditFooterProps) => {
  const { displays } = useDisplays();
  const { views, setConfig } = useConfig();
  const [displayId, setDisplayId] = useState<number | undefined>();
  const [aspectRatio, setAspectRatio] = useState(view.aspectRatio);
  const [designDisplayId, setDesignDisplayId] = useState(view.designDisplayId);

  const iconWidth = 35;
  const iconHeight = 28;

  const getViewDisplayId = useCallback((displays: DisplayInfo[]): number | undefined => {
    let display = displays.find(d => d.id === displayId);
    if (display) {
      return display.id;
    }
    display = displays.find(d => d.id === designDisplayId);
    if (!display && aspectRatio && Math.abs(aspectRatio - 16 / 9) <= 0.01) {
      return undefined;
    }
    display = displays.find(d => aspectRatio && Math.abs(d.width / d.height - aspectRatio) <= 0.01);
    return display?.id;
  }, [displayId, aspectRatio, designDisplayId]);

  useEffect(useCallback(() => {
    setDisplayId(getViewDisplayId(displays));
  }, [displays, getViewDisplayId]), [displays]);

  const updateDisplayId = useCallback((displayId: number | undefined): void => {
    const display = displays.find(d => d.id === displayId);
    const aspectRatio = display ? display.width / display.height : (displayId === -1 ? undefined : 16 / 9);
    const designDisplayId = displayId === -1 ? undefined : displayId;

    setDisplayId(displayId);
    setAspectRatio(aspectRatio);
    setDesignDisplayId(designDisplayId);

    const index = views.findIndex(v => v.id === view.id);
    if (index >= 0) {
      const newViews = Array.from(views);
      newViews[index] = { ...newViews[index], aspectRatio, designDisplayId };
      setConfig({ views: newViews }, newViews[index].isLocal);
    }
  }, [displays, views, setConfig]);

  return (
    <div className="ViewEditFooter" onClick={() => onClick && onClick()}>
      <Button.Group onClick={(e: React.MouseEvent) => e.stopPropagation()}>
        <Button active={displayId === undefined && !!aspectRatio} onClick={() => updateDisplayId(undefined)}>
          <div>
            <FontAwesomeIcon icon="tv" style={{ width: iconWidth, height: iconHeight }}/>
            <small>16:9</small>
          </div>
          <span>{__("Default")}</span>
        </Button>

        { displays.map((display, index) => {
          const width = iconWidth / (16 / 9) * (display.width / display.height);
          return (
            <Popup key={display.id} content={display.name} trigger={
              <Button active={displayId === display.id} onClick={() => updateDisplayId(display.id)}>
                <div>
                  <FontAwesomeIcon icon="tv" style={{ width, height: iconHeight }} preserveAspectRatio="none"/>
                  <small style={{ left: (width - 6) / 2 }}>{index + 1}</small>
                </div>
                <span>{__("Display")} {index + 1}</span>
              </Button>
            }/>
          );
        }) }

        <Button active={displayId === -1 || !aspectRatio} onClick={() => updateDisplayId(-1)}>
          <div>
            <FontAwesomeIcon icon="tv" style={{ width: iconWidth, height: iconHeight }}/>
          </div>
          <span>{__("Auto")}</span>
        </Button>
      </Button.Group>
    </div>
  );
};

export default ViewEditFooter;
