import React, { useEffect, useCallback } from "react";
import { Accordion, Popup, Icon, List } from "semantic-ui-react";
import classNames from "classnames";
import { useAccordion, PanelProps } from "../accordion";
import { noDataSourceMessages, ViewManagerResult } from "components/View";
import EditControls from "components/View/Manager/EditControls";
import WidgetList from "components/Widgets/WidgetList";
import { WidgetPropGroup, useWidgetPropGroups } from "components/Widgets";
import {__} from "@solid/libs/i18n";

import "../style.css";

type EditActivityProps = {
  vm: ViewManagerResult
};

const EditActivity = ({ vm }: EditActivityProps) => {
  const { currentViewId } = vm;
  const basePanels = getBasePanels(vm, new Map<string, string>());
  const { panels, activeIndices, filter, filterEvent, setPanels, recreatePanels, setActiveIndices, onTitleClick } = useAccordion({
    panels: basePanels,
    activeIndices: new Set<number>(basePanels.map((p, index) => index)),
    recreatePanels: (vm, filter, panels) => {
      const newPanels = Array.from(panels);
      const basePanels = getBasePanels(vm, filter);
      newPanels.splice(0, basePanels.length, ...basePanels);
      return newPanels;
    }
  });
  useWidgetPropGroups({ currentViewId, onAddRemoveGroups, filterEvent });

  useEffect(() => {
    recreatePanels(vm, filter);
  }, [vm, filter]);

  useEffect(useCallback(() => {
    const widgetsIndex = getBasePanels(vm, filter).length;
    const widgetsKey = "Widgets";
    const selectWidgets = currentViewId.endsWith("_selectWidgets");

    setPanels(panels => {
      let newPanels = Array.from(panels);
      if (!selectWidgets) {
        newPanels = newPanels.filter(p => p.key !== widgetsKey);
      }
      if (selectWidgets && !panels.some(p => p.key === widgetsKey)) {
        const panel: PanelProps = {
          key: widgetsKey,
          label: __("Widgets"),
          panelContent: <WidgetList viewId={currentViewId}/>,
          contentClass: "EditActivity-WidgetListContent"
        };
        newPanels.splice(widgetsIndex, 0, panel);
      }
      return newPanels;
    });

    if (selectWidgets) {
      setActiveIndices(indices => new Set<number>(indices.add(widgetsIndex)));
    }
  }, [vm, filter, currentViewId]), [currentViewId]);

  function getBasePanels(vm: ViewManagerResult, filter: Map<string, string>): PanelProps[] {
    return [{ key: "Edit", label: __("View Info"), panelContent: <EditControls vm={vm}/>, extraTitleContent: getEditExtraTitleContent(vm) }];
  }

  function getEditExtraTitleContent(vm: ViewManagerResult) {
    const noSourceMessages = noDataSourceMessages(vm);
    return (
      <>
        { (!!vm.currentView?.hasUnsavedChanges || noSourceMessages) &&
        <Popup
          position="bottom right"
          content={
            <List>
              {!!vm.currentView?.hasUnsavedChanges &&
              <List.Item as="li">
                {`\u2217 ${__("View has unsaved changes.")}`}
              </List.Item>
              }
              {noSourceMessages &&
              <>
                <List.Item as="li">
                  {`\u2217 ${__("Some Widgets has no data source:")}`}
                </List.Item>
                <List>
                  {noSourceMessages.map((message, index) =>
                    <span key={`${index}_${message}`}>
                      <List.Item as="li" >
                        {`\u2013 ${message}`}
                      </List.Item>
                      <List.Item as="li">
                        {index < noSourceMessages.length - 1 && __("or")}
                      </List.Item>
                    </span>)}
                </List>
              </>}
            </List>}
          trigger={<Icon className="EditActivity-UnsavedChangesIcon" name="warning sign"/>}/> }
      </>
    );
  }

  function onAddRemoveGroups(addGroups?: WidgetPropGroup[], removeGroups?: WidgetPropGroup[]): void {
    setPanels(panels => {
      let newPanels = Array.from(panels);

      if (removeGroups) {
        newPanels = newPanels.filter(p => !removeGroups.some(s => s.key === p.key));
      }

      const addedIndices: number[] = [];
      if (addGroups) {
        for (const { key, name, content, acceptsFilter, scrollable, extraTitleContent } of addGroups.filter(s => !newPanels.some(p => p.key === s.key))) {
          if (content) {
            const panel: PanelProps =
              { key, label: name, panelContent: content, contentClass: classNames({ "EditActivity-WidgetPropContent_scrollable": scrollable }),
                showFilter: acceptsFilter, extraTitleContent };
            const index = newPanels.push(panel) - 1;
            addedIndices.push(index);
          }
        }
      }

      setActiveIndices(indices => {
        const newIndices = new Set<number>(indices);
        for (const index of addedIndices) {
          newIndices.add(index);
        }
        return newIndices;
      });

      return newPanels;
    });
  }

  return (
    <div className="EditActivity">
      <Accordion panels={panels} exclusive={false} activeIndex={Array.from(activeIndices.keys())}
        onTitleClick={(e, props) => onTitleClick(e as React.MouseEvent<HTMLDivElement>, props)}
        styled fluid className="EditActivity-Accordion"/>
    </div>
  );
};

export default EditActivity;
