import React, { useRef, useState } from "react";
import VDMRequestsTable from "../VDMRequestsTable";
import VDMControls from "../VDMControls";
import useVideoDelivery, { defaultVDMFilter, FilterPropType, VDMContentType, VDMFilterState } from "@core/actions/videoDelivery";
import VDMDownloadsTable from "../VDMDownloadsTable";
import { AodRequest } from "@generated/graphql";
import { __ } from "@solid/libs";

import "./style.css";

type VDMContentProp = {
  changeTitle: (title?: string) => void;
};

const VDMContent = ({ changeTitle }: VDMContentProp) => {
  const [filter, setFilter] = useState<VDMFilterState>(defaultVDMFilter);
  const [downloadsOpen, setDownloadsOpen] = useState<boolean>(false);
  const { list, pending, error, getCellContentText, updateRequests, cancelAll } = useVideoDelivery({ interval: filter.interval, filter, waitRequest: true });
  const downloadRequestRef = useRef<AodRequest | undefined>();

  function onFilterChange(newFilterValue: FilterPropType, key: keyof VDMFilterState) {
    setFilter(prevFilter => ({
      ...prevFilter,
      ...{[key]: newFilterValue}
    }));
  }

  function onBack() {
    setDownloadsOpen(false);
    changeTitle();
  }

  function onDownloadsOpen(rowData: AodRequest) {
    downloadRequestRef.current = rowData;
    setDownloadsOpen(true);
    const devName = getCellContentText({ dataKey: "deviceName", data: { content: VDMContentType.Request, rowData } });
    const from = getCellContentText({ dataKey: "from", data: { content: VDMContentType.Request, rowData } });
    const duration = getCellContentText({ dataKey: "duration", data: { content: VDMContentType.Request, rowData } });
    changeTitle(`${rowData.id} | ${devName} | ${from} | ${duration}`);
  }

  return (
    <div className="VDMContent">
      <VDMControls
        filter={filter}
        onFilterChange={onFilterChange}
        pending={pending}
        onBack={onBack}
        rowData={downloadRequestRef.current}
        controlType={downloadsOpen ? VDMContentType.Download : VDMContentType.Request}
        cancelAll={cancelAll}
      />
      {!downloadsOpen &&
        <VDMRequestsTable
          list={list}
          loading={pending}
          error={error}
          onDownloadsOpen={onDownloadsOpen}
          updateRequests={updateRequests}
          />}
      {downloadsOpen && downloadRequestRef.current &&
        <VDMDownloadsTable
          rowData={downloadRequestRef.current}
        />}
    </div>
  );
};

export default VDMContent;
