import React, {useEffect, useMemo, useState} from "react";
import VMXDesktop from "@libs/vmx/components/VMXDesktop";
import {VMXWall, VMXCell, VMXCellUpdate, VMXNewCell} from "@libs/vmx";
import {withStore} from "@core/store";
import {__} from "@solid/libs/i18n";
import {Type, UUID} from "@solid/types";
import {getSnapshotURL} from "@solid/libs";
import {fetchVMXWalls, fetchVMXCells, updateVMXCells, updateVMXCell, fetchVMXWall} from "./api";
import rfdc from "rfdc";
import { DeviceFunctionalAspectType, useDeviceListByAspectTypesQuery } from "@generated/graphql";

import "./style.css";

enum GridLayout {
  L_1 = "1",
  L_2x2 = "2x2",
  L_3x2 = "3x2",
  L_MAP = "map",
  L_VMX = "vmx"
}

type Dashboard = {
  layout: GridLayout,
  objectList: UUID[],
  vmxWall: UUID
};

const VMXDesktopWrapper = () => {
  const { loading: deviceLoading, data: deviceData } = useDeviceListByAspectTypesQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const [walls, setWalls] = useState<VMXWall[]>([]);
  const [cells, setCells] = useState([]);
  const [currentWall, setCurrentWall] = useState<VMXWall | null>(null);
  const [dashboard, setDashboard] = useState({} as Dashboard);

  const cameras = useMemo(() => deviceData ? deviceData.devicesByAspectTypes : [], [deviceData]);

  useEffect(() => {
    const wallId = localStorage.getItem("vmx.wall");
    if (wallId) {
      setDefaultCurrentWall(wallId);
    }
    return () => {
      setCurrentWall(null);
      setDashboard({} as Dashboard);
      setCells([]);
    };
  }, []);

  const setDefaultCurrentWall = async (wallId: UUID) => {
    const newDashboard: Dashboard = await rfdc()(dashboard);
    newDashboard.vmxWall = await wallId;
    const wall = await fetchVMXWall(wallId);
    await setCurrentWall(wall);
    await setDashboard(newDashboard);
  };

  const saveCurrentWall = (currentWallId: UUID): Promise<void> => {
    return new Promise<void>(async (resolve) => {
      const newDashboard: Dashboard = await rfdc()(dashboard);
      newDashboard.vmxWall = await currentWallId;
      const wall = await fetchVMXWall(currentWallId);
      await setCurrentWall(wall);
      await setDashboard(newDashboard);
      await localStorage.setItem("vmx.wall", currentWallId);
      resolve();
    });
  };

  const fetchWalls = (): Promise<void> => {
    return new Promise<void>(async (resolve) => {
      const data = await fetchVMXWalls(dashboard?.vmxWall);
      data.forEach(({wall_id}) => {
        fetchVMXWall(wall_id.id.uuid);
      });
      await setWalls(data);
      if (!currentWall) setCurrentWall(data[0]);
      resolve();
    });
  };

  const fetchCells = (wall_id: UUID): Promise<void> => {
    return new Promise<void>(async (resolve) => {
      if (wall_id) {
        const data = await fetchVMXCells(wall_id);
        await setCells(data.cells?.descriptors || []);
      }
      resolve();
    });
  };

  const updateCells = (wall_id: UUID, cells: (VMXCell | VMXNewCell)[]): Promise<void> => {
    return new Promise<void>(async (resolve) => {
      await updateVMXCells(wall_id, cells);
      if (wall_id) {
        const data = await fetchVMXCells(wall_id);
        await setCells(data.cells?.descriptors || []);
      }
      resolve();
    });
  };

  const updateCell = (wall_id: UUID, cell_id: UUID, cell: VMXCellUpdate): Promise<void> => {
    return new Promise<void>(async (resolve) => {
      await updateVMXCell(wall_id, cell_id, cell);
      await fetchCells(wall_id);
      resolve();
    });
  };

  const getCameraName = (id: UUID): string => {
    let name = "";
    if (cameras?.length) {
      for (let i = 0; i < cameras.length; i++) {
        if (cameras[i].id === id) {
          name = cameras[i].name;
          break;
        }
      }
    }
    return name;
  };

  return (
    <VMXDesktop
      fetchVMXWalls={fetchWalls}
      fetchVMXCells={fetchCells}
      updateVMXCells={updateCells}
      updateVMXCell={updateCell}
      cells={cells}
      walls={walls}
      currentWall={currentWall}
      isFetching={deviceLoading}
      loadingPlaceholder={__("Loading")}
      currentWallId={dashboard?.vmxWall}
      saveCurrentWall={saveCurrentWall}
      wallsLabel={__("Wall")}
      monitorsLabel={__("Monitor")}
      splitLabel={__("Split")}
      selectPlaceholder={__("Select")}
      dropType={Type.camera}
      getSnapshotURL={getSnapshotURL}
      touringHeader={__("Touring Control")}
      touringHelp={__("Drag and drop Camera to the list on the right to activate the touring")}
      touringTimeText={__("Delay time: {time} sec")}
      getCameraName={getCameraName}
    />
  );
};

export default withStore(VMXDesktopWrapper);
