import React, { useState, useEffect, useRef } from "react";
import { List, Popup, Icon, Input, Button } from "semantic-ui-react";
import classNames from "classnames";
import { TreeNode } from "./TreeView";
import {__} from "@solid/libs/i18n";

type TreeViewItemProps = {
  node: TreeNode;
  siblings: TreeNode[];
  selectedId: string;
  editId: string;
  expandedIdSet: Set<string>;
  selectable: boolean;
  editable: boolean;
  level: number;
  maxLevels: number;
  onClick: (node: TreeNode) => void;
  onItemClick: (node: TreeNode) => void;
  onExpandCollapse: (node: TreeNode, expand: boolean) => void;
  onEdit: (node: TreeNode) => void;
  onApplyEdit: (node: TreeNode, name: string) => void;
  onCancelEdit: (node?: TreeNode) => void;
  onNewChild: (node: TreeNode) => void;
  onDelete: (node: TreeNode) => void;
  onInputBlur: (node: TreeNode) => void;
};

const TreeViewItem = ({
  node,
  siblings,
  selectedId,
  editId,
  expandedIdSet,
  selectable,
  editable,
  level,
  maxLevels,
  onExpandCollapse,
  onClick,
  onItemClick,
  onEdit,
  onApplyEdit,
  onCancelEdit,
  onNewChild,
  onDelete,
  onInputBlur
}: TreeViewItemProps) => {
  const { id, content, nodes, icon, expandedIcon, className } = node;
  const [editName, setEditName] = useState(content?.toString() ?? "");
  const [okDisabled, setOkDisabled] = useState(false);
  const inputRef = useRef<Input>(null);

  useEffect(() => {
    setEditName(content?.toString() ?? "");
  }, [content]);

  useEffect(() => {
    setOkDisabled(!editName || siblings.some(item => item.id !== id && item.content?.toString() === editName));
  }, [siblings, id, editName]);

  useEffect(() => {
    if (editId === id) {
      window.requestAnimationFrame(() => {
        if (inputRef.current) {
          inputRef.current.focus();
        }
      });
    }
  }, [editId, id]);

  function onExpandCollapseClick(): void {
    onExpandCollapse(node, !expandedIdSet.has(id));
  }

  function newChild(): void {
    onNewChild(node);
  }

  function editItem(): void {
    onEdit(node);
  }

  function deleteItem(): void {
    onDelete(node);
  }

  function onInputKeyDown(e: React.KeyboardEvent): void {
    if (e.key === "Enter" && !okDisabled) {
      onApplyEdit(node, editName);
    }
    if (e.key === "Escape") {
      onCancelEdit(node);
    }
  }

  const isEdit = id === editId;
  const isExpanded = expandedIdSet.has(id);

  return (
    <List.Item
      value={id}
      className={classNames("TreeViewItem", { "selected": selectedId === id, selectable }, className)}
      onClick={() => onItemClick(node)}>
      <List.Content>
        <Icon
          name={nodes.length > 0 ? (isExpanded ? "triangle down" : "triangle right") : undefined}
          className={nodes.length > 0 ? "expand-collapse" : "empty"}
          onClick={nodes.length > 0 ? onExpandCollapseClick : undefined}
        />

        <div className="TreeViewItem-Content">
          {isEdit ?
            <Input
              ref={inputRef}
              placeholder={__("Name")}
              fluid
              action
              value={editName}
              onChange={e => setEditName(e.currentTarget.value)}
              onClick={(e: React.MouseEvent) => e.stopPropagation()}
              onKeyDown={onInputKeyDown}
              onBlur={(e: React.FocusEvent) => onInputBlur(node)}>
              <input/>
              <Button positive icon disabled={okDisabled} onClick={() => onApplyEdit(node, editName)}><Icon name="check"/></Button>
              <Button negative icon onClick={() => onCancelEdit(node)}><Icon name="cancel"/></Button>
            </Input> :
            <>
              <div className="TreeViewItem-IconAndName" onClick={() => onClick(node)}>
                {!!icon && <Icon name={isExpanded && nodes.length > 0 ? (expandedIcon ?? icon) : icon}/>}

                <div className="TreeViewItem-Name">
                  {content}
                </div>
              </div>

              {editable &&
              <div className="TreeViewItem-Buttons">
                {level < maxLevels - 1 &&
                <Popup trigger={
                  <Icon name="plus" onClick={newChild}/>}
                  content={__("Add Branch")}
                />}
                <Popup trigger={
                  <Icon name="edit" onClick={editItem}/>}
                  content={__("Edit")}
                />
                <Popup trigger={
                  <Icon name="trash alternate" onClick={deleteItem}/>}
                  content={__("Delete")}
                />
              </div>}
            </>}
        </div>
      </List.Content>

      {isExpanded &&
        <List>
          {nodes.map(node =>
            <TreeViewItem
              key={node.id}
              node={node}
              siblings={nodes}
              selectedId={selectedId}
              editId={editId}
              expandedIdSet={expandedIdSet}
              selectable={selectable}
              editable={editable}
              level={level + 1}
              maxLevels={maxLevels}
              onClick={onClick}
              onItemClick={onItemClick}
              onExpandCollapse={onExpandCollapse}
              onEdit={onEdit}
              onApplyEdit={onApplyEdit}
              onCancelEdit={onCancelEdit}
              onNewChild={onNewChild}
              onDelete={onDelete}
              onInputBlur={onInputBlur}
            />)}
        </List>}
    </List.Item>
  );
};

export default TreeViewItem;
