import React, { useState, useEffect, useRef } from "react";
import { List, SemanticICONS } from "semantic-ui-react";
import { v1 as uuid } from "uuid";
import { HierarchyLevel } from "@generated/graphql";
import TreeViewItem from "./TreeViewItem";
import { EventPubSub } from "utils";
import { clone } from "@solid/libs/utils";

import "./style.css";

export type TreeNode = {
  id: string;
  content: React.ReactNode;
  nodes: TreeNode[];
  icon?: SemanticICONS;
  expandedIcon?: SemanticICONS;
  className?: string;
  object?: HierarchyLevel;
};

export enum TreeViewEventType {
  AddRootItemCmd = "AddRootItemCmd",
  CancelEditCmd = "CancelEditCmd",
  CreateNewChildCmd = "CreateNewChildCmd"
}

export type TreeViewEventArgs = {
  type: TreeViewEventType;
  nodeId?: string;
};

export class TreeViewEvent extends EventPubSub<TreeViewEventArgs> {}

type TreeViewProps = {
  nodes: TreeNode[];
  selectedId?: string;
  event?: TreeViewEvent;
  selectable?: boolean;
  editable?: boolean;
  singleBranchExpand?: boolean;
  maxLevels?: number;
  expandedIds?: string[];
  onSelectedChange?: (id: string) => void;
  onTreeChange?: (nodes: TreeNode[]) => Promise<boolean>;
  onNewChild?: (node: TreeNode | undefined) => void;
  onExpandedChange?: (expandedIds: string[]) => void;
  onNodesUpdated?: () => void;
};

const TreeView = (props: TreeViewProps) => {
  const {
    event,
    selectable = true,
    editable = true,
    singleBranchExpand = false,
    maxLevels = 3,
    expandedIds,
    onSelectedChange,
    onTreeChange,
    onExpandedChange,
    onNodesUpdated
  } = props;
  const [nodes, setNodes] = useState(props.nodes);
  const [selectedId, setSelectedId] = useState("");
  const [editId, setEditId] = useState("");
  const [expandedIdSet, setExpandedIdSet] = useState(new Set<string>());
  const newItemsRef = useRef(new Set<string>());
  const inputBlurTimeoutRef = useRef<NodeJS.Timeout | undefined>();

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    event?.subscribe(id, args => {
      switch (args.type) {
        case TreeViewEventType.AddRootItemCmd:
          onNewChild();
          break;
        case TreeViewEventType.CancelEditCmd:
          onCancelEdit();
          break;
        case TreeViewEventType.CreateNewChildCmd:
          const node = args.nodeId ? findNode(args.nodeId, nodes) : undefined;
          createNewChild(node);
          break;
      }
    });

    return () => {
      event?.unsubscribe(id);
    };
  });

  useEffect(() => {
    setNodes(props.nodes);
    newItemsRef.current = new Set<string>();
  }, [props.nodes]);

  useEffect(() => {
    if (!selectable) {
      setSelectedId("");
      return;
    }
    if (!selectedId && nodes.length > 0) {
      setSelectedId(nodes[0].id);
    }
  }, [props.selectedId, nodes, selectable]);

  useEffect(() => {
    let id = selectable ? selectedId : "";
    if (id) {
      const node = findNode(id, nodes);
      if (!node) {
        id = nodes.length > 0 ? nodes[0].id : "";
        setSelectedId(id);
      }
    }
    onSelectedChange && onSelectedChange(id);
  }, [selectedId, nodes, selectable]);

  useEffect(() => {
    setExpandedIdSet(value => {
      if (value.size === 0) {
        return value;
      }

      const idSet = new Set<string>();
      getIdSet(nodes, idSet);

      let removed = false;
      const newValue = new Set<string>(value);
      for (const id of Array.from(value)) {
        if (!idSet.has(id)) {
          newValue.delete(id);
          removed = true;
        }
      }
      return removed ? newValue : value;
    });
    onNodesUpdated && onNodesUpdated();
  }, [nodes]);

  useEffect(() => {
    if (!expandedIds) {
      return;
    }

    setExpandedIdSet(value => {
      const currentIds = Array.from(value.keys()).sort();
      const newIds = Array.from(expandedIds).sort();
      if (JSON.stringify(currentIds) === JSON.stringify(newIds)) {
        return value;
      }
      return new Set<string>(newIds);
    });
  }, [expandedIds]);

  function onNodeClick(node: TreeNode): void {
    removeNewItems();
    setEditId("");
    if (selectable) {
      setSelectedId(node.id);
    }
  }

  function onExpandCollapse(node: TreeNode, expand: boolean): void {
    removeNewItems();
    setEditId("");
    expandCollapse(node, expand);
  }

  function expandCollapse(node: TreeNode, expand: boolean): void {
    setExpandedIdSet(value => {
      const newValue = new Set<string>(value);
      if (expand) {
        newValue.add(node.id);
        if (singleBranchExpand) {
          const parentNodes = findParentNodes(node.id, nodes);
          if (parentNodes) {
            for (const parent of parentNodes) {
              if (parent.id !== node.id) {
                newValue.delete(parent.id);
              }
            }
          }
        }
      }
      else {
        newValue.delete(node.id);
      }
      if (onExpandedChange) {
        window.requestAnimationFrame(() => onExpandedChange && onExpandedChange(Array.from(newValue.keys())));
      }
      return newValue;
    });
  }

  function onEdit(node: TreeNode): void {
    removeNewItems();
    setEditId(node.id);
    if (selectable) {
      setSelectedId(node.id);
    }
  }

  function findParentNode(childId: string, nodes: TreeNode[]): TreeNode | undefined  {
    const parent = nodes.find(node => node.nodes.some(node => node.id === childId));
    if (parent) {
      return parent;
    }
    for (const node of nodes) {
      const parent = findParentNode(childId, node.nodes);
      if (parent) {
        return parent;
      }
    }
    return undefined;
  }

  async function onApplyEdit(node: TreeNode, name: string): Promise<void> {
    const newNodes: TreeNode[] = clone(nodes);
    const parentNode = findParentNode(node.id, newNodes);
    if (parentNode && parentNode.object) {
      parentNode.object.labelIds = [];
    }
    if (setNodeName(node.id, name, newNodes)) {
      let updated = true;
      if (onTreeChange) {
        updated = await onTreeChange(newNodes);
      }
      if (updated) {
        setNodes(newNodes);
        newItemsRef.current.delete(node.id);
      }
      else {
        removeNewItems();
      }
    }
    setEditId("");
  }

  function onCancelEdit(node?: TreeNode): void {
    removeNewItems();
    setEditId("");
  }

  function onNewChild(node?: TreeNode): void {
    if (props.onNewChild) {
      props.onNewChild(node);
    }
    else {
      createNewChild(node);
    }
  }

  function createNewChild(node?: TreeNode): void {
    const newNodes = removeNewItems(true);
    const id = insertChild(node, newNodes);
    if (id) {
      setNodes(newNodes);
      setEditId(id);
      if (selectable) {
        setSelectedId(id);
      }
      if (node) {
        expandCollapse(node, true);
      }
      newItemsRef.current.add(id);
    }
  }

  function insertChild(node: TreeNode | undefined, nodes: TreeNode[]): string {
    if (!node) {
      const id = uuid();
      nodes.splice(0, 0, { id, content: "", nodes: [] });
      return id;
    }
    const index = nodes.findIndex(n => n.id === node.id);
    if (index >= 0) {
      const id = uuid();
      nodes[index].nodes.splice(0, 0, { id, content: "", nodes: [] });
      return id;
    }
    for (const child of nodes) {
      const id = insertChild(node, child.nodes);
      if (id) {
        return id;
      }
    }
    return "";
  }

  function removeNewItems(returnItemsClone?: boolean): TreeNode[] {
    if (!returnItemsClone && newItemsRef.current.size === 0) {
      return nodes;
    }
    const newNodes: TreeNode[] = clone(nodes);
    if (newItemsRef.current.size === 0) {
      return newNodes;
    }
    if (removeNewItemsFromNodes(newNodes)) {
      setNodes(newNodes);
    }
    newItemsRef.current = new Set<string>();
    return newNodes;
  }

  function removeNewItemsFromNodes(nodes: TreeNode[]): boolean {
    let removed = false;
    for (let i = nodes.length - 1; i >= 0; i--) {
      removed = removeNewItemsFromNodes(nodes[i].nodes) || removed;
      if (newItemsRef.current.has(nodes[i].id)) {
        nodes.splice(i, 1);
        removed = true;
      }
    }
    return removed;
  }

  function setNodeName(id: string, name: string, nodes: TreeNode[]): boolean {
    const node = nodes.find(node => node.id === id);
    if (node) {
      node.content = name;
      return true;
    }
    for (const node of nodes) {
      if (setNodeName(id, name, node.nodes)) {
        return true;
      }
    }
    return false;
  }

  async function onDelete(node: TreeNode): Promise<void> {
    setEditId("");
    const newNodes = removeNewItems(true);
    if (deleteNode(node, newNodes)) {
      let updated = true;
      if (onTreeChange) {
        updated = await onTreeChange(newNodes);
      }
      if (updated) {
        setNodes(newNodes);
      }
    }
  }

  function deleteNode(node: TreeNode, nodes: TreeNode[]): boolean {
    const index = nodes.findIndex(n => n.id === node.id);
    if (index >= 0) {
      nodes.splice(index, 1);
      return true;
    }
    for (const child of nodes) {
      if (deleteNode(node, child.nodes)) {
        return true;
      }
    }
    return false;
  }

  function findNode(id: string, nodes: TreeNode[]): TreeNode | undefined {
    const node = nodes.find(node => node.id === id);
    if (node) {
      return node;
    }
    for (const child of nodes) {
      const node = findNode(id, child.nodes);
      if (node) {
        return node;
      }
    }
    return undefined;
  }

  function getIdSet(nodes: TreeNode[], idSet: Set<string>): void {
    for (const node of nodes) {
      idSet.add(node.id);
      getIdSet(node.nodes, idSet);
    }
  }

  function findParentNodes(id: string, nodes: TreeNode[]): TreeNode[] | undefined {
    const node = nodes.find(node => node.id === id);
    if (node) {
      return nodes;
    }
    for (const child of nodes) {
      const nodes = findParentNodes(id, child.nodes);
      if (nodes) {
        return nodes;
      }
    }
    return undefined;
  }

  function onItemClick(node: TreeNode): void {
    if (inputBlurTimeoutRef.current) {
      clearTimeout(inputBlurTimeoutRef.current);
      inputBlurTimeoutRef.current = undefined;
    }
  }

  function onInputBlur(node: TreeNode): void {
    if (inputBlurTimeoutRef.current) {
      clearTimeout(inputBlurTimeoutRef.current);
    }
    inputBlurTimeoutRef.current = globalThis.setTimeout(() => onCancelEdit(node), 250);
  }

  return (
    <div className="TreeView">
      <List className="TreeView-List">
        {nodes.map(node =>
          <TreeViewItem
            key={node.id}
            node={node}
            siblings={nodes}
            selectedId={selectedId}
            editId={editId}
            expandedIdSet={expandedIdSet}
            selectable={selectable}
            editable={editable}
            level={0}
            maxLevels={maxLevels}
            onClick={onNodeClick}
            onItemClick={onItemClick}
            onExpandCollapse={onExpandCollapse}
            onEdit={onEdit}
            onApplyEdit={onApplyEdit}
            onCancelEdit={onCancelEdit}
            onNewChild={onNewChild}
            onDelete={onDelete}
            onInputBlur={onInputBlur}
          />)}
      </List>
    </div>
  );
};

export default TreeView;
