import React, { useCallback, useEffect, useMemo, useRef, useState } from "react";
import { Button, DropdownItemProps, Icon, Segment } from "semantic-ui-react";
import { useNavigate } from "react-router-dom";
import { EmailAuth, RealmInput, useRealmQuery, useUpdateRealmMutation, ViewId } from "@generated/graphql";
import { useAccessability } from "@core/store/actions/accessability";
import { __ } from "@solid/libs";
import FieldEditor, { FieldSchema } from "components/FieldEditor";
import IconButton from "components/IconButton";
import EmailValidation from "./EmailValidation";

import "./style.css";

const auditEventTtlOptions: DropdownItemProps[] = [
  {key: "0", value: "0", text: __("Cluster default (6 months)")},
  {key: "1", value: "1", text: __("7 days")},
  {key: "2", value: "2", text: __("15 days")},
  {key: "3", value: "3", text: __("1 month")},
  {key: "4", value: "4", text: __("3 months")},
  {key: "5", value: "5", text: __("6 months")},
  {key: "6", value: "6", text: __("1 year")},
  {key: "7", value: "7", text: __("2 years")}
];

const playerIdleTimeoutOptions: DropdownItemProps[] = [
  { key: "UNLIMITED", value: 0, text: __("UNLIMITED") },
  { key: "10 minutes", value: 10, text: __("10 minutes") },
  { key: "30 minutes", value: 30, text: __("30 minutes") },
  { key: "1 hour", value: 60, text: __("1 hour") },
  { key: "8 hours", value: 480, text: __("8 hours") },
  { key: "1 day", value: 1440, text: __("1 day") }
];

const emailAuthOptions: DropdownItemProps[] = [
  { key: EmailAuth.Pass, value: EmailAuth.Pass, text: __("Username/Password") },
  { key: EmailAuth.No, value: EmailAuth.No, text: __("NO AUTH") }
];

const maxDownloadTimeOptions: DropdownItemProps[] = [
  {key: 120, value: 120, text: __("2 hours")},
  {key: 240, value: 240, text: __("4 hours")},
  {key: 480, value: 480, text: __("8 hours")},
];

const informationFields = ["__typename", "emailValidated"];

const RealmSettings = () => {
  const navigate = useNavigate();
  const { config: { limitedRealmAdministration } } = useAccessability({ viewId: ViewId.RealmSettings });
  const {data, loading, error} = useRealmQuery({ fetchPolicy: "no-cache" });
  const [updateRealm, {data: updateData, loading: updateLoading, error: updateError}] = useUpdateRealmMutation();
  const [realmInput, setRealmInput] = useState<Record<string, any>>({});
  const [validateEmailModalOpen, setValidateEmailModalOpen] = useState<boolean>(false);
  const [emailSended, setEmailSended] = useState<boolean>(false);
  const [verificationEmail, setVerificationEmail] = useState<string>("");
  const [settingsChanged, setSettingsChanged] = useState<boolean>(false);
  const [emailAuth, setEmailAuth] = useState<EmailAuth>();
  const initialRealmRef = useRef<Record<string, any>>({});

  const generalFieldSchema: FieldSchema | undefined = useMemo(() => {
    if (!data || !data.realm) {
      return undefined;
    }

    const { realm } = data;
    const schema: FieldSchema = {
      viewType: "table",
      views: [{
        caption: __("General"),
        fields: ["name", "adminEmail", "maxScheduledDeliveryDelay", "auditTtl", "mfAuth", "playerIdleTimeout", "jitterBufferLength", "maxDownloadTime"]
      }],
      fields: {
        name: {
          label: __("System name"),
          type: "string",
          value: realm.name,
          onChange: (value) => updateInput("name", value)
        },
        adminEmail: {
          label: __("System Admin e-mail"),
          type: "string",
          value: realm.adminEmail,
          onChange: (value) => updateInput("adminEmail", value)
        },
        maxScheduledDeliveryDelay: {
          label: __("Maximum Mandatory Delivery delay, minutes"),
          type: "number",
          min: 0,
          max: 999,
          value: realm.maxScheduledDeliveryDelay,
          onChange: (value) => updateInput("maxScheduledDeliveryDelay", value)
        },
        auditTtl: {
          label: __("Audit events TTL"),
          type: "dropdown",
          value: realm.auditTtl,
          options: auditEventTtlOptions,
          onChange: (value) => updateInput("auditTtl", value)
        },
        mfAuth: {
          label: __("Two-Factor Auth (2FA)"),
          type: "boolean",
          value: realm.mfAuth,
          onChange: (value) => updateInput("mfAuth", value)
        },
        playerIdleTimeout: {
          label: __("Player idle timeout"),
          type: "dropdown",
          value: realm.playerIdleTimeout,
          options: playerIdleTimeoutOptions,
          onChange: (value) => updateInput("playerIdleTimeout", value)
        },
        jitterBufferLength: {
          label: __("Jitter buffer length, ms"),
          type: "number",
          min: 0,
          max: 3000,
          value: typeof realm.jitterBufferLength === "number" ? realm.jitterBufferLength : 1000,
          onChange: (value) => updateInput("jitterBufferLength", value)
        },
        maxDownloadTime: {
          label: __("Max download file time length, minutes"),
          type: "dropdown",
          value: realm.maxDownloadTime,
          options: maxDownloadTimeOptions,
          onChange: (value) => updateInput("maxDownloadTime", value)
        }
      }
    };

    return schema;
  }, [data]);

  const notificationsFieldSchema: FieldSchema | undefined = useMemo(() => {
    if (!data || !data.realm || limitedRealmAdministration) {
      return undefined;
    }

    const { realm } = data;
    const validatedValue = emailSended && verificationEmail ? __("An email with verification link was sent to {{email}}. Please check the email inbox to verify.", { email: verificationEmail })
      : realm.emailValidated ? "VALIDATED" : "NOT VALIDATED";

    const schema: FieldSchema = {
      viewType: "table",
      views: [{
        caption: __("Notifications"),
        fields: ["emailEnabled", "emailHost", "emailPort", "emailTls", "emailFrom", "emailAuth", "emailUser", "emailPass", "emailValidated"]
      }],
      fields: {
        emailEnabled: {
          label: __("SMTP server enabled"),
          type: "boolean",
          value: realm.emailEnabled,
          onChange: (value) => updateInput("emailEnabled", value)
        },
        emailHost: {
          label: __("SMTP server host"),
          type: "string",
          value: realm.emailHost,
          onChange: (value) => updateInput("emailHost", value)
        },
        emailPort: {
          label: __("SMTP server port"),
          type: "number",
          value: realm.emailPort,
          onChange: (value) => updateInput("emailPort", value)
        },
        emailTls: {
          label: __("Use SMTP TLS security"),
          type: "boolean",
          value: realm.emailTls,
          onChange: (value) => updateInput("emailTls", value)
        },
        emailFrom: {
          label: __("FROM field of mail"),
          type: "string",
          value: realm.emailFrom,
          onChange: (value) => updateInput("emailFrom", value)
        },
        emailAuth: {
          label: __("SMTP authentication"),
          type: "dropdown",
          value: realm.emailAuth,
          options: emailAuthOptions,
          onChange: (value) => updateInput("emailAuth", value)
        },
        emailUser: {
          label: __("SMTP user name"),
          type: "string",
          value: realm.emailUser,
          hidden: emailAuth === EmailAuth.No,
          onChange: (value) => updateInput("emailUser", value)
        },
        emailPass: {
          label: __("SMTP user password"),
          type: "password",
          value: realm.emailPass,
          hidden: emailAuth === EmailAuth.No,
          onChange: (value) => updateInput("emailPass", value)
        },
        emailValidated: {
          label: __("EMAIL validated"),
          type: "label",
          value: validatedValue,
          icon: <IconButton icon="check" hint={__("Validate")} onClick={() => setValidateEmailModalOpen(true)} />
        }
      }
    };

    return schema;
  }, [data, limitedRealmAdministration, emailSended, emailAuth]);

  const gisFieldSchema = useMemo(() => {
    if (!data || !data.realm) {
      return undefined;
    }

    const schema: FieldSchema = {
      viewType: "table",
      views: [{
        caption: __("GIS"),
        fields: ["ImportData"],
      }],
      fields: {
        ImportData: {
          label: __("Import GIS data"),
          type: "label",
          value: false,
          icon: <IconButton icon="map" hint={__("GeoMap")} onClick={() => {
            navigate("/view/EditMap");
          }} />
        }
      },
    };

    return schema;
  }, [data]);

  useEffect(() => {
    if (!data || !data.realm) {
      return;
    }

    const { realm } = data;
    const initialRealm = {};

    for (const fieldName of Object.keys(realm)) {
      initialRealm[fieldName] = realm[fieldName];
    }

    initialRealmRef.current = initialRealm;
    setRealmInput(initialRealm);
    setEmailAuth(initialRealm["emailAuth"]);

  }, [data]);

  useEffect(() => {
    if (!data || !data.realm || Object.keys(realmInput).length === 0) {
      return;
    }
    checkIsSettingsChanged(realmInput);
  }, [realmInput, data]);

  useEffect(() => {
    if (updateData && updateData.updateRealm) {
      initialRealmRef.current = {...realmInput};
      setSettingsChanged(false);
    }
  }, [updateData]);

  const onEmailValidationDialogClose = useCallback((sended: boolean, email?: string) => {
    setValidateEmailModalOpen(false);

    if (!emailSended && sended) {
      setEmailSended(true);
      email && setVerificationEmail(email);
    }
  }, [emailSended]);

  const update = useCallback(() => {
    const input: RealmInput = {
      id: realmInput.id,
      name: realmInput.name,
    };

    for (const fieldName of Object.keys(realmInput)) {
      if (!informationFields.includes(fieldName)) {
        input[fieldName] = realmInput[fieldName];
      }
    }

    if (input.emailAuth === EmailAuth.No) {
      input.emailPass = "";
      input.emailUser = "";
    }

    updateRealm({ variables: { realm: input } });

  }, [realmInput]);

  function updateInput(field: string, value: any) {
    if (field === "emailAuth") {
      setEmailAuth(value);
    }
    setRealmInput(prevState => ({...prevState, ...{ [field]: value }}));
  }

  function checkIsSettingsChanged(input: Record<string, any>) {
    const isChanged: boolean = JSON.stringify(input) !== JSON.stringify(initialRealmRef.current);
    setSettingsChanged(isChanged);
  }

  return (
    <Segment className="RealmSettings" loading={loading || updateLoading} error={error || updateError}>
      <div className="RealmSettings-TopButtons">
        <Button onClick={() => navigate(-1)}>
          <Icon name="arrow left"/>{__("Back")}
        </Button>
        {data &&
        <Button positive disabled={!settingsChanged} onClick={() => update()}>
          <Icon name="check"/>{__("Save")}
        </Button>}
      </div>
      <div className="RealmSettings-Tables">
        {generalFieldSchema && <FieldEditor schema={generalFieldSchema} />}
        {notificationsFieldSchema && <FieldEditor schema={notificationsFieldSchema} />}
        {gisFieldSchema && <FieldEditor schema={gisFieldSchema} />}
      </div>
      <EmailValidation dialogOpen={validateEmailModalOpen} onClose={onEmailValidationDialogClose} />
    </Segment>
  );
};

export default RealmSettings;
