import React, {useEffect, useRef, useState} from "react";
import classNames from "classnames";
import {Link} from "react-router-dom";
import {UUID} from "@solid/types";
import {MSEMediaPlayer} from "@solid/player/msemediaplayer";
import {Logger} from "@solid/libs/logger";
import {PlayerProps, PlayerOnClipEnd, toPlayerMetadataDrawType} from "@core/playerTypes";
import {__} from "@solid/libs/i18n";

import "@solid/player/mediaplayer.css";
import "@solid/player/mediaplayercontrols.css";
import "./PlayerCell.css";

type PlayerCellProps = PlayerProps;

const PlayerCell = React.forwardRef((props: PlayerCellProps, ref: React.Ref<HTMLDivElement>) => {
  const [timestamp, setTimestamp] = useState(0);
  const [isPlay, setIsPlay] = useState(false);

  const playerNodeRef = useRef<HTMLDivElement>(null);
  const timelineRef = useRef<HTMLDivElement>(null);

  const playerRef = useRef<MSEMediaPlayer | null>(null);

  // on mount
  useEffect(() => {
    const {obj, time, isActive} = props;
    createPlayer(obj, time, isActive);

    return () => {
      playerRef.current && playerRef.current.destroy();
    };
  }, []);

  useEffect(() => {
    if (!playerRef.current) {
      return;
    }
    const {obj, time, isActive} = props;
    if (obj && isActive) {
      playerRef.current.play(obj, time?.getTime());
    }
    else {
      playerRef.current.pause();
    }
  }, [props.obj, props.time, props.isActive]);

  const createPlayer = (obj: UUID, time?: Date, isActive?: boolean) => {
    if (!playerNodeRef.current) {
      return;
    }

    let player = playerRef.current;
    if (player) {
      player.destroy();
    }

    playerRef.current = new MSEMediaPlayer({node: playerNodeRef.current, logger: new Logger()});
    player = playerRef.current;
    player
      .init({obj, time: time?.getTime()})
      .then(() => {
        if (!player) return;

        player.subscribe("play", () => {
          setIsPlay(true);
        });

        player.subscribe("pause", () => {
          setIsPlay(false);
        });
        player.subscribe("stop", () => {
          setIsPlay(false);
          onStop();
        });

        player.subscribe("frame", (timestamp: number, width: number, height: number) => {
          if (!timestamp) return;

          setTimestamp(timestamp);
        });

        if (props.metadataType) {
          player.setMetadataDrawType(toPlayerMetadataDrawType(props.metadataType));
        }

        if (obj && isActive) {
          player.play();
        }
      });
  };

  function onStop() {
    if (props.onClipEnd === PlayerOnClipEnd.Loop) {
      playerRef.current && playerRef.current.play();
    }
    if (props.onStop) {
      props.onStop();
    }
  }

  let timestampString = "";
  if (timestamp) {
    const date = new Date();
    date.setTime(timestamp);
    timestampString = String(date.getHours()).padStart(2, "0") + ":" + String(date.getMinutes()).padStart(2, "0") + ":" + String(date.getSeconds()).padStart(2, "0")/* + "." + String(date.getMilliseconds()).padStart(3, "0")*/;
  }

  return (
    <div ref={ref} className={classNames("cell", { "cell_noHeader": props.hideHeader })}>
      { !props.hideHeader &&
      <div className="header">
        <div className="name">{props.header}</div>
        <i className="fas fa-times close" onClick={() => props.onClose()}/>
      </div> }
      <div className="video" ref={playerNodeRef}/>
      <div className="controls">
        <div className="timeline" ref={timelineRef}/>
        <div className="player-controls">
          <div className="left">
            {!props.isLiveOnly && <Link to={`/camera/${props.obj}/${encodeURIComponent(props.header)}/archive`}><button className="go-to-archive">{__("Archive")}</button></Link>}
            {!isPlay && <i className="fas fa-play" onClick={() => { playerRef.current && playerRef.current.play(); }}/>}
            {isPlay && <i className="fas fa-pause" onClick={() => { playerRef.current && playerRef.current.pause(); }}/>}
            <div className="timestamp">{timestampString}</div>
          </div>
          <div className="right"/>
        </div>
      </div>
    </div>
  );
});

export default PlayerCell;
