import React, {RefObject} from "react";
import ReactDOM from "react-dom";
import { BoxPanel, Widget } from "@lumino/widgets";
import { WrapperWidget } from "./RWidget";
import LazyComponent from "components/LazyComponent";
import { ModuleInfo } from "@core/types";
import {
  AddWidgetFunc,
  ModulesUpdateArgs,
  WidgetCloseArgs,
  LayoutType,
  RTabPanel,
  AddRowColEvent,
  AddRowColArgs,
  WidgetResizeEvent,
  ViewEvent,
  ViewIdEvent
} from "./index";
import {
  WidgetCloseEvent,
  WidgetCloseFunc,
  isWrapped,
  findChild,
  addWidget as addWidgetHelper,
  setMinMaxSize,
  getWidgetRect
} from "./RBaseWidget";
import { EventPubSub, EventPubSubFunc } from "utils";
import WidgetErrorBoundary from "./WidgetErrorBoundary";
import { WrapperBoxPanel } from "./RBoxPanel";
import { getModule } from "@core/utils";
import ViewHeader from "components/View/ViewHeader";
import ViewEditFooter from "components/View/ViewEditFooter";
import AddRowColPanel, { PanelSide } from "components/View/AddRowColPanel";
import { WidgetEvent } from "components/Widgets";

type ModulesUpdateFunc = EventPubSubFunc<ModulesUpdateArgs>;

class ModulesUpdateEvent extends EventPubSub<ModulesUpdateArgs> {}

type ModuleWidgetInfo = {
  id: string;
  name: string;
  renderComponents: JSX.Element[];
  addedWidget?: Widget;
  root?: WrapperBoxPanel;
  box0?: WrapperBoxPanel;
  box1?: WrapperBoxPanel;
  topPadding?: WrapperWidget;
  bottomPadding?: WrapperWidget;
  leftPadding?: WrapperWidget;
  rightPadding?: WrapperWidget;
  content?: Widget;
  aspectRatio?: number;
  widgets?: ModuleInfo[];
  addRowColEvent?: AddRowColEvent;
  paddingClickEvent?: ViewIdEvent;
  widgetEvent?: WidgetEvent;
};

type RWidgetLoaderProps = {
  modules: ModuleInfo[];
  layoutRef?: RefObject<LayoutType>;
  tabPanelRef?: RefObject<RTabPanel>;
  widgetClass?: string;
  showHeader?: boolean;
  ignoreFixedSizes?: boolean;
  hideCloseButton?: boolean;
  hideHeaderControls?: boolean;
  addWidget?: AddWidgetFunc;
  onChildClose?: WidgetCloseFunc;
  onChildClosing?: WidgetCloseFunc;
  onModulesUpdating?: ModulesUpdateFunc;
  onModulesUpdated?: ModulesUpdateFunc;
};

type RWidgetLoaderState = {
  widgetInfo: ModuleWidgetInfo[];
};

const headerHeight = 28;
const editFooterHeight = 80;
const addRowColSize = 28;

export class RWidgetLoader extends React.Component<RWidgetLoaderProps, RWidgetLoaderState> {
  private readonly closingWidgets = new Map<string, ModuleWidgetInfo>();
  private readonly restoringLayoutWidgets = new Map<string, ModuleWidgetInfo>();
  readonly childClose = new WidgetCloseEvent();
  readonly modulesUpdating = new ModulesUpdateEvent();
  readonly modulesUpdated = new ModulesUpdateEvent();
  readonly contentResize = new WidgetResizeEvent();
  readonly editView = new ViewEvent();
  readonly shareView = new ViewEvent();
  readonly deleteView = new ViewEvent();
  readonly viewToWindow = new ViewEvent();
  private subscriptionId = `${Date.now()}.${Math.random()}`;
  private layout: LayoutType | null | undefined;
  public nextActiveViewId?: string;
  private layoutChangeDisableCount = 0;
  private layoutChangeEnableTimeout?: NodeJS.Timeout;
  private _isMounted = false;

  constructor(props: RWidgetLoaderProps) {
    super(props);
    this.state = { widgetInfo: this.createWidgets() };
  }

  override componentDidMount() {
    this._isMounted = true;
  }

  /* eslint-disable react/no-did-update-set-state */
  override componentDidUpdate(prevProps: RWidgetLoaderProps) {
    if (!this.modulesEqual(prevProps.modules, this.props.modules) ||
        !this.modulesEqual(this.props.modules, this.state.widgetInfo)) {
      this.setState(state => ({ widgetInfo: this.createWidgets(state) }));
    }
  }
  /* eslint-enable react/no-did-update-set-state */

  override componentWillUnmount() {
    this._isMounted = false;

    this.childClose.unsubscribeAll();
    this.modulesUpdating.unsubscribeAll();
    this.modulesUpdated.unsubscribeAll();
    for (const { addedWidget } of this.state.widgetInfo) {
      if (addedWidget && isWrapped(addedWidget)) {
        addedWidget.wrapper.widgetClose.unsubscribe(this.subscriptionId);
      }
    }
  }

  override shouldComponentUpdate(nextProps: RWidgetLoaderProps, nextState: RWidgetLoaderState): boolean {
    return !this.modulesEqual(nextProps.modules, this.props.modules) ||
      !this.modulesEqual(nextState.widgetInfo, this.state.widgetInfo) ||
      !this.modulesEqual(nextProps.modules, nextState.widgetInfo);
  }

  private createWidgets(state?: RWidgetLoaderState): ModuleWidgetInfo[] {
    if (!this.props.addWidget) {
      return state ? state.widgetInfo : [];
    }

    this.layout = this.props.layoutRef ? this.props.layoutRef.current : undefined;

    let widgetInfo: ModuleWidgetInfo[] = [];
    if (state) {
      widgetInfo = widgetInfo.concat(state.widgetInfo);
    }

    const needUpdate = !this.modulesEqual(widgetInfo, this.props.modules);
    if (needUpdate) {
      this.onModulesUpdating(widgetInfo);
    }

    for (const info of Array.from(widgetInfo)) {
      const { id, addedWidget, widgets } = info;
      const module = this.props.modules.find(m => m.id === id);
      if (!module || JSON.stringify(module.widgets) !== JSON.stringify(widgets)) {
        if (addedWidget) {
          addedWidget.hide();
          this.closingWidgets.set(id, info);
          if (isWrapped(addedWidget)) {
            addedWidget.wrapper.widgetClose.unsubscribe(this.subscriptionId);
          }
        }
        widgetInfo = widgetInfo.filter(i => i.id !== id);
      }
    }

    for (let i = 0; i < this.props.modules.length; i++) {
      const moduleInfo = this.props.modules[i];
      const { id, name, module, hasLayout, widgets, layout, editMode, aspectRatio } = moduleInfo;

      if (!widgetInfo.some(info => info.id === id)) {
        const info: ModuleWidgetInfo = { id, name, aspectRatio, widgets, renderComponents: [], widgetEvent: new WidgetEvent() };

        const root = new WrapperBoxPanel(id, name, { direction: "top-to-bottom" });
        root.wrapper.widgetClose.subscribe(this.subscriptionId, (args: WidgetCloseArgs) => this.onChildClose(args));
        root.wrapper.widgetResize.subscribe(this.subscriptionId, ({ width, height }) => this.resize(id, undefined, width, height));
        info.root = root;
        info.addedWidget = this.props.addWidget(root, i);

        const tabPanel = this.props.tabPanelRef?.current;
        if (this.props.showHeader) {
          const node = document.createElement("div");
          const header = new WrapperWidget(node, `${id}_header`, name, { top: !!tabPanel });
          header.wrapper.setMinMaxSize(undefined, undefined, headerHeight, headerHeight);
          header.wrapper.disableLayout();
          addWidgetHelper(root, header);
          const component = ReactDOM.createPortal(
            <WidgetErrorBoundary>
              <ViewHeader view={moduleInfo} hideCloseButton={this.props.hideCloseButton} hideHeaderControls={this.props.hideHeaderControls}
                loader={this} onClose={id => this.closeWidget(id)}/>
            </WidgetErrorBoundary>, node);
          info.renderComponents.push(component);
        }

        info.topPadding = this.addPadding(info, root, "top");

        const box0 = new WrapperBoxPanel(`${id}_box_0`, "", { direction: "left-to-right" });
        info.box0 = box0;
        addWidgetHelper(root, box0);
        BoxPanel.setStretch(box0, 1);

        info.bottomPadding = this.addPadding(info, root, "bottom");

        info.leftPadding = this.addPadding(info, box0, "left");

        if (editMode) {
          this.createAddRowColPanel(info, box0, "left");
        }

        const box1 = new WrapperBoxPanel(`${id}_box_1`, "", { direction: "top-to-bottom" });
        info.box1 = box1;
        addWidgetHelper(box0, box1);
        BoxPanel.setStretch(box1, 1);

        if (editMode) {
          this.createAddRowColPanel(info, box0, "right");
        }

        info.rightPadding = this.addPadding(info, box0, "right");

        if (editMode) {
          this.createAddRowColPanel(info, box1, "top");
          this.createAddRowColPanel(info, box1, "bottom");
        }

        const contentIndex = editMode ? 1 : 0;
        const addWidget = (dock: Widget, index?: number) => {
          if (isWrapped(dock)) {
            dock.wrapper.widgetResize.subscribe(this.subscriptionId, ({ width, height }) => this.onContentResize(id, width, height));
          }
          info.content = addWidgetHelper(box1, dock, contentIndex);
          return info.content;
        };

        const props = {
          viewId: id,
          name,
          index: i,
          widgets,
          layout,
          addWidget,
          loader: this,
          tabPanelRef: this.props.tabPanelRef,
          nextActiveViewId: this.nextActiveViewId,
          layoutRef: this.props.layoutRef,
          widgetClass: this.props.widgetClass,
          addRowColEvent: info.addRowColEvent,
          paddingClickEvent: info.paddingClickEvent,
          widgetEvent: info.widgetEvent,
          aspectRatio,
          ...moduleInfo.props
        };

        const component = <LazyComponent render={getModule(module)} props={props} key={id} />;
        let content: JSX.Element;
        if (hasLayout) {
          content = component;
        }
        else {
          const node = document.createElement("div");
          const widget = new WrapperWidget(node, `${id}_0`, name, { top: !!tabPanel && !this.props.showHeader });
          addWidget(widget);
          content = ReactDOM.createPortal(<WidgetErrorBoundary>{component}</WidgetErrorBoundary>, node);
        }
        info.renderComponents.push(content);

        this.restoringLayoutWidgets.set(id, info);
        widgetInfo.push(info);

        if (editMode) {
          const node = document.createElement("div");
          const footer = new WrapperWidget(node, `${id}_footer`, name);
          footer.wrapper.setMinMaxSize(undefined, undefined, editFooterHeight, editFooterHeight);
          footer.wrapper.disableLayout();

          addWidgetHelper(root, footer);

          if (!info.paddingClickEvent) {
            info.paddingClickEvent = new ViewIdEvent();
          }

          const component = ReactDOM.createPortal(
            <WidgetErrorBoundary>
              <ViewEditFooter view={moduleInfo} onClick={() => info.paddingClickEvent?.publish({ id })}/>
            </WidgetErrorBoundary>, node);
          info.renderComponents.push(component);
        }
      }

      const index = widgetInfo.findIndex(w => w.id === id);
      if (index >= 0) {
        if (widgetInfo[index].name !== name) {
          widgetInfo[index] = { ...widgetInfo[index], name };
        }
        if (widgetInfo[index].aspectRatio !== aspectRatio) {
          widgetInfo[index] = { ...widgetInfo[index], aspectRatio };
          this.resize(id, widgetInfo[index]);
        }
      }
    }

    const { modules } = this.props;
    widgetInfo.sort((a, b) => modules.findIndex(m => m.id === a.id) - modules.findIndex(m => m.id === b.id));

    if (needUpdate) {
      window.requestAnimationFrame(() => {
        if (!this._isMounted) {
          return;
        }

        this.onModulesUpdated();
      });
    }

    this.nextActiveViewId = undefined;
    return widgetInfo;
  }

  private addPadding(info: ModuleWidgetInfo, parent: Widget, kind: "top" | "bottom" | "left" | "right"): WrapperWidget {
    const { id } = info;
    const node = document.createElement("div");
    const padding = new WrapperWidget(node, `${id}_${kind}_padding`, "");
    padding.addClass("Widget-Content_noSizeLimits");
    padding.addClass("Widget-Content_padding");
    padding.wrapper.disableLayout();

    addWidgetHelper(parent, padding);

    BoxPanel.setStretch(padding, 1);

    if (!info.paddingClickEvent) {
      info.paddingClickEvent = new ViewIdEvent();
    }

    node.onclick = () => info.paddingClickEvent?.publish({ id });

    padding.hide();
    return padding;
  }

  private createAddRowColPanel(info: ModuleWidgetInfo, parent: Widget, side: PanelSide): WrapperWidget {
    const node = document.createElement("div");
    const panel = new WrapperWidget(node, `${info.id}_${side}_addRowCol`, "");
    panel.addClass("Widget-Content_noSizeLimits");
    panel.wrapper.disableLayout();

    if (side === "top" || side === "bottom") {
      panel.wrapper.setMinMaxSize(undefined, undefined, addRowColSize, addRowColSize);
    }
    else {
      panel.wrapper.setMinMaxSize(addRowColSize, addRowColSize);
    }

    addWidgetHelper(parent, panel);

    if (!info.addRowColEvent) {
      info.addRowColEvent = new AddRowColEvent();
    }

    /* eslint-disable @typescript-eslint/indent */
    const component = ReactDOM.createPortal(
      <WidgetErrorBoundary>
        <AddRowColPanel viewId={info.id} side={side} contentResize={this.contentResize} onClick={side => {
          const args: AddRowColArgs = { side };
            info.addRowColEvent?.publish(args);
        }}/>
      </WidgetErrorBoundary>, node);
    /* eslint-enable @typescript-eslint/indent */

    info.renderComponents.push(component);
    return panel;
  }

  private closeWidgets() {
    for (const info of Array.from(this.closingWidgets.values())) {
      if (info.addedWidget) {
        info.addedWidget.close();
      }
    }
    this.closingWidgets.clear();
  }

  private updateWidgetsOrder() {
    if (!this.props.addWidget) {
      return;
    }
    for (let i = 0; i < this.state.widgetInfo.length; i++) {
      const { addedWidget } = this.state.widgetInfo[i];
      if (addedWidget) {
        this.props.addWidget(addedWidget, i);
      }
    }
  }

  private restoreWidgetsLayout() {
    if (this.layout) {
      for (const { id, addedWidget } of Array.from(this.restoringLayoutWidgets.values())) {
        const layout = this.layout[id];
        if (layout && addedWidget && isWrapped(addedWidget)) {
          addedWidget.wrapper.restoreLayout({ [id]: layout });
        }
      }
    }
    this.restoringLayoutWidgets.clear();
  }

  restoreWidgetLayout(widgetId: string, layout: LayoutType): void {
    const info = this.state.widgetInfo.find(info => info.id === widgetId);
    if (!info || !info.addedWidget || !isWrapped(info.addedWidget)) {
      return;
    }
    this.setLayoutChangeEnabled(false);
    info.addedWidget.wrapper.restoreLayout({ [widgetId]: layout });
    this.setLayoutChangeEnabled(true);
  }

  private updateWidgetTitles() {
    for (const { id, name } of this.props.modules) {
      const title = this.props.tabPanelRef?.current?.getTitle(id);
      if (title !== name) {
        this.props.tabPanelRef?.current?.setTitle(id, name);
      }
    }
  }

  private onModulesUpdating(widgetInfo?: ModuleWidgetInfo[]): void {
    this.setLayoutChangeEnabled(false, widgetInfo);
    this.modulesUpdating.publish({ modules: this.props.modules });
    if (this.props.onModulesUpdating) {
      this.props.onModulesUpdating({ modules: this.props.modules });
    }
  }

  private onModulesUpdated(): void {
    this.closeWidgets();
    this.updateWidgetsOrder();
    this.restoreWidgetsLayout();
    this.updateWidgetTitles();
    this.modulesUpdated.publish({ modules: this.props.modules });
    if (this.props.onModulesUpdated) {
      this.props.onModulesUpdated({ modules: this.props.modules });
    }
    this.setLayoutChangeEnabled(true);
  }

  private closeWidget(id: string): void {
    const info = this.state.widgetInfo.find(info => info.id === id);
    if (info?.addedWidget) {
      if (this.props.onChildClosing) {
        this.props.onChildClosing({ id, name: info.name });
      }
      else {
        info.addedWidget.close();
      }
    }
  }

  closeChild(id: string): void {
    const info = this.state.widgetInfo.find(info => info.id === id);
    info?.addedWidget?.close();
  }

  private onChildClose(args: WidgetCloseArgs) {
    const { id } = args;
    const info = this.state.widgetInfo.find(info => info.id === id);
    if (info?.addedWidget && isWrapped(info.addedWidget)) {
      info.addedWidget.wrapper.widgetClose.unsubscribe(this.subscriptionId);
    }
    if (info?.root) {
      info.root.wrapper.widgetResize.unsubscribe(this.subscriptionId);
    }
    if (info?.content && isWrapped(info.content)) {
      info.content.wrapper.widgetResize.unsubscribe(this.subscriptionId);
    }
    if (this.closingWidgets.has(id)) {
      this.closingWidgets.delete(id);
      return;
    }
    this.setState(({ widgetInfo }) => ({ widgetInfo: widgetInfo.filter(info => info.id !== id) }));
    this.childClose.publish(args);
    this.props.onChildClose && this.props.onChildClose(args);
  }

  private modulesEqual(a: Array<ModuleInfo | ModuleWidgetInfo>, b: Array<ModuleInfo | ModuleWidgetInfo>): boolean {
    return JSON.stringify(a.map(({ id, name, aspectRatio, widgets }) => ({ id, name, aspectRatio, widgets })).sort((a, b) => a.id.localeCompare(b.id))) ===
      JSON.stringify(b.map(({ id, name, aspectRatio, widgets }) => ({ id, name, aspectRatio, widgets })).sort((a, b) => a.id.localeCompare(b.id)));
  }

  private resize(id: string, info?: ModuleWidgetInfo, rootWidth?: number, rootHeight?: number): void {
    const module = this.props.modules.find(m => m.id === id);
    const wi = info ?? this.state?.widgetInfo.find(i => i.id === id);
    if (!module || !wi) {
      return;
    }

    const { aspectRatio, editMode, ignoreAspectRatio } = module;
    const { root, box0, box1, topPadding, bottomPadding, leftPadding, rightPadding } = wi;

    if (ignoreAspectRatio || !root || !root.parent || !box0 || !box1 || !topPadding || !bottomPadding || !leftPadding || !rightPadding) {
      return;
    }

    if (!aspectRatio) {
      this.setLayoutChangeEnabled(false);
      topPadding.hide();
      bottomPadding.hide();
      leftPadding.hide();
      rightPadding.hide();
      this.setLayoutChangeEnabled(true);
      return;
    }

    let width = rootWidth;
    let height = rootHeight;

    if (!width || !height) {
      const rect = getWidgetRect(root);
      width = rect.width;
      height = rect.height;
    }

    if (!width || !height) {
      return;
    }

    const rect = getWidgetRect(root.parent);
    if (rect.height - height > 100) {
      height = rect.height; // First time comes incorrect height, but parent has correct.
    }

    if (editMode) {
      height -= editFooterHeight;
    }
    if (editMode) {
      height -= addRowColSize * 2;
      width -= addRowColSize * 2;
    }

    this.setLayoutChangeEnabled(false);

    topPadding.hide();
    bottomPadding.hide();
    leftPadding.hide();
    rightPadding.hide();

    if (aspectRatio - width / height > 0.05) {
      const h = width / aspectRatio;
      const padding = (height - h) / 2;

      BoxPanel.setSizeBasis(box0, h);

      BoxPanel.setSizeBasis(topPadding, padding);
      topPadding.show();

      BoxPanel.setSizeBasis(bottomPadding, padding);
      bottomPadding.show();
    }
    else if (aspectRatio - width / height < -0.05) {
      const w = height * aspectRatio;
      const padding = (width - w) / 2;

      BoxPanel.setSizeBasis(box1, w);

      BoxPanel.setSizeBasis(leftPadding, padding);
      leftPadding.show();

      BoxPanel.setSizeBasis(rightPadding, padding);
      rightPadding.show();
    }

    this.setLayoutChangeEnabled(true);
  }

  private onContentResize(id: string, width: number, height: number): void {
    this.resize(id);
    this.contentResize.publish({ id, width, height});
  }

  findChild(widgetId: string): Widget | undefined {
    const info = this.state.widgetInfo.find(info => info.id === widgetId);
    if (info && info.addedWidget) {
      return info.addedWidget;
    }
    for (const { addedWidget } of this.state.widgetInfo) {
      let widget: Widget | undefined;
      if (addedWidget) {
        widget = findChild(addedWidget, widgetId);
      }
      if (widget) {
        return widget;
      }
    }
    return undefined;
  }

  addClass(widgetId: string, className: string): boolean {
    const widget = this.findChild(widgetId);
    if (widget) {
      widget.addClass(className);
    }
    return !!widget;
  }

  removeClass(widgetId: string, className: string): boolean {
    const widget = this.findChild(widgetId);
    if (widget) {
      widget.removeClass(className);
    }
    return !!widget;
  }

  setSelected(widgetId: string, selected: boolean, exclusive: boolean = false): boolean {
    const result = selected ? this.addClass(widgetId, "Widget-Content_selected") : this.removeClass(widgetId, "Widget-Content_selected");
    if (result && selected && exclusive) {
      const info = this.state.widgetInfo.find(info => info.widgets?.some(w => w.id === widgetId));
      if (info && info.widgets) {
        for (const widget of info.widgets) {
          if (widget.id !== widgetId) {
            this.setSelected(widget.id, false);
          }
        }
      }
    }
    return result;
  }

  setLayoutChangeEnabled(enabled: boolean, widgetInfo?: ModuleWidgetInfo[]): void {
    this.layoutChangeDisableCount = enabled ? Math.max(this.layoutChangeDisableCount - 1, 0) : this.layoutChangeDisableCount + 1;
    if (enabled && this.layoutChangeDisableCount > 0)  {
      return;
    }

    const wi = widgetInfo ?? this.state.widgetInfo ?? [];
    const setEnabled = () => {
      for (const info of wi) {
        if (info.root && isWrapped(info.root)) {
          info.root.wrapper.forceLayoutChangedEnabled(enabled);
        }
      }
    };

    if (!enabled) {
      setEnabled();
    }
    else {
      if (this.layoutChangeEnableTimeout) {
        clearTimeout(this.layoutChangeEnableTimeout);
      }
      this.layoutChangeEnableTimeout = setTimeout(setEnabled, 500);
    }
  }

  getWidgetEvent(viewId: string): WidgetEvent | undefined {
    return this.state.widgetInfo.find(wi => wi.id === viewId)?.widgetEvent;
  }

  setMinMaxSize(widgetId: string, minWidth?: number, maxWidth?: number, minHeight?: number, maxHeight?: number): void {
    const widget = this.findChild(widgetId);
    if (widget && isWrapped(widget)) {
      widget.hide();
      widget.wrapper.setMinMaxSize(minWidth, maxWidth, minHeight, maxHeight);
      setMinMaxSize(widget, true);
      widget.show();
    }
  }

  override render() {
    let components: JSX.Element[] = [];
    for (const info of this.state.widgetInfo) {
      components = components.concat(info.renderComponents);
    }
    return components;
  }
}
