import { SplitPanel, Widget } from "@lumino/widgets";
import { Message } from "@lumino/messaging";
import { RBaseWidget, RBaseWidgetProps, WrapperHelper, IWrappedWidget, ILayoutWidget } from "./RBaseWidget";
import { LayoutType } from "components/Layout";
import { isArray } from "utils";

type RSplitPanelLayout = {
  sizes: number[];
};

export class WrapperSplitPanel extends SplitPanel implements IWrappedWidget, ILayoutWidget {
  readonly wrapper: WrapperHelper;
  private restoringLayout: boolean = false;
  private restoredSizes: number[];

  constructor(id: string, name?: string, options?: SplitPanel.IOptions) {
    super(options);
    this.restoredSizes = this.relativeSizes();
    this.wrapper = new WrapperHelper(this, id, name);
  }

  override processMessage(msg: Message): void {
    super.processMessage(msg);
    if (msg.type === "update-request" && this.restoringLayout) {
      this.restoringLayout = false;
      this.wrapper.layoutRestored.publish({ id: this.id });
    }
  }

  override onCloseRequest(msg: Message) {
    super.onCloseRequest(msg);
    this.wrapper.publishClose();
  }

  protected override onResize(msg: Widget.ResizeMessage): void {
    super.onResize(msg);
    this.wrapper.widgetResize.publish(msg);
  }

  getLayout(): LayoutType | undefined {
    const layout: RSplitPanelLayout = { sizes: this.relativeSizes() };
    return layout;
  }

  restoreLayout(obj: LayoutType): void {
    const layout = obj as RSplitPanelLayout;
    if (isArray<number>(layout.sizes, "number") && JSON.stringify(layout.sizes) !== JSON.stringify(this.relativeSizes())) {
      this.restoringLayout = true;
      const sizes = Array.from(layout.sizes);
      if (this.widgets.length > sizes.length) {
        const avg = 1 / (sizes.length ? sizes.length : this.widgets.length);
        while (sizes.length < this.widgets.length) {
          sizes.push(avg);
        }
        const sum = sizes.reduce((accumulator, currentValue) => accumulator + currentValue, 0);
        if (sum > 0) {
          const ratio = 1 / sum;
          for (let i = 0; i < sizes.length; i++) {
            sizes[i] *= ratio;
          }
        }
      }

      this.restoredSizes = sizes;

      // Sometimes sizes are not restored from the first attempt.
      // Apply sizes again if they were changed during 1 second after first set.
      // TODO: find the root cause of the issue.
      this.forceRelativeSizes(sizes);
    }
    else {
      this.wrapper.layoutRestored.publish({ id: this.id });
    }
  }

  forceRelativeSizes(sizes: number[]): void {
    let iter = 0;
    const checkSizes = (sizes: number[]): void => {
      if (!this.sizesEqual(this.relativeSizes(), sizes)) {
        this.setRelativeSizes(sizes);
        return;
      }
      iter++;
      if (iter < 10) {
        setTimeout(() => checkSizes(sizes), 100);
      }
    };

    this.setRelativeSizes(sizes);
    setTimeout(() => checkSizes(sizes), 100);
  }

  sizesEqual(a: number[], b: number[], delta?: number): boolean {
    if (a.length !== b.length) {
      return false;
    }
    for (let i = 0; i < a.length; i++) {
      if (Math.abs(a[i] - b[i]) >= (delta ?? 0.05)) {
        return false;
      }
    }
    return true;
  }

  childResized(id: string): void {
    if (!this.sizesEqual(this.restoredSizes, this.relativeSizes(), 0.01)) {
      this.restoredSizes = this.relativeSizes();
      this.wrapper.publishLayoutChanged(this.id);
    }
    else {
      let parent = this.parent;
      while (parent) {
        if (parent instanceof WrapperSplitPanel) {
          parent.childResized(id);
          break;
        }
        parent = parent.parent;
      }
    }
  }
}

type RSplitPanelProps = RBaseWidgetProps & {
  orientation: SplitPanel.Orientation,
  sizes?: number[];
};

export class RSplitPanel extends RBaseWidget<RSplitPanelProps, WrapperSplitPanel> {
  protected createLayoutContainer(): WrapperSplitPanel {
    return new WrapperSplitPanel(this.props.id, this.props.name, { orientation: this.props.orientation });
  }

  override addWidget(dock: Widget, index?: number): Widget {
    const widget = super.addWidget(dock, index);
    this.layoutContainer && this.props.sizes && this.layoutContainer.setRelativeSizes(this.props.sizes);
    return widget;
  }

  override render() {
    return this.renderComponent();
  }
}
