import React, {useEffect, useRef, useState} from "react";
import { Icon, Popup, List } from "semantic-ui-react";
import { Label } from "@generated/graphql";
import classNames from "classnames";
import { useSnapshot } from "@core/actions";
import Loading from "components/Loading";
import { getLabelIcon } from "components/LabelsAndHierarchies/utils";
import {__} from "@solid/libs/i18n";

import "./style.css";

type CameraListItemProps = {
  deviceId: string;
  name: string;
  labels: Label[];
  onClick?: (deviceId: string, name: string) => void;
};

const maxLabelNames = 10;

const CameraListItem = ({ deviceId, name, labels, onClick }: CameraListItemProps) => {
  const imageRef = useRef<HTMLImageElement>(null);
  const { loading, imageSrc } = useSnapshot({ load: true, deviceId, imageRef });
  const [labelList, setLabelList] = useState<Label[]>([]);

  useEffect(() => {
    const labelList = [...labels];
    labelList
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .slice(0, maxLabelNames);
    setLabelList(labelList);
  }, [labels]);

  return (
    <div className={classNames("THCameraListItem", { "clickable": !!onClick })} onClick={onClick ? () => onClick(deviceId, name) : undefined}>
      <div className="THCameraListItem-Snapshot">
        <img ref={imageRef} src={imageSrc} alt={name} className={classNames({ "THCameraListItem-Image_loading": loading || !imageSrc })}/>
        { loading && <Loading/> }
      </div>
      <div className="THCameraListItem-Text">
        <span>{name}</span>
        {labels.length > 0 &&
        <Popup
          trigger={<Icon name="tags"/>}
          content={
            <>
              <b>{__("Labels")}</b>
              <hr/>
              <List>
                {labelList
                  .map(label =>
                    <List.Item key={label.id}>
                      <List.Icon name={getLabelIcon(label)}/>
                      <List.Content>{label.name}</List.Content>
                    </List.Item>)}
              </List>
              {labels.length > maxLabelNames && <div>{__("...and {{number}} more", {number: labels.length - maxLabelNames})}</div>}
            </>
          }
        />}
      </div>
    </div>
  );
};

export default CameraListItem;
