import React, { useState, useEffect, useMemo } from "react";
import { Message, Button, Icon, Divider } from "semantic-ui-react";
import {useDevicesByAspectTypesShortQuery, useLabelsQuery, DeviceFunctionalAspectType, Label} from "@generated/graphql";
import WithQueryStatus from "components/WithQueryStatus";
import CameraListItem from "components/LabelsAndHierarchies/CameraListItem";
import AssignLabelsDialog from "components/LabelsAndHierarchies/AssignLabelsDialog";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import {DeviceShort} from "@core/actions";

import "./style.css";

type CameraListProps = {
  searchName?: string;
  searchLabels?: Label[];
  editable?: boolean;
  mandatoryLabelSearch?: boolean;
};

type DeviceShortLabels = DeviceShort & {
  labels: Label[];
};

const loadStep = 30;
const loadByNameStep = loadStep / 2;
const loadByLabelsStep = loadStep / 2;

const CameraList = ({ searchName, searchLabels = [], editable = true, mandatoryLabelSearch = false }: CameraListProps) => {
  const { loading, error, data } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const { loading: labelLoading, error: labelError, data: labelData } = useLabelsQuery();
  const [devices, setDevices] = useState<DeviceShortLabels[]>([]);
  const [devicesByName, setDevicesByName] = useState<DeviceShortLabels[]>([]);
  const [devicesByLabels, setDevicesByLabels] = useState<DeviceShortLabels[]>([]);
  const [loadLimit, setLoadLimit] = useState(loadStep);
  const [loadByNameLimit, setLoadByNameLimit] = useState(loadByNameStep);
  const [loadByLabelsLimit, setLoadByLabelsLimit] = useState(loadByLabelsStep);
  const [assignLabelsOpen, setAssignLabelsOpen] = useState(false);
  const [deviceId, setDeviceId] = useState("");
  const [deviceName, setDeviceName] = useState("");
  const deviceLabels = useMemo(() => getDeviceLabels(), [labelData]);

  useEffect(() => {
    error && console.error("Devices query error:", error);
  }, [error]);

  useEffect(() => {
    if (labelError) {
      console.error("Labels query error:", labelError);
      Log.error(`${__("Labels query error")}: ${labelError.message}`);
    }
  }, [labelError]);

  useEffect(() => {
    if (!data) {
      return;
    }

    const text = searchName?.toLocaleUpperCase();
    const devByName = text ? data.devicesByAspectTypes
      .filter(dev => dev.name.toLocaleUpperCase().includes(text))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(dev => ({ ...dev, labels: deviceLabels.get(dev.id) ?? [] })) : [];

    const devByLabels = searchLabels.length > 0 || mandatoryLabelSearch ? data.devicesByAspectTypes
      .filter(dev => searchLabels.some(label => label.objects.some(object => object.objectId === dev.id)))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(dev => ({ ...dev, labels: deviceLabels.get(dev.id) ?? [] })) : [];

    const devices = [...data.devicesByAspectTypes];
    const devAll = !text && searchLabels.length === 0 && !mandatoryLabelSearch
      ? devices
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
        .map(dev => ({ ...dev, labels: deviceLabels.get(dev.id) ?? [] }))
      : [];

    setDevices(devAll);
    setDevicesByName(devByName);
    setDevicesByLabels(devByLabels);

    setLoadLimit(loadStep);
    setLoadByNameLimit(loadByNameStep);
    setLoadByLabelsLimit(loadByLabelsStep);
  }, [data, deviceLabels, searchName, searchLabels]);

  function getDeviceLabels(): Map<string, Label[]> {
    const map = new Map<string, Label[]>();
    if (!labelData) {
      return map;
    }
    for (const label of labelData.labels) {
      const deviceIds = new Set<string>(label.objects.map(({ objectId }) => objectId));
      for (const deviceId of Array.from(deviceIds.keys())) {
        let labels = map.get(deviceId);
        if (!labels) {
          labels = [];
          map.set(deviceId, labels);
        }
        labels.push(label);
      }
    }
    return map;
  }

  function onLoadMoreClick(): void {
    setLoadLimit(value => value + loadStep);
    setLoadByNameLimit(value => value + loadByNameStep);
    setLoadByLabelsLimit(value => value + loadByLabelsStep);
  }

  function onCameraClick(deviceId: string, name: string): void {
    setDeviceId(deviceId);
    setDeviceName(name);
    setAssignLabelsOpen(true);
  }

  let deviceCount = devices.length;
  if (searchName && (searchLabels.length > 0 || mandatoryLabelSearch)) {
    deviceCount = devicesByName.length + devicesByLabels.length;
  }
  else if (searchName) {
    deviceCount = devicesByName.length;
  }
  else if (searchLabels.length > 0 || mandatoryLabelSearch) {
    deviceCount = devicesByLabels.length;
  }

  const byNameLimit = searchName && searchLabels.length > 0 ? loadByNameLimit : loadLimit;
  const byLabelsLimit = searchName && searchLabels.length > 0 ? loadByLabelsLimit : loadLimit;

  return (
    <div className="THCameraList">
      <WithQueryStatus loading={loading || labelLoading} error={error}>
        {deviceCount === 0 ?
          <div className="THCameraList-Empty">
            <Message info icon>
              <Icon name="info circle"/>
              <Message.Content>
                <Message.Header>{__("No devices selected")}</Message.Header>
              </Message.Content>
            </Message>
          </div> :

          <div className="THCameraList-Content">
            <div className="THCameraList-List">
              {!searchName && searchLabels.length === 0 && !mandatoryLabelSearch ?
                <>
                  <div className="THCameraList-Divider">
                    <div>{__("All Devices")}</div>
                    <Divider/>
                  </div>

                  {devices.slice(0, loadLimit).map(({ id, name, labels }) =>
                    <CameraListItem key={id} deviceId={id} name={name} labels={labels} onClick={editable ? onCameraClick : undefined}/>)}
                </> :

                <>
                  {searchName &&
                  <div className="THCameraList-Divider">
                    <div>{__("Filtered by Name")}</div>
                    <Divider/>
                  </div>}

                  {devicesByName.slice(0, byNameLimit).map(({ id, name, labels }) =>
                    <CameraListItem key={id} deviceId={id} name={name} labels={labels} onClick={editable ? onCameraClick : undefined}/>)}

                  {(searchLabels.length > 0 || mandatoryLabelSearch) &&
                  <div className="THCameraList-Divider">
                    <div>{__("Filtered by Labels")}</div>
                    <Divider/>
                  </div>}

                  {devicesByLabels.slice(0, byLabelsLimit).map(({ id, name, labels }) =>
                    <CameraListItem key={id} deviceId={id} name={name} labels={labels} onClick={editable ? onCameraClick : undefined}/>)}
                </>}
            </div>
          </div>}

        {(devices.length > loadLimit || devicesByName.length > byNameLimit || devicesByLabels.length > byLabelsLimit) &&
        <div className="THCameraList-Footer">
          <Message info icon className="THCameraList-Message">
            <Icon name="info circle"/>
            <Message.Content>
              <Message.Header>{__("Found {{number}} matches...", {number: new Intl.NumberFormat().format(deviceCount)})}</Message.Header>
              <Button onClick={onLoadMoreClick}>{__("Load more...")}</Button>
            </Message.Content>
          </Message>
        </div>}

        <AssignLabelsDialog open={assignLabelsOpen} deviceId={deviceId} name={deviceName} onClose={() => setAssignLabelsOpen(false)}/>
      </WithQueryStatus>
    </div>
  );
};

export default CameraList;
