/* eslint-disable react/jsx-boolean-value */
import React, { ChangeEvent, useCallback, useEffect, useMemo, useRef, useState } from "react";
import { MapContainer, TileLayer, LayersControl, GeoJSON } from "react-leaflet";
import {
  DeviceFunctionalAspectType,
  LatLng,
  LatLngInput,
  AuditEntry,
  useGeoLayersQuery,
  useUpdateGeoLayersMutation,
  useSystemParametersQuery,
  useDevicesByAspectTypesQuery
} from "@generated/graphql";
import { Device, isError, useAuditEntries } from "@core/actions";
import DraggableMarker from "./DraggableMarker";
import WithQueryStatus from "components/WithQueryStatus";
import ResizeComponent from "./ResizeComponent";
import ReviewCompoent from "./ReviewCompoent";
import { UUID, Event, PlayerMode } from "@solid/types";
import ReactLeafletGoogleLayer from "react-leaflet-google-layer";
import { Button, Form, Header, Icon, Input, Label, List, ListItem, Message, Modal, Popup, Segment } from "semantic-ui-react";
import { CommonWidgetEvent, EventClickEventArgs, WidgetEventArgs, WidgetProps } from "components/Widgets";
import { clone, isElectron, __ } from "@solid/libs";
import classNames from "classnames";
import mapIcon from "./images/marker-icon-grey.png";
import leaflet, { GeoJSON as LeafletGeoJson, LatLngTuple } from "leaflet";
import { v4 as uuid } from "uuid";
import produce from "immer";
import axios from "axios";
import { kml } from "@tmcw/togeojson";
import { useLocation } from "react-router-dom";
import { Log } from "@solid/libs/log";
import { PlayerEvent, PlayerEventArgs } from "@core/playerTypes";
import {GEOMapProperties} from "./GEOMapCellStub/GEOMapCellStub";

import "leaflet/dist/leaflet.css";
import "./style.css";

type EventWithEntry = Event & {
  entry: AuditEntry;
};

const defaultMaxDeviceEvents = 10;
const eventCategories = ["47"];
const nullPosition: [number, number] = [0, 0];

type DeviceTracking = {
  playerState?: { mode?: PlayerMode.Archive | PlayerMode.Live, lastPlayedTime?: number };
  data?: LatLngTuple[];
  device: Device;
};

type DeviceEventsWithEntry = {
  [id in UUID]: {
    isChecked: boolean;
    events: EventWithEntry[];
  }
};

type PointFeatureGeometry = {
  type: "Point",
  coordinates: [number, number]
};

type PointFeatureProperties = {
  id: string,
  name: string
};

type PointFeatureCollection = GeoJSON.FeatureCollection<PointFeatureGeometry, PointFeatureProperties>;

type ImportLayersDatatype = GeoJSON.FeatureCollection<GeoJSON.Geometry | null, GeoJSON.GeoJsonProperties> & {
  id: UUID,
  name: string,
  visible: boolean
};

type GEOMapProps = WidgetProps & {
  id?: UUID,
  setPosition?: (position: LatLng) => void,
  defaultPositionProperties?: GEOMapProperties
  position?: LatLngInput,
  isView?: boolean,
  withGISSettings?: boolean,
  defaultPosition?: boolean
  setDefaultPositionToStore?: (position: GEOMapProperties) => void
};

enum MapTypes {
  "OSM",
  "satellite",
  "roadmap",
  "hybrid"
}

type GoogleMutantType = "roadmap" | "satellite" | "terrain" | "hybrid";

const google: GoogleMutantType[] = ["satellite", "roadmap", "hybrid"];

const queryTypes = [
  { type: DeviceFunctionalAspectType.Media },
  { type: DeviceFunctionalAspectType.Sensor },
];

const GEOMap = ({id, setPosition, position, cellProps, setCellProps, isView = true, widgetEvent, index: widgetIndex = 0, defaultPosition: defaultPositionProp = false, setDefaultPositionToStore, defaultPositionProperties, ...props}: GEOMapProps) => {
  const [createPosition, setCreatePosition] = useState<LatLng>();
  const { data: systemParamData, error: systemParamError, loading: systemParamLoading } = useSystemParametersQuery();
  const { data, loading, error } = useDevicesByAspectTypesQuery({ variables: { types: queryTypes } });
  const [updateGeoLayers, {/* data: updateGeoLayersData, */ loading: updateGeoLayersLoading, error: updateGeoLayersError }] = useUpdateGeoLayersMutation();
  const { data: geoLayersData, loading: geoLayersLoading, error: geoLayersError, refetch } = useGeoLayersQuery();
  const [devices, setDevices] = useState<Device[]>([]);
  const [positions, setPositions] = useState<Map<UUID, LatLngTuple>>(new Map());
  const [device, setDevice] = useState<Device | undefined>(undefined);
  const [center, setCenter] = useState<leaflet.LatLngTuple | undefined>(undefined);
  const [APIKey, setAPIKey] = useState<string | undefined>(undefined);
  const [mapDefaultLayer, setMapDefaultLayer] = useState<keyof typeof MapTypes | undefined>(undefined);
  const [defaultPosition, setDefaultPosition] = useState<leaflet.LatLngTuple>();
  const [exportGeoJson, setExportGeoJson] = useState<boolean>(false);
  const [gisLayersHidden, setGisLayersHidden] = useState<boolean>(true);
  const [editField, setEditField] = useState<ImportLayersDatatype | undefined>(undefined);
  const [newFieldName, setNewFieldName] = useState<string>("");
  const [layersData, setLayersData] = useState<ImportLayersDatatype[]>([]);
  const [layersLatLngList, setLayersLatLngList] = useState<LatLngTuple[]>([]);
  const [showDevices, setShowDevices] = useState<boolean>(true);
  const [showAllMarkers, setShowAllMarkers] = useState<boolean>(false);
  const [reviewToMarkers, setReviewToMarkers] = useState<boolean>(true);
  const uploadGisFile = useRef<HTMLInputElement>(null);
  const geoJsonLayer = useRef<LeafletGeoJson | null>(null);
  const { subscription: { data: subData, error: subError } } = useAuditEntries();
  const [deviceEvents, setDeviceEvents] = useState<DeviceEventsWithEntry>({});
  const [deviceTracking, setDeviceTracking] = useState<Map<number, DeviceTracking>>(new Map());
  const mapRef = useRef<HTMLDivElement>(null);
  const { pathname } = useLocation();
  const editMode = pathname.includes("/view/EditMap");
  const maxDeviceEvents = defaultMaxDeviceEvents;
  const resizeComponentPositions = useMemo(() => { return [...positions.values()]; }, [positions]);

  const activateEvent = (events: Event[], id: UUID): void => {
    if (events.length === 0 || !id) {
      return;
    }

    const args: EventClickEventArgs = {
      widgetIndex,
      deviceId: id,
      deviceName: devices.find(cam => cam.id === id)?.name ?? "",
      categories: eventCategories
    };

    widgetEvent?.publish({ event: CommonWidgetEvent.EventClick, args });
  };

  const GeoJsonMarker = new leaflet.Icon({
    iconUrl: mapIcon,
    iconSize: [25, 41],
  });

  const geoJsonPointToLayer = (feature: GeoJSON.Feature<PointFeatureGeometry, PointFeatureProperties>, latLng: leaflet.LatLngExpression) => {
    return leaflet.marker(latLng, {
      icon: GeoJsonMarker
    }).bindTooltip(feature.properties.name || "POI");
  };

  const getUserLocation = async () => {
    return axios.get("http://ip-api.com/json").then(({ data }) => data);
  };

  function onWidgetEvent({ event, args }: WidgetEventArgs): void {
    if (!data) return;
    if (event === PlayerEvent.GeoPosition) {
      const { obj, geo, widgetIndex } = args as PlayerEventArgs;

      if (geo) {
        setDeviceTracking(prevTracking => {
          const newDeviceTracking = new Map(prevTracking);
          const newTrackingPosition: LatLngTuple = [geo[0], geo[1]];
          const deviceTracking = newDeviceTracking.get(widgetIndex);
          let trackingChanged = false;

          if (deviceTracking) {
            // the same device in player
            if (deviceTracking.device.id === obj) {
              const deviceTrackingData = Array.from(deviceTracking.data ?? []);
              const isNewGeoLocation =
                deviceTrackingData.length === 0
                || (
                  deviceTrackingData[deviceTrackingData.length - 1][0] !== newTrackingPosition[0]
                  && deviceTrackingData[deviceTrackingData.length - 1][1] !== newTrackingPosition[1]
                );
              if (isNewGeoLocation) {
                deviceTrackingData.push(newTrackingPosition);
                newDeviceTracking.set(widgetIndex, { ...deviceTracking, data: deviceTrackingData });
                trackingChanged = true;
              }
              else {
                trackingChanged = false;
              }

            }
            // device changed in player
            else {
              const device = findDeviceById(obj);
              if (device) {
                newDeviceTracking.set(widgetIndex, { device, data: [newTrackingPosition] });
                trackingChanged = true;
              }
              else {
                trackingChanged = false;
              }
            }

          }
          else {
            const device = findDeviceById(obj);
            if (device) {
              newDeviceTracking.set(widgetIndex, { device, data: [newTrackingPosition] });
              trackingChanged = true;
            }
            else {
              trackingChanged = false;
            }
          }

          return trackingChanged ? newDeviceTracking : prevTracking;
        });
      }
    }

    if (event === PlayerEvent.Mode) {
      const { obj, widgetIndex, time, mode } = args as PlayerEventArgs;

      setDeviceTracking(prevTracking => {
        const newTracking = new Map(prevTracking);
        let trackingData = newTracking.get(widgetIndex);
        if (!trackingData) {
          const device = findDeviceById(obj);
          if (device) {
            trackingData = { device, data: [] };
          }
          else {
            return prevTracking;
          }
        }

        const prevPlayerState = trackingData.playerState;
        const prevMode = prevPlayerState?.mode;
        const prevTime = prevPlayerState?.lastPlayedTime;

        if (!prevMode && !prevTime && mode === PlayerMode.Archive) {
          newTracking.set(widgetIndex, { ...trackingData, playerState: { mode, lastPlayedTime: time } });
          return newTracking;
        }
        else if (!prevMode && prevTime && mode === PlayerMode.Archive) {
          newTracking.set(widgetIndex, { ...trackingData, playerState: { mode, lastPlayedTime: time } });
          return newTracking;
        }
        else if (mode !== PlayerMode.Stopped && !prevMode) {
          newTracking.set(widgetIndex, { ...trackingData, playerState: { mode, lastPlayedTime: time } });
          return newTracking;
        }
        else if (mode === PlayerMode.Stopped) {
          newTracking.set(widgetIndex, { ...trackingData, playerState: { mode: prevMode, lastPlayedTime: prevMode === PlayerMode.Archive ? time : undefined } });
          return newTracking;
        }
        else if (
          (prevMode && prevMode === PlayerMode.Archive && prevTime !== time && mode === PlayerMode.Archive) ||
          (prevMode && prevMode !== mode && (mode === PlayerMode.Archive || mode === PlayerMode.Live))
        ) {
          newTracking.set(widgetIndex, { ...trackingData, data: [], playerState: { mode, lastPlayedTime: time } });
          return newTracking;
        }

        return prevTracking;
      });
    }
  }

  useEffect(() => {
    if (!systemParamData?.systemParameters) return;
    const { googleMapsApiKey, mapDefaultLayer, geoDefaultPosition } = systemParamData.systemParameters;
    googleMapsApiKey && setAPIKey(googleMapsApiKey);
    setMapDefaultLayer(mapDefaultLayer as keyof typeof MapTypes ?? "OSM");
    if (!geoDefaultPosition) {
      console.warn("GEO_POSITION_DEFAULT is undefined");
      setDefaultPosition(nullPosition);
      return;
    }
    setDefaultPosition([geoDefaultPosition.lat, geoDefaultPosition.lng]);
  }, [systemParamData]);


  useEffect(() => {
    if (data) {
      const filteredDevices: Device[] = data.devicesByAspectTypes.filter(({ position, id: deviceId }) => position !== null && deviceId !== id);
      setDevices(filteredDevices);

      const positions: Map<UUID, LatLngTuple> = filteredDevices.reduce<Map<UUID, LatLngTuple>>((acc, { position, id }) => {
        acc.set(id, [position!.lat, position!.lng]);
        return acc;
      }, new Map());

      if (!defaultPositionProp) {
        setPositions(prevPositions => {
          // comparing prev positions with new positions, needs to avoid resetting ResizeComponent to original state
          if (prevPositions.size !== positions.size) return positions;
          let isNewPositions = false;

          for (const [id, position] of prevPositions) {
            const newPosition = positions.get(id);
            if (!newPosition) {
              isNewPositions = true;
              break;
            }
            if (newPosition[0] !== position[0] || newPosition[1] !== position[1]) {
              isNewPositions = true;
              break;
            }
          }

          return isNewPositions ? positions : prevPositions;
        });
      } else if (defaultPositionProp) {
        const position = defaultPositionProperties?.position || [0, 0];
        const zoom = defaultPositionProperties?.zoom || [0, 0];
        const defaultPositionMap = new Map();
        defaultPositionMap.set(zoom, position);
        setPositions(defaultPositionMap);
      }

      if (id) {
        setDevice(device => data.devicesByAspectTypes.find(({ id: deviceId }) => id === deviceId));
      }
      else if (positions.size === 0) {
        // if in view mode and there is no devices center to default position
        const defaultPositionMap = new Map();
        defaultPositionMap.set("0", defaultPosition);
        setPositions(defaultPositionMap);
      }
    }
  }, [data, id, defaultPosition]);


  useEffect(() => {
    const defaultLatLng = {
      lat: defaultPosition?.[0] ?? nullPosition[0],
      lng: defaultPosition?.[1] ?? nullPosition[1]
    };

    if (!device) {
      setCenter(defaultPosition);
      setCreatePosition(defaultLatLng);
      return;
    }

    if (device.position) {
      setCenter([device.position.lat, device.position.lng]);
      return;
    }

    // set device coords as default position
    setDevice(({
      ...device,
      position: defaultLatLng
    }));

    if (isElectron()) {
      getUserLocation().then((data) => {
        setDevice(({
          ...device,
          position: {
            lat: data.lat,
            lng: data.lon
          }
        }));
      });
    } else {
      // try to get coords from browser
      navigator.geolocation.getCurrentPosition((location) => {
        setDevice(({
          ...device,
          position: {
            lat: location.coords.latitude,
            lng: location.coords.longitude
          }
        }));
      });
    }

  }, [device]);


  useEffect(() => {
    if (!device) {
      return;
    }

    if (position && position.lat && position.lng) {
      setDevice(({
        ...device,
        position: {
          lat: position.lat,
          lng: position.lng
        }
      }));
    }
  }, [position]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }

    const mapControls =
      <Popup
        trigger={
          <Icon
            name="download"
            className="EventList-TitleIcon"
            onClick={() => setExportGeoJson(true)} />}
        content={__("Export GeoJson")}
        position="bottom right"
      />;

    const title =
      <div className="title">
        <div className="HeaderTitle">{cellProps?.title}</div>
        <div className="TitleIcon">
          {mapControls}
        </div>
      </div>;

    setCellProps({ title });

    const id = `${Date.now()}.${Math.random()}`;
    isView && widgetEvent?.subscribe(id, onWidgetEvent);

    return function cleanup() {
      widgetEvent?.unsubscribe(id);
    };
  }, []);

  useEffect(() => {
    if (exportGeoJson) {
      const geojson: PointFeatureCollection = {
        type: "FeatureCollection",
        features: []
      };

      devices.forEach(device => {
        if (!device.position) {
          return;
        }

        geojson.features.push({
          type: "Feature",
          properties: {
            id: device.id,
            name: device.name
          },
          geometry: {
            type: "Point",
            coordinates: [
              device.position.lng, device.position.lat,
            ]
          }
        });
      });

      const geoJsonContent = "data:application/json;charset=UTF-8," + encodeURIComponent(JSON.stringify(geojson));
      const link = document.createElement("a");
      link.setAttribute("href", geoJsonContent);
      link.setAttribute("download", "mapData.json");
      document.body.appendChild(link);
      link.click();

      setExportGeoJson(false);
    }
  }, [exportGeoJson]);

  useEffect(() => {
    subError && console.error("Image map audit subscription error:", subError);

    if (!subData) {
      return;
    }

    const entry = subData.auditEntries;
    if (isError(entry)) {
      console.error("Image map audit subscription error:", entry.error);
      return;
    }

    const { context, triggeredAt, modifiedAt, message, witnesses, category } = entry;

    const event: EventWithEntry = {
      eventId: context,
      from: new Date(triggeredAt).getTime(),
      to: new Date(triggeredAt).getTime(),
      when: new Date(triggeredAt).getTime(),
      lastModified: new Date(modifiedAt).getTime(),
      message,
      objects: witnesses.map(({ id, name }) => ({ objId: id, name })),
      category: parseInt(category.id),
      entry: { ...entry, metadata: [] }
    };

    setDeviceEvents(value => {
      const deviceIds = witnesses
        .filter(w => devices.some(device => device.id === w.id))
        .map(w => w.id);

      if (deviceIds.length === 0) {
        return value;
      }

      const newValue: DeviceEventsWithEntry = clone(value);

      for (const id of deviceIds) {
        let devEvents = newValue[id];
        if (!devEvents) {
          devEvents = { isChecked: false, events: [] };
          newValue[id] = devEvents;
        } else {
          devEvents.isChecked = false;
        }

        const index = devEvents.events.findIndex(ev => ev.eventId === event.eventId);
        if (index < 0) {
          devEvents.events.push(event);
        } else {
          devEvents.events[index] = event;
        }

        devEvents.events.sort((a, b) => b.when - a.when);

        if (devEvents.events.length > maxDeviceEvents) {
          devEvents.events = devEvents.events.slice(0, maxDeviceEvents);
        }
      }

      return newValue;
    });
  }, [subData, subError]);

  useEffect(() => {
    if (geoLayersData && !geoLayersLoading && !geoLayersError && !updateGeoLayersLoading) {
      const layers = !geoLayersData.geoLayers
        ? []
        : geoLayersData.geoLayers.geoLayers.map(layer => ({
          ...JSON.parse(layer.data),
          name: layer.name,
          visible: true,
          id: uuid()
        }));
      setLayersData(layers);
    }
  }, [geoLayersData, geoLayersLoading, geoLayersError]);

  useEffect(() => {
    if (geoJsonLayer.current) {
      const data: GeoJSON.FeatureCollection<GeoJSON.Geometry | null, GeoJSON.GeoJsonProperties> = {
        type: "FeatureCollection",
        features: []
      };

      if (!geoLayersLoading) {
        const changedLayersData = layersData.map(layer => ({
          data: JSON.stringify({
            type: "FeatureCollection",
            features: layer.features
          }),
          name: layer.name,
          mime: "application/geo+json"
        }));
        updateGeoLayers({
          variables: {
            input: {
              geoLayers: [...changedLayersData]
            }
          }
        });
        refetch();
      }

      layersData.forEach(layer => {
        if (layer.visible) {
          data.features = [...data.features, ...layer.features];
        }
      });

      geoJsonLayer.current.clearLayers().addData(data);

      // TODO: why Point only?
      const latLngList: LatLngTuple[] = [];
      for (const feature of data.features) {
        if (feature.geometry?.type === "Point") {
          const [lat, lng] = feature.geometry.coordinates;
          latLngList.push([lat, lng]);
        }
      }

      setLayersLatLngList(latLngList);
    }
  }, [layersData, geoLayersData, geoJsonLayer.current]);

  function findDeviceById(obj: UUID): Device | undefined {
    const device = data?.devicesByAspectTypes.find(dev => dev.id === obj);
    !device && console.error(`Tracking device ${obj} doesn't exist in realm`);
    return device;
  }

  const readImportingGISData = (event: ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) {
      return;
    }

    const reader = new FileReader();
    const name = file.name.split(".")[0];
    const ex = file.name.split(".")[1];

    if (ex === "kml") {
      reader.readAsText(file);

      reader.onload = async function () {
        if (reader.result && typeof reader.result === "string" && name) {
          const kmlDocument = new DOMParser().parseFromString(reader.result, "text/xml");
          const geoJSON = kml(kmlDocument);
          if (geoJSON) {
            const uploadData: ImportLayersDatatype = {
              id: uuid(),
              name,
              visible: true,
              ...geoJSON,
            };
            setLayersData(produce(draft => {
              draft.push(uploadData);
            }));
            setReviewToMarkers(true);
          }
        }
      };
    } else if (ex === "json" || ex === "geojson") {
      reader.readAsText(file);

      reader.onload = async function () {
        if (reader.result && typeof reader.result === "string" && name && ex) {
          try {
            const geoJSON: GeoJSON.FeatureCollection<GeoJSON.Geometry | null, GeoJSON.GeoJsonProperties> = JSON.parse(reader.result);
            if (leaflet.geoJson(geoJSON)) {
              const uploadData: ImportLayersDatatype = ({
                id: uuid(),
                name,
                visible: true,
                ...geoJSON
              });
              setLayersData([...layersData, uploadData]);
              setReviewToMarkers(true);
            }
          } catch (e: any) {
            Log.error(__(e.message));
          }
        }
      };
    }
  };

  const changeLayerPosition = useCallback((gisId: UUID, typeChange: "up" | "down") => {
    setLayersData(produce((draft => {
      const index = draft.findIndex(layer => layer.id === gisId);
      if (typeChange === "up" && index) {
        [draft[index], draft[index - 1]] = [draft[index - 1], draft[index]];
      } else if (typeChange === "down" && index !== draft.length - 1) {
        [draft[index], draft[index + 1]] = [draft[index + 1], draft[index]];
      }
    })));
  }, []);

  { !!updateGeoLayersError && <Message error content={updateGeoLayersError.message} />; }

  const featureCollection: ImportLayersDatatype = {
    type: "FeatureCollection",
    // TODO: correct type
    // @ts-ignore
    features: [...layersData.map(layer => layer.features)]
  };

  return (
    <Segment className={classNames("GeoMap", { "GIS-settings": editMode })}>
      <WithQueryStatus loading={loading || !devices || geoLayersLoading || systemParamLoading} error={error || geoLayersError || systemParamError} noData={!data}>
        <div className="geomap-set">
          {!gisLayersHidden &&
            <div className="geomap-controls">
              <label htmlFor="Import File">
                <Form className="geomap-import-button">
                  <Button onClick={() => uploadGisFile.current?.click()}>Import GIS file</Button>
                  <input id="importFile" ref={uploadGisFile} type="file" hidden onChange={readImportingGISData} />
                </Form>
              </label>
              <div className="geomap-controls-layers">
                <Label className="geomap-control-title">GIS Layers</Label>
                <List className="geomap-layers-list">
                  {layersData && layersData.map(layer => (
                    <ListItem className="geomap-layers-item" key={layer.id}>
                      <span>{layer.name}</span>
                      <div className="layer-arrow">
                        <Icon name="caret up" onClick={() => changeLayerPosition(layer.id, "up")} />
                        <Icon name="caret down" onClick={() => changeLayerPosition(layer.id, "down")} />
                      </div>
                      <Icon name="delete" onClick={() => setLayersData(produce(draft => {
                        return draft.filter(del => layer.id !== del.id);
                      }))} />
                      <Icon name="edit" onClick={() => setEditField(layer)} />
                      <input checked={layer.visible} type='checkbox' onChange={e => {
                        if (showAllMarkers) setShowAllMarkers(false);
                        setLayersData(produce((draft) => {
                          const item = draft.find(i => layer.id === i.id);
                          if (item) {
                            item.visible = !item?.visible;
                          }
                        }));
                      }} />
                    </ListItem>
                  ))}
                  {editField?.id && (
                    <Modal open={!!editField?.id} onClose={() => setEditField(undefined)}>
                      <Header>{__("Rename GIS field")}</Header>
                      <Modal.Content>
                        <Input placeholder={__("Field name")} defaultValue={editField.name} onChange={e => setNewFieldName(e.target.value)} />
                      </Modal.Content>
                      <Modal.Actions>
                        <Button primary onClick={() => {
                          setLayersData(produce(draft => {
                            const layer = draft.find(item => item.id === editField.id);
                            layer ? layer.name = newFieldName : null;
                          }));
                          setEditField(undefined);
                        }}>
                          <Icon name="save" />{__("Save")}
                        </Button>
                        <Button onClick={() => setEditField(undefined)}>
                          <Icon name="cancel" />{__("Cancel")}
                        </Button>
                      </Modal.Actions>
                    </Modal>
                  )}
                  <ListItem className="geomap-layers-item">
                    <span>Cameras</span>
                    <input type='checkbox' defaultChecked onChange={(e) => setShowDevices(e.target.checked)}/>
                  </ListItem>
                  <ListItem className="geomap-layers-item">
                    <span>All</span>
                    <input type='checkbox' checked={showAllMarkers} onChange={(e) => {
                      setShowAllMarkers(e.target.checked);
                      setLayersData(produce(draft => draft.forEach(layer => layer.visible = e.target.checked)));
                    }} />
                  </ListItem>
                </List>
              </div>
            </div>
          }
          <div className={classNames({ "geomap-view-with-controls": editMode }, "geomap-view")}>
            <div ref={mapRef} className="geomap-container">
              <MapContainer scrollWheelZoom attributionControl={false} maxBounds={[[-180, 180], [180, -180]]}>
                <GeoJSON ref={geoJsonLayer} data={featureCollection} pointToLayer={geoJsonPointToLayer}/>
                {geoJsonLayer.current && <ReviewCompoent reviewToMarkers={reviewToMarkers} setReviewToMarkers={setReviewToMarkers} latLngList={layersLatLngList} layerWithBounds={geoJsonLayer}/>}
                {APIKey ? (
                  <LayersControl position="topright">
                    <LayersControl.BaseLayer name="OSM" checked={mapDefaultLayer === "OSM"}>
                      <TileLayer url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png" noWrap={true} minZoom={2}/>
                    </LayersControl.BaseLayer>
                    {google.map((type, index) => (
                      <LayersControl.BaseLayer key={`${type}_${index}`} name={`Google ${type}`} checked={type === mapDefaultLayer}>
                        <ReactLeafletGoogleLayer apiKey={APIKey} type={type} noWrap={true} minZoom={2}/>
                      </LayersControl.BaseLayer>
                    ))}
                  </LayersControl>
                ) : (
                  <TileLayer url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png" noWrap={true} minZoom={2} />
                )}
                {showDevices && (
                <>
                  {devices.map((device, index) => {
                    const event = device.id in deviceEvents && deviceEvents[device.id].events.length > 0 ? deviceEvents[device.id].events[0] : undefined;

                    return device.position &&
                    <DraggableMarker key={index} device={device} mode="view" event={event} onCalloutClick={() => activateEvent(deviceEvents[device.id].events, device.id)} />;
                  })}

                  {deviceTracking.size > 0 &&
                    [...deviceTracking.values()].map((tracking, index) =>
                      <DraggableMarker key={`${tracking.device}_${index}`} device={tracking.device} mode="view"  tracking={tracking.data} />
                    )}

                  <ResizeComponent latLngList={id && center ? [center] : resizeComponentPositions} mapRef={mapRef} defaultPosition={defaultPositionProp} defaultPositionProperties={defaultPositionProperties} setDefaultPositionToStore={setDefaultPositionToStore}/>
                  {id && device && (!position || position && (position.lat === null && position.lng === null || position.lat !== null && position.lng !== null)) &&
                    <DraggableMarker key="currentDevice" device={device} mode="edit" updatePosition={setPosition}/> ||
                    createPosition && !isView &&
                    <DraggableMarker key="newDevice" defaultPosition={createPosition} mode="edit" updatePosition={setPosition}/>
                  }
                </>)}
              </MapContainer>
            </div>
            {editMode &&
              <div className="geomap-gis-button" onClick={() => setGisLayersHidden(value => !value)}>
                <Popup content={gisLayersHidden ? __("Show GIS layers") : __("Hide GIS layers")}
                  trigger={<Icon name={gisLayersHidden ? "cog" : "angle left"} size="big" />} />
              </div>}
          </div>
        </div>
      </WithQueryStatus>
    </Segment >
  );
};

export default GEOMap;
