import React, { useState, useEffect, useRef, useMemo, CSSProperties } from "react";
import { AuditEntryWithSnapshots } from "@core/actions";
import { WidgetId } from "@generated/graphql";
import { getSnapshotURL } from "@solid/libs";
import { PlayerProps } from "@core/types";
import Loading from "components/Loading";
import FontAwesomeButton from "components/FontAwesomeButton";
import CellContent from "components/CellContainer/CellContent";
import { WidgetEvent, WidgetEventArgs, Widgets } from "components/Widgets";
import { PlayerEvent, PlayerEventArgs, PlayerCommand, PlayerCommandArgs, MetadataDrawType } from "@core/playerTypes";
import VideoExportDialog from "components/Timeline/VideoExportDialog";
import { TimelineEvent, TimelineGotoTimeEventArgs } from "components/Timeline";
import { parseJSON } from "utils";
import Metadata from "@solid/player/components/Metadata";
import { MSEMediaPlayerMetadataType } from "@solid/types/player";
import {__} from "@solid/libs/i18n";
import { UUID } from "@solid/types";

import "./style.css";

type EventVisualProps = {
  event: AuditEntryWithSnapshots;
  cameraId?: string;
  widgetEvent?: WidgetEvent;
  widgetIndex?: number;
  isTimeline?: boolean;
};

type Snapshot = {
  image: string;
  markup?: string;
};

const CAMERA_SNAPSHOT = "CAMERA_SNAPSHOT";
const SNAPSHOT_ERROR = "SNAPSHOT_ERROR";

const EventVisual = ({ event, cameraId, widgetEvent, widgetIndex, isTimeline = false }: EventVisualProps) => {
  const [isSnapshot, setIsSnapshot] = useState(true);
  const [snapshots, setSnapshots] = useState<Snapshot[]>([]);
  const [snapshotIndex, setSnapshotIndex] = useState(0);
  const [imgSrc, setImgSrc] = useState("");
  const [imgError, setImgError] = useState(false);
  const [loading, setLoading] = useState(false);
  const [isPlaying, setIsPlaying] = useState(false);
  const [showMarkup, setShowMarkup] = useState(true);
  const playerObj = useMemo(() => getPlayerObj(), [event, cameraId]);
  const imgDivRef = useRef<HTMLDivElement>(null);
  const imgRef = useRef<HTMLImageElement>(null);
  const playerTimeRef = useRef<number | undefined>();
  const contextRef = useRef(event.context);
  const [isExportDialogOpen, setIsExportDialogOpen] = useState(false);
  const exportDevice = useMemo(() => ({ id: cameraId ?? "", name: event.witnesses.find(w => w.id === cameraId)?.name }), [event, cameraId]);
  const [imgSize, setImgSize] = useState({ width: 320, height: 200 });
  const markup = useMemo(() => getMarkup(), [event, snapshots, snapshotIndex]);
  const [markupSize, setMarkupSize] = useState(imgSize);
  const [markupStyle, setMarkupStyle] = useState<CSSProperties | undefined>();

  useEffect(() => {
    if (imgDivRef.current) {
      const { width, height } = imgDivRef.current.getBoundingClientRect();
      setImgSize({ width, height });
    }
  }, []);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    widgetEvent?.subscribe(id, onWidgetEvent);

    return function cleanup() {
      widgetEvent?.unsubscribe(id);
    };
  });

  useEffect(() => {
    contextRef.current = event.context;
  }, [event]);

  useEffect(() => {
    if (!cameraId) {
      setIsSnapshot(true);
    }
  }, [cameraId]);

  useEffect(() => {
    setLoading(false);
    setSnapshotIndex(0);
    const snapshots: Snapshot[] = [];
    for (const snapshot of event.snapshots) {
      const snap: Snapshot = { image: `data:image;base64, ${snapshot.snapshot}`, markup: snapshot.tags.find(tag => tag.key === "schema")?.value };
      if (snapshot.snapshot) {
        if (snapshot.default) {
          snapshots.unshift(snap);
        }
        else {
          snapshots.push(snap);
        }
      }
    }
    if (cameraId) {
      snapshots.push({ image: CAMERA_SNAPSHOT });
    }
    setSnapshots(snapshots);
  }, [event, cameraId]);

  useEffect(() => {
    if (!isSnapshot) {
      return;
    }

    if (snapshotIndex >= snapshots.length) {
      setImgSrc("");
      setImgError(true);
      return;
    }

    const snapshot = snapshots[snapshotIndex];
    if (snapshot.image === SNAPSHOT_ERROR) {
      setImgSrc("");
      setImgError(true);
      return;
    }

    if (snapshot.image !== CAMERA_SNAPSHOT) {
      setImgSrc(snapshot.image);
      setImgError(false);
      return;
    }

    if (!cameraId) {
      setImgSrc("");
      setImgError(true);
      return;
    }

    const eventContext = contextRef.current;
    setLoading(true);
    const img = document.createElement("img");
    img.onload = () => {
      if (contextRef.current !== eventContext) {
        return;
      }
      setLoading(false);
      setImgError(false);
      const canvas = document.createElement("canvas");
      const context = canvas.getContext("2d");
      canvas.width = img.naturalWidth;
      canvas.height = img.naturalHeight;
      context?.drawImage(img, 0, 0);
      const src = canvas.toDataURL();
      setImgSrc(src);
      const newSnapshots = Array.from(snapshots);
      newSnapshots[snapshotIndex] = { image: src };
      setSnapshots(newSnapshots);
    };
    img.onerror = () => {
      if (contextRef.current !== eventContext) {
        return;
      }
      setLoading(false);
      setImgError(true);
      const newSnapshots = Array.from(snapshots);
      newSnapshots[snapshotIndex] = { image: SNAPSHOT_ERROR };
      setSnapshots(newSnapshots);
    };
    img.src = getSnapshotURL(cameraId, Math.trunc(new Date(event.triggeredAt).getTime() / 1000), false);
  }, [snapshots, snapshotIndex, isSnapshot]);

  useEffect(() => {
    if (isSnapshot) {
      setIsPlaying(false);
      playerTimeRef.current = undefined;
    }
  }, [isSnapshot]);

  useEffect(() => {
    const size = getMarkupSize();
    setMarkupSize(size);
    setMarkupStyle({
      left: `${(imgSize.width - size.width) / 2}px`,
      top: `${(imgSize.height - size.height) / 2}px`,
      width: `${size.width}px`,
      height: `${size.height}px`
    });
  }, [imgSize, isSnapshot, imgSrc, imgError]);

  function getPlayerObj(): PlayerProps {
    return {
      "data-id": cameraId ?? "",
      name: event.witnesses.find(w => w.id === cameraId)?.name ?? "",
      time: new Date(event.triggeredAt)
    };
  }

  function onWidgetEvent(args: WidgetEventArgs): void {
    const evArgs = args.args as PlayerEventArgs;
    if (evArgs.widgetIndex !== widgetIndex || evArgs.obj !== cameraId) {
      return;
    }

    switch (args.event) {
      case PlayerEvent.Play:
        setIsPlaying(true);
        break;

      case PlayerEvent.Stop:
      case PlayerEvent.Pause:
      case PlayerEvent.PausedByUser:
        setIsPlaying(false);
        break;

      case PlayerEvent.Frame:
        if (evArgs.time) {
          playerTimeRef.current = evArgs.time;
        }
        break;
    }
  }

  function playerCommand(command: PlayerCommand, args?: Partial<PlayerCommandArgs>): void {
    if (!widgetEvent || !cameraId || widgetIndex === undefined) {
      return;
    }
    const evArgs: PlayerCommandArgs = args ?? {};
    evArgs.objs = [cameraId];
    evArgs.widgetIndices = [widgetIndex];
    if (command === PlayerCommand.Play && !args?.time) {
      evArgs.time = playerTimeRef.current ? new Date(playerTimeRef.current) : new Date(event.triggeredAt);
    }
    widgetEvent.publish({ event: command, args: evArgs });
  }

  function downloadSnapshot(): void {
    if (!imgRef.current || !imgSrc) {
      return;
    }

    const img = imgRef.current;
    const canvas = document.createElement("canvas");
    const context = canvas.getContext("2d");
    canvas.width = img.naturalWidth;
    canvas.height = img.naturalHeight;
    context?.drawImage(img, 0, 0);

    const downloadLink = document.createElement("a");

    let name = event.witnesses.find(w => w.id === cameraId)?.name ?? (event.witnesses.length > 0 ? event.witnesses[0].name : "Image");
    if (name) {
      name += " ";
    }
    const date = new Date(event.triggeredAt);
    const fileName = `${name}${date.getFullYear()}${(date.getMonth() + 1).toString().padStart(2, "0")}` +
      `${date.getDate().toString().padStart(2, "0")}-${date.getHours().toString().padStart(2, "0")}` +
      `${date.getMinutes().toString().padStart(2, "0")}${date.getSeconds().toString().padStart(2, "0")}.jpg`;

    downloadLink.setAttribute("download", fileName);

    canvas.toBlob(blob => {
      if (!blob) {
        return;
      }
      const url = URL.createObjectURL(blob);
      downloadLink.setAttribute("href", url);
      downloadLink.click();
    }, "image/jpeg");
  }

  function fullScreen(): void {
    if (isSnapshot) {
      imgDivRef.current?.requestFullscreen();
    }
    else {
      playerCommand(PlayerCommand.FullScreen);
    }
  }

  function onExportClick(): void {
    setIsExportDialogOpen(true);
  }

  const addDownloadJob = (deviceId: UUID, start: number, end: number) => {
    if (!widgetEvent || !widgetIndex) {
      return;
    }
    const duration = Math.round((end - start) / 1000);
    playerCommand(PlayerCommand.GetAOD, { deviceId, start: Math.round(start), duration });
  };

  function goToTime(): void {
    if (isTimeline) {
      const args: TimelineGotoTimeEventArgs = { time: new Date(event.triggeredAt) };
      widgetEvent?.publish({ event: TimelineEvent.GotoTime, args });
    }
  }

  function switchSnapshot(): void {
    setSnapshotIndex(value => value + 1 < snapshots.length ? value + 1 : 0);
  }

  function getMarkup() {
    if (snapshotIndex >= snapshots.length) {
      return undefined;
    }
    const snapshot = snapshots[snapshotIndex];
    const data = parseJSON(snapshot.markup);
    return data ? data["list"] : undefined;
  }

  function getMarkupSize() {
    const img = imgRef.current;
    if (!img || !img.naturalWidth || !img.naturalHeight || !imgSrc || imgError) {
      return imgSize;
    }
    let ratio: number;
    if (Math.abs(img.naturalWidth - imgSize.width) < Math.abs(img.naturalHeight - imgSize.height)) {
      ratio = imgSize.width / img.naturalWidth;
    }
    else {
      ratio = imgSize.height / img.naturalHeight;
    }
    return { width: img.naturalWidth * ratio, height: img.naturalHeight * ratio };
  }

  const timelineIcon = isTimeline ? Widgets.find(w => w.id === WidgetId.Timeline)?.icon : undefined;

  return (
    <div className="EventVisual">
      <div ref={imgDivRef} className="EventVisual-Image">
        {isSnapshot ?
          <>
            <img ref={imgRef} src={imgSrc} alt=""/>
            {showMarkup && !!markup && !imgError && !!imgSrc &&
            <div className="EventVisual-Markup" style={markupStyle}>
              <Metadata metadataType={MSEMediaPlayerMetadataType.TRIGGERED} list={markup} size={markupSize} time={new Date(event.triggeredAt).getTime()}/>
            </div>}
          </> :
          <CellContent
            obj={playerObj}
            isArchive
            stepBackControl={false}
            metadataType={showMarkup ? MetadataDrawType.Alert : MetadataDrawType.None}
            widgetEvent={widgetEvent}
            index={widgetIndex}/>}
      </div>

      <div className="EventVisual-Buttons">
        {isSnapshot ?
          <FontAwesomeButton icon="play" disabled={!cameraId} hint={__("Play Video")} onClick={() => setIsSnapshot(false)}/> :
          <>
            <FontAwesomeButton icon="undo-alt" hint={__("Replay")}
              onClick={() => playerCommand(PlayerCommand.Play, { time: new Date(event.triggeredAt) })}/>
            <FontAwesomeButton icon={isPlaying ? "pause" : "play"} hint={isPlaying ? __("Pause") : __("Play")}
              onClick={() => isPlaying ? playerCommand(PlayerCommand.Pause) : playerCommand(PlayerCommand.Play)}/>
            <FontAwesomeButton icon="image" hint={__("Show Image")} onClick={() => setIsSnapshot(true)}/>
          </>}

        <div className="EventVisual-Space"/>

        <FontAwesomeButton icon="layer-group" hint={showMarkup ? __("Hide Analytics Markup") : __("Show Analytics Markup")} hintPosition="top right"
          disabled={isSnapshot ? imgError || !imgSrc || !markup : false} onClick={() => setShowMarkup(value => !value)}/>
        {isSnapshot ?
          <>
            <FontAwesomeButton icon="exchange-alt" hint={__("Alternative Image(s)")} hintPosition="top right"
              disabled={snapshots.length <= 1} onClick={switchSnapshot}/>
            <FontAwesomeButton icon="save" hint={__("Save Image")} hintPosition="top right"
              disabled={imgError || !imgSrc} onClick={downloadSnapshot}/>
          </> :
          <FontAwesomeButton icon="download" hint={__("Download Video")} hintPosition="top right" onClick={onExportClick}/>}
        <FontAwesomeButton icon="expand" hint={__("Full Screen")} hintPosition="top right"
          disabled={isSnapshot ? imgError || !imgSrc : false} onClick={fullScreen}/>
        {isTimeline &&
        <FontAwesomeButton svgIcon={timelineIcon} hint={__("Go to Time")} hintPosition="top right" onClick={goToTime}/>}
      </div>

      {loading && <Loading/>}

      <VideoExportDialog
        title={__("Download Video")}
        okButtonText={__("Download")}
        devices={[exportDevice]}
        selectedDevice={exportDevice}
        open={isExportDialogOpen}
        minTime={Math.min(new Date(event.triggeredAt).getTime(), Date.now()) - 24 * 60 * 60 * 1000}
        onGetTime={() => new Date(event.triggeredAt).getTime()}
        onOk={addDownloadJob}
        onClose={() => setIsExportDialogOpen(false)}/>
    </div>
  );
};

export default EventVisual;
