import React, { useState } from "react";
import { List, SemanticICONS } from "semantic-ui-react";

import { __ } from "@solid/libs";

import "./style.css";

export enum EventTypeFilter {
  ALERT = "alert",
  INFO = "info",
  NEEDACKNOWLEDGE = "needAcknowledge"
}

const filterEventsOption =  [
  { label: "Need acknowledge", value: EventTypeFilter.NEEDACKNOWLEDGE, checked: true, iconName: "user circle"},
  { label: "Info", value: EventTypeFilter.INFO, checked: true, iconName: "info circle" },
  { label: "Alert", value: EventTypeFilter.ALERT, checked: true, iconName: "warning sign" },
];

const eventTypes = [EventTypeFilter.NEEDACKNOWLEDGE, EventTypeFilter.INFO, EventTypeFilter.ALERT];

type EventListFilterProps = {
  filter?: EventTypeFilter[];
  onChange?: (value?: EventTypeFilter[]) => void;
};

const EventListFilter = ({ onChange, filter }: EventListFilterProps) => {
  const [values, setValues] = useState<Set<EventTypeFilter>>(new Set(filter ?? eventTypes));

  const onItemClick = (value: EventTypeFilter) => {
    const inSet = values.has(value);
    if (inSet) {
      const newValues = new Set(values);
      newValues.delete(value);
      onChange?.([...newValues]);
      setValues(prevValues => {
        const newSet = new Set(prevValues);
        newSet.delete(value);
        return newSet;
      });
    }
    else {
      onChange?.([...values, value]);
      setValues(prevValues => {
        const newSet = new Set(prevValues);
        newSet.add(value);
        return newSet;
      });
    }
  };

  const onAllClick = () => {
    const newValues = new Set(values);
    if (newValues.size < filterEventsOption.length) {
      onChange?.([...eventTypes]);
    }
    else {
      onChange?.([]);
    }

    setValues(prevValues => {
      let newSet = new Set(prevValues);
      if (newSet.size < filterEventsOption.length) {
        newSet = new Set(eventTypes);
      }
      else {
        newSet = new Set();
      }
      return newSet;
    });
  };

  return (
    <List divided relaxed className="EventListFilter">
      <List.Item className="EventListFilter-Item" onClick={() => onAllClick()}>
        <List.Content>
          <List.Icon name='circle' className="EventListFilter-Icon" verticalAlign='middle' />
          <List.Description>{__("All")}</List.Description>
        </List.Content>
        { values.size === filterEventsOption.length && <List.Icon name='check' className="EventListFilter-Icon" verticalAlign='middle' />}
      </List.Item>
      { filterEventsOption.map((option, index) =>
        <List.Item className="EventListFilter-Item" onClick={() => onItemClick(option.value)} key={`${index}_${option.value}`}>
          <List.Content>
            <List.Icon name={(option.iconName as SemanticICONS)} className="EventListFilter-Icon" verticalAlign='middle' />
            <List.Description>{option.label}</List.Description>
          </List.Content>
          { values.has(option.value) && <List.Icon name='check' className="EventListFilter-Icon" verticalAlign='middle' /> }
        </List.Item>
      )}
    </List>
  );
};

export default EventListFilter;
