import React, { useState, useEffect, useRef, useCallback, useMemo } from "react";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { Message, List, Segment, Form, Button, Ref } from "semantic-ui-react";
import { useReactiveVar } from "@apollo/client";

import {
  useAuditEntriesWithSnapshotsLazyQuery,
  EventInfo,
  DeviceFunctionalAspectType,
  useAcknowledgeEventMutation,
  AuditEntry,
  AuditEntryInput,
  useDeviceListByAspectTypesQuery,
  WidgetId
} from "@generated/graphql";
import { DragObjectType, DragObject, isEvent, PlayerProps } from "@core/types";
import { withStore, WithStoreProps } from "@core/store";
import { formatDateTime } from "@core/utils";
import {MetadataDrawType} from "@core/playerTypes";
import { SolidConfigRoot, currentVersion } from "@core/store/actions";
import { isError, AuditEntryWithSnapshots } from "@core/actions";
import { Utils } from "@solid/libs/utils";
import { parseJSON, queryToInput } from "utils";
import {__} from "@solid/libs/i18n";
import { AuditEntriesAutoUpdateWidgets, getAuditEntryInput, getStoreEvent, isAcknowledged, useAuditEntries } from "@core/actions/auditEntries";

import { EventInfoVisual, WidgetProps } from "components/Widgets";
import WithQueryStatus from "components/WithQueryStatus";
import CellContent from "components/CellContainer/CellContent";
import Loading from "components/Loading";
import EventInfoSnapshots from "./EventInfoSnapshots";
import { currentPlayingEvent } from "components/CellContainer/EventVideo";

import "./style.css";

type EventInfoProps = WidgetProps & WithStoreProps & {
  isCell?: boolean;
  event?: EventInfo;
  visual?: EventInfoVisual;
  metadataType?: MetadataDrawType;
  autoUpdate?: boolean;
  displayAcknowledgeForm?: boolean;
  requireAcknowledgeDetails?: boolean;
  eventAcknowledgment?: boolean;
};

const throttlingTime = 7000;
const infoWrapWidth = 450;

const EventInfoComponent = ({
  isCell,
  viewId,
  index,
  isViewActive,
  store: { workspace: { solidConfigJSON, event: storeEvent, updatedEvent } },
  setStore,
  event: propsEvent,
  visual = EventInfoVisual.Image,
  metadataType,
  autoUpdate = true,
  displayAcknowledgeForm = false,
  requireAcknowledgeDetails = false,
  eventAcknowledgment = false,
}: EventInfoProps) => {
  const { data: devData, error: devError } = useDeviceListByAspectTypesQuery(
    { variables: { types: [
      { type: DeviceFunctionalAspectType.Media },
      { type: DeviceFunctionalAspectType.Sensor }
    ] } });
  const [getAuditEntries, { data, loading, error }] = useAuditEntriesWithSnapshotsLazyQuery({ fetchPolicy: "no-cache" });
  const [acknowledgeEvent, { data: ackUpdData, loading: ackUpdLoading, error: ackUpdError }] = useAcknowledgeEventMutation();

  const [subErrMsg, setSubErrMsg] = useState("");
  const [event, setEvent] = useState(propsEvent ?? storeEvent);
  const [entry, setEntry] = useState<EventInfo | undefined>();
  const [playerProps, setPlayerProps] = useState<PlayerProps | undefined>();
  const [ackDetails, setAckDetails] = useState("");
  const [ackDetailsError, setAckDetailsError] = useState("");

  const updatedEventRef = useRef<EventInfo>();
  const eventHistory = useRef(new Map<string, AuditEntry>());
  const cleanupHistoryIntervalRef = useRef<NodeJS.Timeout | undefined>();
  const infoWrapperRef = useRef<HTMLDivElement>(null);

  const playingEvent = useReactiveVar(currentPlayingEvent);
  const { subscribe, unsubscribe, subscription: { data: subData, loading: subLoading, error: subError } } = useAuditEntries();

  const isEventVideoLive: boolean = useMemo(() => {
    const currentConfig = parseJSON<SolidConfigRoot>(solidConfigJSON)?.[currentVersion];
    if (!currentConfig || !viewId) return false;

    const eventVideoWidget = currentConfig.allViews
      ?.find(view => view.id === viewId)
      ?.widgets?.find(w => w.widgetId === WidgetId.EventVideo);
    if (!eventVideoWidget) return false;

    return !eventVideoWidget.props || eventVideoWidget.props?.["autoUpdate"] === true;
  }, [viewId, solidConfigJSON]);

  const currentLanguage: string = localStorage.getItem("language") ?? "";
  const widgetId = `${viewId}_widget${index}`;
  const autoUpdateOnEvent = useReactiveVar(AuditEntriesAutoUpdateWidgets);
  const subscribeToEvents = autoUpdate && isViewActive && autoUpdateOnEvent;
  const [skipSubscriptionEvents, setSkipSubscriptionEvents] = useState<boolean>(!subscribeToEvents);

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: [DragObjectType.Event],
    drop: (item, monitor) => {
      if (isEvent(item.obj)) {
        setEvent(item.obj);
        unsubscribe(widgetId);
      }
      return undefined;
    },
    collect: monitor => ({
      dragOver: monitor.isOver()
    }),
  });

  useEffect(() => {
    propsEvent && setEvent(propsEvent);
  }, [propsEvent]);

  useEffect(() => {
    if (isCell || autoUpdate) {
      storeEvent = undefined;
      setStore({ workspace: {event: null}});
      setEvent(null);
    }
  }, [isCell]);

  useEffect(() => {
    subscribeToEvents && subscribe(widgetId);
  }, [autoUpdate, isViewActive, autoUpdateOnEvent]);

  const resizeObserverRef = useRef<ResizeObserver>();
  const infoWrapperResizeObserver = useRef<ResizeObserver>();

  const playerId = `${viewId}_widget${index}_player`;

  useEffect(() => {
    const player = document.getElementById(playerId);
    if (player) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === player) {
            const height = entry.contentRect.width * 9 / 16;
            if (entry.contentRect.height !== height) {
              player.style.height = `${height}px`;
            }
            break;
          }
        }
      }));
      resizeObserverRef.current.observe(player);
    }

    cleanupHistoryIntervalRef.current = setInterval(cleanupHistory, 60 * 1000);

    return function cleanup() {
      setStore({ workspace: { event: null } });
      subscribeToEvents && unsubscribe(widgetId);
      if (player) {
        resizeObserverRef.current?.unobserve(player);
      }
      if (cleanupHistoryIntervalRef.current) {
        clearInterval(cleanupHistoryIntervalRef.current);
        cleanupHistoryIntervalRef.current = undefined;
      }
    };
  }, []);

  const getPlayerProps = useCallback((event: EventInfo): PlayerProps | undefined => {
    const witness = event.entry.witnesses.find(w => devData?.devicesByAspectTypes.some(d => d.id === w.id));
    if (!witness) {
      return undefined;
    }
    return { "data-id": witness.id, name: witness.name, time: new Date(event.entry.triggeredAt) };
  }, [devData]);

  const setPlayerPropsRef = useRef(Utils.throttle((event: EventInfo) => setPlayerProps(getPlayerProps(event)), throttlingTime));

  useEffect(() => {
    setPlayerPropsRef.current = Utils.throttle((event: EventInfo) => setPlayerProps(getPlayerProps(event)), throttlingTime);
  }, [getPlayerProps]);

  useEffect(() => {
    if (devData && event) {
      setPlayerPropsRef.current(event);
    }
    devError && console.error("EventInfo getting devices error:", devError);
  }, [devError, devData, event]);

  useEffect(() => {
    if (skipSubscriptionEvents) return;

    if (!subLoading && !subError && subData && !isError(subData.auditEntries)) {
      const entry = { ...subData.auditEntries, metadata: [] };
      if (entry.snapshots.some(snap => snap.default && !!snap.snapshot)) {
        eventHistory.current.set(entry.context, entry);
      }
      let snapshots = entry.snapshots;
      if (!snapshots.some(snap => snap.default && !!snap.snapshot)) {
        const historyEntry = eventHistory.current.get(entry.context);
        if (historyEntry) {
          snapshots = Array.from(historyEntry.snapshots);
        }
      }
      let hasSnapshots = true;
      if (!snapshots.some(snap => snap.default && !!snap.snapshot) &&
          (entry.category.id === "47" || entry.category.id === "61")) {
        hasSnapshots = false;
      }
      if (isEventVideoLive) {
        playingEvent && entry.context === playingEvent.context && setEvent({ entry: { ...entry, snapshots }, hasSnapshots });
      }
      else {
        setEvent({ entry: { ...entry, snapshots }, hasSnapshots });
      }
    }
    if (subData && isError(subData.auditEntries)) {
      console.error("EventInfo subscription error:", subData.auditEntries.error);
      setSubErrMsg(subData.auditEntries.error);
    }
    else if (subError) {
      console.error("EventInfo subscription error:", subError);
      setSubErrMsg(subError.message);
    }
    else {
      setSubErrMsg("");
    }
  }, [subLoading, subError, subData, playingEvent, isEventVideoLive, autoUpdateOnEvent]);

  useEffect(() => {
    if (storeEvent && !eventAcknowledgment) {
      setEvent(storeEvent);
      unsubscribe(widgetId);
    }
  }, [storeEvent]);

  useEffect(() => {
    setSkipSubscriptionEvents(!subscribeToEvents);
  }, [autoUpdateOnEvent]);

  useEffect(() => {
    if (!event) {
      return;
    }

    setEntry(event);

    if (!event.hasSnapshots && visual === EventInfoVisual.Image) {
      getAuditEntries({
        variables: {
          filter: {
            from: event.entry.triggeredAt,
            to: event.entry.triggeredAt,
            witnesses: event.entry.witnesses.map(w => w.id),
            categories: ["47", "61"]
          }
        }});
    }
  }, [event]);

  useEffect(() => {
    if (!loading && !error && data && data.auditEntries) {
      const entry = data.auditEntries.find(entry => event && entry.context === event.entry.context);
      if (entry) {
        setEntry({ entry: { ...entry, metadata: [] }, hasSnapshots: true });
      }
      for (const entry of data.auditEntries) {
        if (entry.snapshots.some(snap => snap.default && !!snap.snapshot)) {
          eventHistory.current.set(entry.context, { ...entry, metadata: [] });
        }
      }
    }
  }, [loading, error, data]);

  useEffect(() => {
    setAckDetails(entry?.entry.note ?? "");
    setAckDetailsError("");

    const infoWrapper = infoWrapperRef.current;
    if (infoWrapper) {
      infoWrapperResizeObserver.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === infoWrapper) {
            if (entry.contentRect.width >= infoWrapWidth) {
              infoWrapper.setAttribute("style", "flex-wrap: no-wrap;");
            }
            else {
              infoWrapper.setAttribute("style", "flex-wrap: wrap;");
            }
            break;
          }
        }
      }));
      infoWrapperResizeObserver.current.observe(infoWrapper);
    }

    return () => {
      infoWrapper && infoWrapperResizeObserver.current?.unobserve(infoWrapper);
    };
  }, [entry]);

  useEffect(() => {
    const updated = updatedEventRef.current;
    if (ackUpdData && entry && updated) {
      if (entry.entry.context === updated.entry.context) {
        setEntry(updated);
      }
      updatedEventRef.current = undefined;
      const updatedEvent = queryToInput<AuditEntryWithSnapshots, AuditEntryInput>(updated.entry);
      setStore({ workspace: { updatedEvent } });
    }
  }, [ackUpdData]);

  useEffect(() => {
    if (ackDetails) {
      setAckDetailsError("");
    }
  }, [ackDetails]);

  useEffect(() => {
    if (updatedEvent && entry && updatedEvent.context === entry.entry.context) {
      setEntry({ entry: updatedEvent, hasSnapshots: entry.hasSnapshots });
    }
  }, [updatedEvent]);

  function acknowledge(): void {
    if (entry) {
      if (requireAcknowledgeDetails && !ackDetails) {
        setAckDetailsError(__("Acknowledge details required"));
        return;
      }

      setAckDetailsError("");

      const updatedEvent = getStoreEvent({ ...entry.entry, note: ackDetails });
      updatedEventRef.current = { ...entry, entry: updatedEvent };

      const entryInput = getAuditEntryInput({...entry.entry, note: ackDetails});
      acknowledgeEvent({ variables: { entry: entryInput }});
    }
  }

  function cleanupHistory(): void {
    const keepTime = 5 * 60 * 1000;
    const keysToDelete: string[] = [];
    for (const [context, entry] of eventHistory.current) {
      if (Date.now() - new Date(entry.triggeredAt).getTime() > keepTime) {
        keysToDelete.push(context);
      }
    }
    for (const context of keysToDelete) {
      eventHistory.current.delete(context);
    }
  }

  const audit = entry?.entry;

  return (
    <Ref innerRef={dropRef}>
      <Segment className={classNames("EventInfo", { "EventInfo_dragOver": dragOver })} >
        { !!devError && <Message visible error header={__("Devices query error")} content={devError.message}/> }
        { !!subErrMsg &&
          <Message visible error header={__("Subscription error")} content={`${subErrMsg} ${__("Resubscribing...")}`}/> }
        { !!ackUpdError && <Message visible error header={__("Acknowledgement error")} content={ackUpdError.message}/> }
        { !event && !isCell && <div className="EventInfo-Empty"><p>{__("Drop Event here")}</p></div> }
        { !!event &&
        <WithQueryStatus error={error}>
            { !audit && <Message visible content={__("Event List entry is not found.")}/> }
            { !!audit &&
            <div className="EventInfo-InfoWrapper" ref={infoWrapperRef} >
              <List>
                <List.Item><List.Header>{__("Id")}</List.Header>{Utils.shortUuid(audit.context, false)}</List.Item>
                <List.Item><List.Header>{__("Triggered At")}</List.Header>{formatDateTime(new Date(audit.triggeredAt), currentLanguage)}</List.Item>
                <List.Item><List.Header>{__("Modified At")}</List.Header>{formatDateTime(new Date(audit.modifiedAt), currentLanguage)}</List.Item>
                <List.Item><List.Header>{__("Message")}</List.Header>{audit.message}</List.Item>
                { displayAcknowledgeForm && (audit.category.id === "47" || audit.category.id === "61") &&
                <List.Item className="EventInfo-WideListItem">
                  <Loading active={ackUpdLoading} text={__("Acknowledging...")}/>
                  <Form onSubmit={(e) => { e.preventDefault(); }}>
                    <Form.Field
                      control={Form.TextArea}
                      className="EventInfo-AcknowledgeDetails"
                      value={ackDetails}
                      placeholder={__("User notes")}
                      rows={3}
                      onInput={(e: React.FormEvent<HTMLInputElement>) => setAckDetails(e.currentTarget.value)}
                      readOnly={isAcknowledged(audit)}
                      disabled={ackUpdLoading}
                      error={ackDetailsError ? { content: ackDetailsError, pointing: "below" } : undefined}
                  />
                    <Button
                      fluid
                      className="EventInfo-AcknowledgeButton"
                      onClick={acknowledge}
                      disabled={isAcknowledged(audit) || ackUpdLoading}
                  >
                      { isAcknowledged(audit) ? __("Acknowledged") : __("Acknowledge") }
                    </Button>
                  </Form>
                </List.Item> }
              </List>
              <List>
                <List.Item><List.Header>{__("Witnesses")}</List.Header>
                </List.Item>
                { visual === EventInfoVisual.Image &&
                <>
                  <EventInfoSnapshots entry={audit} devData={devData?.devicesByAspectTypes} />
                  {loading && <Loading className="EventInfo-ImageLoading" layout="relative vertical"/>}
                </> }
                { visual === EventInfoVisual.Player && !!playerProps &&
                <List.Item className="EventInfo-WideListItem EventInfo-PlayerListItem">
                  <CellContent id={playerId} metadataType={metadataType} obj={playerProps} isViewActive={isViewActive} isArchive/>
                </List.Item> }
              </List>
            </div> }
        </WithQueryStatus> }
      </Segment>
    </Ref>
  );
};

export default withStore(EventInfoComponent);
