import React, { useEffect, useState } from "react";
import { Button, Message } from "semantic-ui-react";
import { useNavigate } from "react-router-dom";
import { __, API, Auth } from "@solid/libs";
import { useAcceptLicenseMutation, useLicenseTextQuery } from "@generated/graphql";
import { useStore } from "@core/store";
import WithQueryStatus from "components/WithQueryStatus";

import "./style.css";

enum EnumRequestState {
  IDLE,
  Success,
  Pending,
  Failed
}

const EulaPage = () => {
  const [eula, setEula] = useState<string>("");
  const [requestState, setRequestState] = useState<EnumRequestState>(EnumRequestState.IDLE);
  const {data, error, loading} = useLicenseTextQuery();
  const [errorMessage, setErrorMessage] = useState("");
  const { setStore,  store: { session: { isLoggedIn } }  } = useStore();

  const navigate = useNavigate();

  const toSelectRealm = () => {
    navigate("/selectrealm");
  };

  const logout = async () => {
    const api = new API();
    Auth.setAPI(api);
    try {
      await Auth.logout();
      await setStore({ session: {isLoggedIn: false} });
      navigate("/");
    } catch (error) {
      console.log(error);
    }
  };

  const [acceptLicense] = useAcceptLicenseMutation({
    onCompleted: toSelectRealm,
    onError: (error) => setErrorMessage(error.message)
  });

  const handleAcknowledge = () => {
    acceptLicense();
  };

  useEffect(() => {
    if (error) {
      setRequestState(
        EnumRequestState.Failed
      );
      return;
    }

    if (loading || !data) {
      setRequestState(
        EnumRequestState.Pending
      );
      return;
    }

    if (data.license?.text) {
      setEula(data.license.text);
      setRequestState(
        EnumRequestState.Success
      );
    } else {
      setRequestState(
        EnumRequestState.Pending
      );
    }
  }, [data?.license, loading, error]);

  const Content = {
    [EnumRequestState.Success]: () => eula ? (
      // Rule disabled because getting data here is from internal storage.
      // eslint-disable-next-line react/no-danger
      <div className="EulaPage-Window-Content" dangerouslySetInnerHTML={{__html: eula}} />
    ) : null,
    [EnumRequestState.IDLE]: () => null,
    [EnumRequestState.Pending]: () => null,
    [EnumRequestState.Failed]: () => (
      <WithQueryStatus error={error}/>
    )
  }[requestState];

  return (
    <div className="Eula-Wrapper">
      {isLoggedIn &&
      <WithQueryStatus loading={loading}>
        <Message visible error content={errorMessage} hidden={!errorMessage} />
        <div className="message-block">
          <Message visible compact className="EulaPage-Window">
            <Content/>
          </Message>
          <div className="Buttons-panel">
            <Button className="Reject-Button" onClick={() => logout()}> {__("REJECT")}</Button>
            <Button primary autoFocus className="Accept-Button" onClick={() => handleAcknowledge()}> {__("AGREE")}</Button>
          </div>
        </div>
      </WithQueryStatus>}
    </div>
  );
};

export default EulaPage;
