import React, { useState, useCallback } from "react";
import {useNavigate, useParams} from "react-router-dom";
import { Button, Form, Header, Icon, List, Message, Segment } from "semantic-ui-react";
import WithQueryStatus from "components/WithQueryStatus";
import { __ } from "@solid/libs/i18n";
import {API} from "@solid/libs/api";
import { Log } from "@solid/libs/log";

import "./style.css";

const api = new API();

type ChangePasswordWithSecretProps = {
  mode: "email" | "password"
};

const ChangePasswordWithSecret = ({mode}: ChangePasswordWithSecretProps) => {
  const { secret = ""} = useParams<{secret: string}>();
  const [password, setPassword] = useState("");
  const [confirmedPassword, setConfirmedPassword] = useState("");
  const [isNotConfirmed, setIsNotConfirmed] = useState(true);
  const navigate = useNavigate();

  const restore = async () => {
    if (password.length >= 0 && password !== confirmedPassword || password.length === 0 && password === confirmedPassword) {
      setIsNotConfirmed(false);
      return;
    }

    if (password.length > 0 && password === confirmedPassword) {
      try {
        setIsNotConfirmed(true);
        if (mode === "email") {
          await api.activateEmail({secret, password});
        } else
        if (mode === "password") {
          await api.restorePassword({secret, password});
        }
        navigate("/");
      } catch (e: any) {
        Log.error(`[${e.code}] ${e.message}`);
      }
    }
  };

  const onPasswordInput = useCallback((e) => {
    setPassword(e.currentTarget.value);
  }, [setPassword]);

  const onConfirmedPasswordInput = useCallback((e) => {
    setConfirmedPassword(e.currentTarget.value);
  }, [setConfirmedPassword]);

  return (
    <Segment className="Email-Confirm">
      <WithQueryStatus>
        <Form onSubmit={(e) => e.preventDefault()}>
          <Form.Field>
            <Header as="h1" className="Email-Confirm-Header">
              {mode === "email"
                ? __("Email confirmation")
                : __("Password recovery")}
            </Header>
            <div className="form-text">
              <List>
                <List.Item>
                  <List.Content>
                    <List.Description>{__("Set your password")}</List.Description>
                  </List.Content>
                </List.Item>
              </List>
            </div>
          </Form.Field>
          <Message visible error content={__("Passwords in 'New password' and 'Confirm new password' fields not equals")} hidden={isNotConfirmed} />
          <Form.Field>
            <div className="ui left icon input">
              <input type="password" id="password" tabIndex={1} defaultValue={password} placeholder={__("New password")}
                onInput={onPasswordInput} autoComplete="new-password"/>
              <Icon name="unlock alternate"/>
            </div>
          </Form.Field>
          <Form.Field>
            <div className="ui left icon input">
              <input type="password" id="confirmPassword" tabIndex={2} defaultValue={confirmedPassword} placeholder={__("Confirm new password")}
                onInput={onConfirmedPasswordInput} autoComplete="new-password"/>
              <Icon name="unlock alternate"/>
            </div>
          </Form.Field>
          <Button
            id="restore"
            tabIndex={3}
            primary
            fluid
            onClick={() => restore()}>
            {__("Confirm")}
          </Button>
        </Form>
      </WithQueryStatus>
    </Segment>
  );
};

export default ChangePasswordWithSecret;
