import React, { useCallback, useEffect, useMemo, useRef, useState } from "react";
import { Label } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { DragObject, isObject } from "@core/types";
import { MediaStreamType, useDeviceForPlayerQuery, WidgetId } from "@generated/graphql";
import { CommonWidgetEvent, CommonWidgetEventArgs, WidgetEventArgs, WidgetProps } from "components/Widgets";
import PlayerCellExternal from "components/PlayerCellExternal";
import { PropType } from "utils";
import { formatDateTime } from "@core/utils";
import { MetadataDrawType, PlayerCommand, PlayerOptions, PlayerProps } from "@core/playerTypes";
import { PlayerMode } from "@solid/types/player_widget";
import WithQueryStatus from "components/WithQueryStatus";
import CameraCellHeaderIcon from "./CameraCellHeaderIcon";
import { __ } from "@solid/libs/i18n";
import VideoExportDialog from "components/Timeline/VideoExportDialog";
import { playerCommand } from "components/Timeline/timelineUtils";

import { UUID } from "@solid/types";

import "../style.css";
import "./style.css";

const defaultArchiveStart = Date.now() - 24 * 60 * 60 * 1000;

type CellContentProps = Pick<DragObject, "obj"> & WidgetProps & PlayerOptions & {
  id?: string;
  onStop?: () => void;
  outOfSync?: boolean;
};

const CellContent = ({
  id,
  obj,
  isViewActive = true,
  widgetId,
  setCellProps,
  metadataType = MetadataDrawType.Alert,
  ptzControls = true,
  digitalZoom = true,
  fullScreenControl = true,
  stepBackControl = true,
  calendar = true,
  allowExport = true,
  userDefinedEvent = true,
  onClipEnd,
  playOnStart,
  onStop,
  widgetEvent,
  outOfSync,
  index: widgetIndex,
  widgets,
  isArchive
}: CellContentProps) => {
  const deviceId = isObject(obj) ? obj["data-id"] : "";
  const [name, setName] = useState(isObject(obj) ? obj.name : "");
  const [header, setHeader] = useState(getHeader(name, obj));
  const [AODDialogOpen, setAODDialogOpen] = useState(false);
  const { data, error, loading } = useDeviceForPlayerQuery({ variables: { id: deviceId }, skip: !deviceId });
  const [archiveStartTime, setArchiveStartTime] = useState(defaultArchiveStart);
  const [mode, setMode] = useState<PlayerMode>(PlayerMode.Stopped);
  const [muted, mute] = useState(true);
  const isMountedRef = useRef(false);
  const isTimeline = !!widgets?.some(w => w.widgetId === WidgetId.Timeline);
  const currentTimeRef = useRef(Date.now());

  const isAudioEnabled = useMemo((): boolean => {
    if (data && data.device?.aspects) {
      return data.device.aspects.some(aspect => aspect.__typename === "DFA_Media" && aspect.enabled && aspect.streamType === MediaStreamType.Audio);
    }
    return false;
  }, [data]);

  useEffect(() => {
    isMountedRef.current = true;
    const id = `${Date.now()}.${Math.random()}`;
    widgetEvent?.subscribe(id, onWidgetEvent);

    return function cleanup() {
      widgetEvent?.unsubscribe(id);
      isMountedRef.current = false;
    };
  });

  useEffect(() => {
    if (!isMountedRef.current) {
      return;
    }

    error && console.error("Could not get camera info:", error);
    if (data && data.device) {
      setName(data.device.name);
    }
  }, [data, error]);

  useEffect(() => {
    if (!isMountedRef.current) {
      return;
    }

    setHeader(getHeader(name, obj));
  }, [name, obj]);

  useEffect(() => {
    if (!isMountedRef.current) {
      return;
    }

    if (setCellProps) {
      setCellProps({
        title: (
          <div className="CellContent-Header">
            <div>
              <CameraCellHeaderIcon mode={mode} />
              <div className="CellContent-Title">{header}</div>
            </div>
            { !!outOfSync &&
            <Label basic color="red" className="CellContent-OutOfSyncLabel">{__("Out of Sync")}</Label> }
            { isAudioEnabled &&
            <FontAwesomeIcon icon={muted ? "volume-mute" : "volume-up"} className="VolumeIcon" onClick={() => muteAudio(!muted)} /> }
          </div>
        ),
        hideIcon: widgetId !== WidgetId.EventVideo
      });
    }
  }, [header, mode, outOfSync, isAudioEnabled, muted]);

  function getTime(obj: PropType<DragObject, "obj">): Date | undefined {
    return isObject(obj) ? (obj.time && !isNaN(obj.time.getTime()) ? obj.time : undefined) : undefined;
  }

  function getHeader(name: string, obj: PropType<DragObject, "obj">): string {
    const time = getTime(obj);
    return isObject(obj) ? name + (time  ? " " + formatDateTime(time) : "") : "";
  }

  const onWidgetEvent = useCallback((args: WidgetEventArgs): void => {
    if (args.event === CommonWidgetEvent.GetName) {
      const eventArgs = args.args as CommonWidgetEventArgs;
      if (eventArgs.widgetIndex === widgetIndex) {
        eventArgs.name = name;
      }
    }
  }, [name]);

  const onGetTime = (): number => {
    return currentTimeRef.current;
  };

  const addDownloadJob = (deviceId: UUID, start: number, end: number) => {
    if (!widgetEvent || typeof widgetIndex !== "number") {
      return;
    }
    const duration = Math.round((end - start) / 1000);
    playerCommand(widgetEvent, [{ id: deviceId, widgetIndex }], PlayerCommand.GetAOD, { deviceId, start: Math.round(start), duration });
  };

  const muteAudio = (muted: boolean) => {
    if (!widgetEvent || typeof widgetIndex !== "number") return;
    mute(muted);
    playerCommand(widgetEvent, [{ id: deviceId, widgetIndex }], PlayerCommand.MuteAudio, { muted });
  };

  let disableLocalTransport = !!data?.device?.cloudDirect;
  if (data?.device?.platform && data.device.platform.aspects.length > 0) {
    const aspect = data.device.platform.aspects[0];
    if (aspect.__typename === "DFA_Avatar" && aspect.webRtcEnabled !== null && aspect.webRtcEnabled !== undefined) {
      disableLocalTransport = !aspect.webRtcEnabled;
    }
  }

  const objTime = getTime(obj);
  const isArchivePlayerMode = widgetId === WidgetId.EventVideo && objTime ?
    Date.now() - objTime.getTime() >= 30000 :
    isArchive || widgetId === WidgetId.ArchiveViewer || widgetId === WidgetId.EventVideo;

  const playerProps: PlayerProps = {
    obj: deviceId,
    header,
    time: objTime,
    isActive: isViewActive,
    hideHeader: true,
    metadataType,
    ptzControls,
    digitalZoom,
    fullScreenControl,
    stepBackControl: stepBackControl && !isTimeline,
    calendar,
    allowExport,
    userDefinedEvent,
    onClipEnd,
    playOnStart: (playOnStart ?? widgetId !== WidgetId.ArchiveViewer) || isTimeline,
    isArchive: isArchivePlayerMode,
    onStop,
    onClose: () => { console.log("Yep"); },
    onChangeMode: mode => setMode(mode),
    onAddDownloadJob: (startTime: number) => {
      currentTimeRef.current = startTime;
      setAODDialogOpen(true);
    },
    onArchiveLengthChange: (hours: number) => {
      if (!isMountedRef.current) return;
      const archiveStartTime = hours > 0 ? Date.now() - hours * 3600 * 1000 : defaultArchiveStart;
      setArchiveStartTime(archiveStartTime);
    },
    widgetEvent,
    widgetIndex,
    showPauseButton: isTimeline,
    disableLocalTransport
  };

  return (
    <>
      { isObject(obj) &&
      <WithQueryStatus loading={loading} error={error}>
        <div id={id} className="CellContainer-Player">
          <PlayerCellExternal {...playerProps} muted={!isAudioEnabled || muted} />
          <VideoExportDialog
            title={__("Add Download Job")}
            okButtonText={__("Add")}
            devices={[{id: deviceId, name }]}
            selectedDevice={{ id: deviceId, name }}
            open={AODDialogOpen}
            minTime={archiveStartTime}
            onGetTime={onGetTime}
            onOk={addDownloadJob}
            onClose={() => setAODDialogOpen(false)}
          />
        </div>
      </WithQueryStatus>}
    </>
  );
};

export default CellContent;
