import React, { useEffect, useRef, useState, useCallback } from "react";
import { Segment, Label } from "semantic-ui-react";
import { ReactSVG } from "react-svg";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { WidgetProps, Widgets } from "components/Widgets";
import { DragObject, isObject, DragObjectType } from "@core/types";
import { useDeviceNameLazyQuery, CreatingView, WidgetInfo, WidgetId } from "@generated/graphql";
import { useSnapshot } from "@core/actions";
import { withStore, WithStoreProps } from "@core/store";
import { parseJSON } from "utils";
import {__} from "@solid/libs/i18n";

import "./style.css";
import "../style.css";

type CameraCellStubProps = WithStoreProps & WidgetProps & {
  object?: DragObject;
};

const CameraCellStub = ({
  viewId,
  index,
  setCellProps,
  widgetId,
  store: { workspace: { creatingViews } },
  setStore,
  ...props
}: CameraCellStubProps) => {
  const [getDeviceName, { data, error }] = useDeviceNameLazyQuery();
  const [deviceId, setDeviceId] = useState("");
  const imageRef = useRef<HTMLImageElement>(null);
  const { imageSrc } = useSnapshot({ load: true, deviceId, imageRef });
  const [object, setObject] = useState<DragObject | undefined>(props.object);
  const widget = Widgets.find(w => w.id === widgetId);

  const findWidget = useCallback((): [CreatingView[] | undefined, WidgetInfo[] | undefined, number | undefined, number | undefined] => {
    if (!creatingViews || index === undefined) {
      return [undefined, undefined, undefined, undefined];
    }

    const views = Array.from(creatingViews);
    const viewIndex = views.findIndex(v => v.viewId + "_selectWidgets" === viewId);
    if (viewIndex < 0 || viewIndex >= views.length) {
      return [undefined, undefined, undefined, undefined];
    }

    const view = views[viewIndex];
    if (!view || !view.widgets || index >= view.widgets.length) {
      return [undefined, undefined, undefined, undefined];
    }

    const widgets = Array.from(view.widgets);
    return [views, widgets, viewIndex, index];
  }, [creatingViews, viewId, index]);

  const setObjectToStore = useCallback((object: DragObject) => {
    const [views, widgets, viewIndex, widgetIndex] = findWidget();
    if (!views || !widgets || viewIndex === undefined || widgetIndex === undefined) {
      return;
    }

    const props = parseJSON(widgets[widgetIndex].propsJSON) ?? {};
    props["object"] = object;
    widgets[widgetIndex] = { ...widgets[widgetIndex], propsJSON: JSON.stringify(props) };
    views[viewIndex] = { ...views[viewIndex], widgets };
    setStore({ workspace: { creatingViews: views } });
  }, [findWidget]);

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: [DragObjectType.Camera],

    drop: useCallback((item, monitor) => {
      setObject(item);
      setObjectToStore(item);
      return undefined;
    }, [setObjectToStore]),

    collect: monitor => ({
      dragOver: monitor.isOver()
    }),
  });

  useEffect(() => {
    setObject(props.object);
  }, [props.object]);

  useEffect(() => {
    if (object && isObject(object.obj)) {
      const deviceId = object.obj["data-id"];
      setDeviceId(deviceId);

      if (object.obj.name) {
        setTitle(object.obj.name);
      } else {
        getDeviceName({ variables: { id: deviceId } });
      }
    } else {
      setDeviceId("");
      setTitle(__("No camera assigned"), true);
    }
  }, [object]);

  useEffect(() => {
    error && console.error("Could not get camera name:", error);
    if (data?.device) {
      setTitle(data.device.name);
    }
  }, [data, error]);

  function setTitle(title: string, noIndicator: boolean = false): void {
    if (!setCellProps || !title) {
      return;
    }

    if (noIndicator) {
      setCellProps({ title, hideIcon: false });
    } else {
      const isLive = widgetId === WidgetId.CameraLive;
      /* eslint-disable react/jsx-indent */
      setCellProps({
        title:
          <>
            <Label color={isLive ? "green" : "red"} className="CellContent-HeaderLabel">{isLive ? __("Live") : __("Archive")}</Label>
            <span>{title}</span>
          </>,
        hideIcon: true
      });
      /* eslint-enable react/jsx-indent */
    }
  }

  useEffect(() => {
    return () => {
      setObject(undefined);
      setDeviceId("");
    };
  }, []);

  return (
    <div ref={dropRef} className={classNames({ "CameraCellStub-Root_dragOver": dragOver })}>
      { !object || !isObject(object.obj) ?
        <Segment className="CameraCellStub-Empty">
          { !!widget && <ReactSVG role="icon" src={widget.icon} /> }
          <p>{widget?.title}</p>
        </Segment> :
        <Segment className="CameraCellStub" key={object.obj["data-id"]}>
          <img ref={imageRef} src={imageSrc} alt={object.obj.name}/>
        </Segment> }
    </div>
  );
};

export default withStore(CameraCellStub);
