import React, { useState, useEffect, useRef, useMemo } from "react";
import { Segment, Form, Button, Icon, Message, Table, Popup, Tab, Modal, Header } from "semantic-ui-react";
import classNames from "classnames";
import produce from "immer";
import {
  LabelsDocument,
  DeviceFunctionalAspectType,
  LabelsQuery,
  RealmObjectType,
  PoliciesQuery,
  PoliciesDocument,
  SetsQuery,
  useSetsQuery,
  useCreateSetMutation,
  useUpdateSetMutation,
  SetsDocument,
  LabelType,
  useDeviceListByAspectTypesQuery,
  BaseDeviceType,
  HealthStatus
} from "generated/graphql";
import { PropType } from "utils";
import type { ArrayElement } from "@solid/types";
import ListText, { ListItem } from "components/Admin/Helpers/ListText";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import IconButton from "components/IconButton";
import CameraList, { ShowHideFilterEvent, SelectionStyle } from "components/CameraList";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import {useApolloClient} from "@apollo/client";
import { getListItemIcon } from "@core/actions";

import "./style.css";
import HelpMD from "./help.md";

type CreateUpdateSetProps = {
  set?: DeviceSet;
  onBack?: () => void;
};

type DeviceSet = PropType<SetsQuery, "sets">[0];

type Device = {
  id: string;
  name: string;
  deviceType?: BaseDeviceType;
  status: boolean;
};

const deviceQueryTypes = [{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }];

const CreateUpdateSet = ({ set, onBack }: CreateUpdateSetProps) => {
  const client = useApolloClient();
  const { data: setsData, error: setsError, loading: setsLoading } = useSetsQuery();
  const { data: devicesData, error: devicesError, loading: devicesLoading } = useDeviceListByAspectTypesQuery({ variables: { types: deviceQueryTypes } });
  const [assignedDevices, setAssignedDevices] = useState<Device[]>([]);
  const assignedDeviceIds = useMemo(() => assignedDevices.map(dev => dev.id), [assignedDevices]);
  const [createSet, { error: createError, loading: createLoading }] = useCreateSetMutation();
  const [updateSet, { error: updateError, loading: updateLoading }] = useUpdateSetMutation();
  const [name, setName] = useState(set?.name ?? "");
  const [nameError, setNameError] = useState("");
  const [availDevicesOpen, setAvailDevicesOpen] = useState(false);
  const [selectedAvailableDevices, setSelectedAvailableDevices] = useState<string[]>([]);
  const showHideFilterEventRef = useRef(new ShowHideFilterEvent());
  const deviceMap = useMemo(() => {
    const map = new Map<string, Device>();
    if (devicesData) {
      for (const device of devicesData.devicesByAspectTypes) {
        map.set(device.id, {
          id: device.id,
          name: device.name,
          deviceType: device.deviceType || undefined,
          status: !device.enabled || device.healthStatus !== HealthStatus.Normal
        });
      }
    }
    return map;
  }, [devicesData]);
  const hiddenDeviceIds = useMemo(() => {
    const idSet = new Set<string>(assignedDeviceIds);
    return Array.from(idSet.keys());
  }, [setsData, assignedDeviceIds]);
  const [activeTab, setActiveTab] = useState(0);
  const [changesDialogOpen, setChangesDialogOpen] = useState(false);
  const backOnUpdateRef = useRef(false);
  const nextTabRef = useRef(0);
  const prevTabRef = useRef(-1);
  const tabCount = 2;

  useEffect(() => {
    setName(set?.name ?? "");
    const assignedDevices = getAssignedDevices(set?.devices ?? []);
    setAssignedDevices(assignedDevices);
  }, [set]);

  useEffect(() => {
    if (updateError && prevTabRef.current >= 0) {
      setActiveTab(prevTabRef.current);
      prevTabRef.current = -1;
    }
  }, [updateError]);

  const cancel = () => {
    onBack && onBack();
  };

  function getNameError(name: string): string {
    let error = "";
    const setId = set?.id;
    if (!name.trim()) {
      error = __("Set name should be not empty");
    } else
    if (setsData?.sets.find(set => set.id !== setId && set.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("Set with the same name already exists");
    }
    return error;
  }

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  async function onCreateUpdate(backOnUpdate = false): Promise<void> {
    if (!set) {
      for (let i = 0; i < tabCount; i++) {
        if (!validateTab(i)) {
          setActiveTab(i);
          return;
        }
      }

      const response = await createSet({
        variables: { set: { name: name.trim(), addDeviceIds: assignedDeviceIds } }
      });

      // update cache
      // update SetsDocument, LabelsDocument
      const setId = response.data?.createSet.id;
      if (setId) {
        // update sets
        const setsCacheData = client.readQuery<SetsQuery>({
          query: SetsDocument
        });

        const newSet: ArrayElement<SetsQuery["sets"]> = {
          __typename: "Set",
          id: setId,
          name: name.trim(),
          isSystemManaged: false,
          devices: assignedDevices.map((device) => {
            return {
              __typename: "ObjectDescriptor",
              id: device.id,
              name: device.name
            };
          }),
        };

        const setsCache = setsCacheData?.sets ?? [];
        const sets = produce(setsCache, (draft) => {
          draft.push(newSet);
        });

        client.writeQuery<SetsQuery>({
          query: SetsDocument,
          data: {
            sets
          }
        });

        // update labels
        const labelsCacheData = client.readQuery<LabelsQuery>({
          query: LabelsDocument
        });

        const newLabel: ArrayElement<LabelsQuery["labels"]> = {
          __typename: "Label",
          id: setId,
          name: name.trim(),
          type: LabelType.Set,
          objects: assignedDevices.map((device) => {
            return {
              __typename: "LabelObject",
              objectId: device.id,
              type: RealmObjectType.Device
            };
          }),
        };

        const labelsCache = labelsCacheData?.labels ?? [];
        const labels = produce(labelsCache, (draft) => {
          draft.push(newLabel);
        });

        client.writeQuery<LabelsQuery>({
          query: LabelsDocument,
          data: {
            labels
          }
        });

        if (response.data?.createSet.warning) {
          Log.error(response.data.createSet.warning);
        }
        onBack && onBack();
      }
    } else {
      if (!validateTab(activeTab)) {
        return;
      }

      backOnUpdateRef.current = backOnUpdate;
      prevTabRef.current = activeTab;

      const setId = set.id;

      let response;
      switch (activeTab) {
        case 0:
          response = await updateSet({
            variables: { id: set.id, set: { name: name.trim() } }
          });

          // update cache
          // update SetsDocument, PoliciesDocument, LabelsDocument
          if (response.data?.updateSet) {
            // update sets
            const setsCacheData = client.readQuery<SetsQuery>({
              query: SetsDocument
            });

            const setsCache = setsCacheData?.sets ?? [];
            const setIndex = setsCache.findIndex((set) => set.id === setId);
            const sets = produce(setsCache, (draft) => {
              draft[setIndex].name = name.trim();
            });

            client.writeQuery<SetsQuery>({
              query: SetsDocument,
              data: {
                sets
              }
            });

            // update policies
            const policiesCacheData = client.readQuery<PoliciesQuery>({
              query: PoliciesDocument
            });

            const policiesCache = policiesCacheData?.policies ?? [];
            if (policiesCache.length > 0) {
              const policies = produce(policiesCache, (draft) => {
                for (const policy of draft) {
                  for (const statement of policy.statements) {
                    for (let i = 0; i < statement.resources.length; i++) {
                      const resource = statement.resources[i];
                      if (resource.id === setId) {
                        resource.name = name.trim();
                        break;
                      }
                    }
                  }
                }
              });

              client.writeQuery<PoliciesQuery>({
                query: PoliciesDocument,
                data: {
                  policies
                }
              });
            }

            // update labels
            const labelsCacheData = client.readQuery<LabelsQuery>({
              query: LabelsDocument
            });

            const labelsCache = labelsCacheData?.labels ?? [];
            if (labelsCache.length > 0) {
              const labelIndex = labelsCache.findIndex((label) => label.type === LabelType.Zone && label.id === setId);
              const labels = produce(labelsCache, (draft) => {
                draft[labelIndex].name = name.trim();
              });

              client.writeQuery<LabelsQuery>({
                query: LabelsDocument,
                data: {
                  labels
                }
              });
            }
          }
          break;
        case 1:
          const addDeviceIds = assignedDeviceIds.filter(id => !set.devices.some(dev => dev.id === id));
          const removeDeviceIds = set.devices.filter(dev => !assignedDeviceIds.includes(dev.id)).map(dev => dev.id);
          response = await updateSet({
            variables: {
              id: set.id,
              set: {
                addDeviceIds,
                removeDeviceIds
              }
            }
          });

          // update cache
          // update SetsDocument, LabelsDocument

          // update sets
          const setsCacheData = client.readQuery<SetsQuery>({
            query: SetsDocument
          });

          const setsCache = setsCacheData?.sets ?? [];
          const setIndex = setsCache.findIndex((set) => set.id === setId);
          const sets = produce(setsCache, (draft) => {
            draft[setIndex].devices = assignedDevices.map((device) => { return { __typename: "ObjectDescriptor", id: device.id, name: device.name }; });
          });

          client.writeQuery<SetsQuery>({
            query: SetsDocument,
            data: {
              sets
            }
          });

          // update labels
          const labelsCacheData = client.readQuery<LabelsQuery>({
            query: LabelsDocument
          });

          const labelsCache = labelsCacheData?.labels ?? [];
          if (labelsCache.length > 0) {
            const labelIndex = labelsCache.findIndex((label) => label.type === LabelType.Set && label.id === setId);
            const labels = produce(labelsCache, (draft) => {
              draft[labelIndex].objects = assignedDevices.map((device) => { return { __typename: "LabelObject", objectId: device.id, type: RealmObjectType.Device }; });
            });

            client.writeQuery<LabelsQuery>({
              query: LabelsDocument,
              data: {
                labels
              }
            });
          }

          break;
      }

      if (response?.data?.updateSet) {
        prevTabRef.current = -1;
        backOnUpdateRef.current && onBack && onBack();
      }
    }
  }

  function onActiveTabChange(index: number | string | undefined): void {
    if (typeof index !== "number") {
      return;
    }

    if (!set) {
      if (index < activeTab || validateTab(activeTab)) {
        setActiveTab(index);
      }
    } else {
      if (!hasChangesOnTab(activeTab)) {
        setActiveTab(index);
        return;
      }

      if (!validateTab(activeTab)) {
        return;
      }

      nextTabRef.current = index;
      setChangesDialogOpen(true);
    }
  }

  function hasChangesOnTab(tabIndex: number): boolean {
    if (!set) {
      switch (tabIndex) {
        case 0: return !!name.trim();
        case 1: return assignedDeviceIds.length > 0;
        default: return false;
      }
    } else {
      switch (tabIndex) {
        case 0: return name.trim() !== set.name;
        case 1: return set.devices.some(dev => !assignedDeviceIds.includes(dev.id)) ||
          assignedDeviceIds.some(id => !set.devices.some(dev => dev.id === id));
        default: return false;
      }
    }
  }

  function validateTab(tabIndex: number): boolean {
    let result = true;
    switch (tabIndex) {
      case 0:
        const nameError = getNameError(name);
        if (nameError) {
          setNameError(nameError);
          result = false;
        }
        break;
    }
    return result;
  }

  function discardTabChanges(tabIndex: number): void {
    switch (tabIndex) {
      case 0:
        setName(set?.name ?? "");
        setNameError("");
        break;
      case 1:
        const assignedDevices = getAssignedDevices(set?.devices ?? []);
        setAssignedDevices(assignedDevices);
        break;
    }
  }

  function addDevices(): void {
    if (selectedAvailableDevices.length === 0) {
      return;
    }

    setAvailDevicesOpen(false);

    setAssignedDevices(value => value.concat(selectedAvailableDevices
      .map(id => deviceMap.get(id))
      .filter(dev => !!dev)
      .map(dev => dev!)));
    setSelectedAvailableDevices([]);
  }

  function removeDevice(device: Device): void {
    setAssignedDevices(value => value.filter(dev => dev.id !== device.id));
  }

  function getAssignedDevices(devices: { id: string, name: string }[]): Device[] {
    const assignedDevices: Device[] = [];
    for (const dev of devices) {
      const device = deviceMap.get(dev.id);
      device && assignedDevices.push(device);
    }
    return assignedDevices;
  }

  function getAssignedDevicesListText(device: Device): ListItem[] {
    if (!device.deviceType) return [{ ...device }];
    const icon = getListItemIcon(device.deviceType, device.status);
    return [{ ...device, faIcon: icon }];
  }

  const error = createError ?? updateError;

  const panes = [
    {
      menuItem: __("Set Properties"),
      render: () => (
        <Tab.Pane>
          <Form className="CreateUpdateSet-Form" onSubmit={e => { e.preventDefault(); }}>
            <Form.Field
              control={Form.Input}
              label={__("Name")}
              placeholder={__("Name")}
              autoFocus
              value={name}
              error={nameError ? { content: nameError, pointing: "below" } : undefined}
              onInput={onNameInput}
            />
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Devices"),
      render: () => (
        <Tab.Pane>
          <div className={classNames("CreateUpdateSet-Devices", { "AvailOpen": availDevicesOpen })}>
            <div className="CreateUpdateSet-AssignedDevices">
              <div className="CreateUpdateSet-TopTableButtons">
                <Button onClick={() => setAvailDevicesOpen(true)}>
                  {__("Assign Devices")}
                </Button>
              </div>
              <div className="CreateUpdateSet-Table">
                <Table celled compact>
                  <Table.Header>
                    <Table.Row>
                      <Table.HeaderCell/>
                      <Table.HeaderCell width={16}>{__("Name")}</Table.HeaderCell>
                    </Table.Row>
                  </Table.Header>

                  <Table.Body>
                    {assignedDevices.map(device =>
                      <Table.Row key={device.id}>
                        <Table.Cell collapsing>
                          <Popup trigger={
                            <Icon name="remove" className="CreateUpdateSet-IconButton" onClick={() => removeDevice(device)}/>
                            }
                            content={__("Remove Device from Set")}
                          />
                        </Table.Cell>
                        <Table.Cell>
                          <ListText items={getAssignedDevicesListText(device)} icons />
                        </Table.Cell>
                      </Table.Row>)}

                    {assignedDevices.length === 0 &&
                    <Table.Row>
                      <Table.Cell colSpan={2} textAlign="center">
                        {__("No devices assigned")}
                      </Table.Cell>
                    </Table.Row>}
                  </Table.Body>
                </Table>
              </div>
            </div>

            {availDevicesOpen &&
            <div className="CreateUpdateSet-AvailableDevices">
              <div className="CreateUpdateSet-CameraListHeader">
                {__("Select devices to assign")}
                <IconButton
                  icon="search"
                  hint={__("Show/Hide Devices Filter")}
                  hintPosition="bottom right"
                  onClick={() => showHideFilterEventRef.current.publish({})}
                />
              </div>
              <CameraList
                deviceTypes={[{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }]}
                selectable
                selectionStyle={SelectionStyle.Checkbox}
                multiselect
                selectedIds={selectedAvailableDevices}
                onSelectedMultiple={selection => setSelectedAvailableDevices(selection)}
                hiddenIds={hiddenDeviceIds}
                allFilters
                allowLabelHierarchyEdit
                openMultipleBranches
                showHideFilterEvent={showHideFilterEventRef.current}
              />
              <div className="CreateUpdateSet-BottomTableButtons">
                <Button positive disabled={selectedAvailableDevices.length === 0} onClick={addDevices}>{__("Select")}</Button>
                <Button onClick={() => setAvailDevicesOpen(false)}>{__("Cancel")}</Button>
              </div>
            </div>}
          </div>
        </Tab.Pane>
      )
    }
  ];

  return (
    <Segment className="CreateUpdateSet">
      <WithQueryStatus error={setsError || devicesError} loading={setsLoading || devicesLoading}>
        {!!error && <Message error content={error.message}/>}

        <div className="CreateUpdateSet-TopButtons">
          <Button onClick={() => cancel()}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
          {!set &&
          <>
            <Button disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab - 1)}>
              <Icon name="arrow alternate circle left"/>{__("Back")}
            </Button>
            {activeTab < tabCount - 1 &&
            <Button positive onClick={() => onActiveTabChange(activeTab + 1)}>
              <Icon name="arrow alternate circle right"/>{__("Next")}
            </Button>}
          </>}
          {(!!set || activeTab >= tabCount - 1) &&
          <Button
            positive
            disabled={!!set && !hasChangesOnTab(activeTab)}
            onClick={() => onCreateUpdate(true)}
          >
            <Icon name={!set ? "plus" : "check"}/>{!set ? __("Create") : __("Save")}
          </Button>}
        </div>

        <Tab
          panes={panes}
          className="CreateUpdateSet-Tab"
          activeIndex={activeTab}
          onTabChange={(e, { activeIndex }) => onActiveTabChange(activeIndex)}
        />

        <Modal open={changesDialogOpen} onClose={() => setChangesDialogOpen(false)}>
          <Header>{__("Unsaved Changes")}</Header>
          <Modal.Content>{__("Set has unsaved changes. Would you like to save the changes?")}</Modal.Content>
          <Modal.Actions>
            <Button positive onClick={() => {
              onCreateUpdate();
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="check"/>{__("Save")}
            </Button>
            <Button negative onClick={() => {
              discardTabChanges(activeTab);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="undo"/>{__("Discard")}
            </Button>
            <Button onClick={() => setChangesDialogOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {(createLoading || updateLoading) && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment>
  );
};

export default CreateUpdateSet;
