import React, { useRef, useState } from "react";
import { Button, Header, Segment, Table } from "semantic-ui-react";

import { __ } from "@solid/libs";

import ButtonsWrapper from "components/Admin/Sensors/ButtonsWrapper";
import CameraList, { SelectionStyle, ShowHideFilterEvent } from "components/CameraList";
import IconButton from "components/IconButton";

import "./style.css";
import { DeviceWitness } from "@core/actions";

type WitnessesToAssignProps = {
  hiddenIds: string[];
  witnesses?: DeviceWitness[];
  getSelectedWitnesses: (witnesses: DeviceWitness[]) => void;
  onCancel: () => void;
};

const WitnessesToAssign = ({ witnesses, hiddenIds, getSelectedWitnesses, onCancel }: WitnessesToAssignProps) => {
  const [selectedWitnesses, setSelectedWitnesses] = useState<DeviceWitness[]>([]);
  const showHideFilterEventRef = useRef(new ShowHideFilterEvent());

  function selectWitnesses(selection: string[]) {
    const selectedWitnesses = witnesses?.filter(witness => selection.includes(witness.id));
    setSelectedWitnesses(selectedWitnesses || []);
  }

  function onSelectClick() {
    getSelectedWitnesses(selectedWitnesses);
    setSelectedWitnesses([]);
  }

  return (
    <Segment className="WitnessesToAssign">
      <ButtonsWrapper>
        <Header as="h5" className="WitnessesToAssign-Header">{__("Select cameras to assign")}</Header>
        { witnesses && witnesses.length > 0 &&
          <IconButton
            icon="search"
            hint={__("Show/Hide Cameras Filter")}
            hintPosition="bottom right"
            onClick={() => showHideFilterEventRef.current.publish({})}
          /> }
      </ButtonsWrapper>

      <Segment className="WitnessesToAssign-Controls">
        { witnesses && witnesses.length > 0 ?
          <CameraList
            selectable
            selectionStyle={SelectionStyle.Checkbox}
            multiselect
            selectedIds={selectedWitnesses.map(witness => witness.id)}
            onSelectedMultiple={selection => selectWitnesses(selection)}
            hiddenIds={hiddenIds}
            allFilters
            allowLabelHierarchyEdit
            openMultipleBranches
            showHideFilterEvent={showHideFilterEventRef.current}
          /> :

          <Table celled compact>
            <Table.Body>
              <Table.Row>
                <Table.Cell colSpan={2} textAlign="center">
                  {__("No cameras to assign")}
                </Table.Cell>
              </Table.Row>
            </Table.Body>
          </Table> }

        <ButtonsWrapper>
          <Button positive disabled={selectedWitnesses.length === 0} onClick={() => onSelectClick()}>{__("Select")}</Button>
          <Button onClick={() => onCancel()}>{__("Cancel")}</Button>
        </ButtonsWrapper>
      </Segment>
    </Segment>
  );
};

export default WitnessesToAssign;
