import React, { useEffect, useRef } from "react";
import { Popup, Icon, Button, Segment, Ref, Table } from "semantic-ui-react";
import {getEmptyImage} from "react-dnd-html5-backend";
import { DropTargetMonitor, useDrag, useDrop } from "react-dnd";
import classNames from "classnames";

import { __ } from "@solid/libs";
import { DeviceWitness, getListItemIcon } from "@core/actions";
import { HealthStatus } from "@generated/graphql";

import ButtonsWrapper from "components/Admin/Sensors/ButtonsWrapper";
import ListText, { ListItem } from "components/Admin/Helpers/ListText";

import "./style.css";

type AssignedWitnessesProp = {
  witnesses?: DeviceWitness[];
  removeWitness: (id: string) => void;
  getWitnessesToAssign: () => void;
  moveWitness: (dragIndex: number, hoverIndex: number) => void;
};

const AssignedWitnesses = React.memo(({ witnesses, removeWitness, moveWitness, getWitnessesToAssign }: AssignedWitnessesProp) => {
  const isDraggingRef = useRef(false);

  return (
    <Segment className="AssignedWitnesses">
      <ButtonsWrapper>
        <Button onClick={() => getWitnessesToAssign()}>{__("Assign Cameras")}</Button>
      </ButtonsWrapper>
      <Table className="AssignedWitnesses-DragTable">
        <Table.Header>
          <Table.Row>
            <Table.HeaderCell width={1}/>
            <Table.HeaderCell width={1}>#</Table.HeaderCell>
            <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
          </Table.Row>
        </Table.Header>
        <Table.Body>
          { witnesses && witnesses.length > 0 ? witnesses.map((device, index) =>
            <WitnessDragItem
              key={device.id}
              device={device}
              index={index}
              moveItem={moveWitness}
              onRemove={removeWitness}
              onDragging={isDragging => { isDraggingRef.current = isDragging; }}
          />) :
          <Table.Row>
            <Table.Cell colSpan={4} textAlign="center">
              {__("No assigned cameras")}
            </Table.Cell>
          </Table.Row> }
        </Table.Body>
      </Table>
    </Segment>
  );
});

export default AssignedWitnesses;

type WitnessDragItemProps = {
  device: DeviceWitness;
  index: number;
  moveItem: (dragIndex: number, hoverIndex: number) => void;
  onDragging: (isDragging: boolean) => void;
  onRemove: (id: string) => void;
};

type DragItem = {
  type: string;
  index: number;
} & DeviceWitness;

const dragItemType = "WitnessDragItem";

const WitnessDragItem = ({ device, index, moveItem, onDragging, onRemove }: WitnessDragItemProps) => {
  const itemRef = useRef<HTMLDivElement>(null);

  const [, drop] = useDrop({
    accept: dragItemType,
    hover(item: DragItem, monitor: DropTargetMonitor) {
      if (!itemRef.current) {
        return;
      }
      const dragIndex = item.index;
      const hoverIndex = index;

      // Don't replace items with themselves
      if (dragIndex === hoverIndex) {
        return;
      }

      // Time to actually perform the action
      moveItem(dragIndex, hoverIndex);

      // Note: we're mutating the monitor item here!
      // Generally it's better to avoid mutations,
      // but it's good here for the sake of performance
      // to avoid expensive index searches.
      item.index = hoverIndex;
    },
  });

  const [{ isDragging }, drag, preview] = useDrag({
    item: { type: dragItemType, index, ...device },
    collect: (monitor: any) => ({
      isDragging: monitor.isDragging(),
    }),
  });

  useEffect(() => {
    preview(getEmptyImage(), { captureDraggingState: true });
  }, []);

  useEffect(() => {
    onDragging(isDragging);
  }, [isDragging]);

  function getListTextItem(dev: DeviceWitness): [ListItem] {
    if (!dev.deviceType) return [{ ...dev }];
    const disabled = !dev.enabled || dev.healthStatus !== HealthStatus.Normal;
    const icon = getListItemIcon(dev.deviceType, disabled);
    return [{ ...dev, faIcon: icon }];
  }

  drag(drop(itemRef));

  return (
    <Ref innerRef={itemRef}>
      <Table.Row className={classNames("WitnessDragItem", { "WitnessDragItem_dragging": isDragging })}>
        <Table.Cell width={1}>
          <Popup
            trigger={<Icon name="remove" onClick={() => onRemove(device.id)}/>}
            content={__("Remove camera from sensor")}
          />
        </Table.Cell>
        <Table.Cell width={1}>{index}</Table.Cell>
        <Table.Cell><ListText items={getListTextItem(device)} icons /></Table.Cell>
      </Table.Row>
    </Ref>
  );
};
