import React, { useEffect, useRef, useState } from "react";
import { Button, Icon, Message, Popup, Segment, Tab } from "semantic-ui-react";
import { useNavigate } from "react-router-dom";
import diff from "microdiff";

import { clone, Log, __ } from "@solid/libs";
import { Device, aspectToAspectInput, useDeviceActions, getEntityWithoutExceptions, SensorShort } from "@core/actions";
import { AuthType, useDeviceLazyQuery, DeviceBaseConfigType, useProbeSensorLazyQuery, DeviceFunctionalAspectType, SensorEvent, DeviceInput, DeviceBaseConfigInput, useCreateSensorMutation, useUpdateDeviceMutation, Dfa_Sensor, ObjectDescriptor, WarningField, useZonesQuery, useDevicesByAspectTypesShortLazyQuery } from "@generated/graphql";
import { useAccessability } from "@core/store/actions/accessability";

import { AutoForm } from "components/AutoForm";
import { WidgetProps } from "components/Widgets";
import ControlSetPanels from "components/Admin/Sets/ControlSetPanels";
import ButtonsWrapper from "../ButtonsWrapper";
import SensorProperties from "./SensorProperties";
import UserEvents from "./UserEvents";
import SensorWitnesses from "./SensorWitnesses";
import SensorPosition from "./SensorPosition";
import SameDevicesModal from "components/Admin/SameDevicesModal";

import "./style.css";

export type DeviceProperties = {
  [Property in keyof Device]?: Device[Property];
};

export interface ComponentWithFormRef {
  getFormRef: () => AutoForm | null
}

type SensorSettingsProps = {
  id?: string;
  sensors?: SensorShort[],
  onCreated?: (id: string) => void;
} & WidgetProps;

const SensorSettings = ({
  id,
  sensors,
  setCellProps,
  onCreated
}: SensorSettingsProps) => {
  const navigate = useNavigate();
  const { config: accessConfig } = useAccessability();
  const [getSensor, { loading: sensorLoading, error: sensorError, data: sensorData }] = useDeviceLazyQuery({ fetchPolicy: "no-cache" });
  const [probeDevice, { data: probeData, error: probeError, loading: probeLoading }] = useProbeSensorLazyQuery({ fetchPolicy: "no-cache" });
  const [createDevice, { /*data: createData, */error: createError, loading: createLoading }] = useCreateSensorMutation();
  const [updateDevice, { /*data: updateData, */error: updateError, loading: updateLoading }] = useUpdateDeviceMutation();
  const [getAvatars, { data: avatarData, error: avatarError, loading: avatarLoading }] = useDevicesByAspectTypesShortLazyQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] } });
  const { data: zoneData, error: zoneError, loading: zoneLoading } = useZonesQuery();

  const { updateCachedDevice } = useDeviceActions({ skipSubscription: true });

  const [initialSensor, setInitialSensor] = useState<DeviceProperties>();
  const [currentSensor, setCurrentSensor] = useState<DeviceProperties>();
  const [activeTab, setActiveTab] = useState<number>(0);
  const [updating, setUpdating] = useState<boolean>(false);
  const [isFirstProbe, setIsFirstProbe] = useState<boolean>(false);
  const [sensorExist, setSensorExist] = useState<boolean>(false);
  const [sameSensorNames, setSameSensorNames] = useState<string[]>([]);

  const propertiesRef = useRef<ComponentWithFormRef>(null);
  const positionRef = useRef<ComponentWithFormRef>(null);

  useEffect(() => {
    id && getSensor({ variables: { id } });
    !id && getAvatars();
  }, [id]);

  useEffect(() => {
    console.log(id);
    if (!setCellProps) return;
    const title: string = !id ? __("Create Sensor") : __("Configure Sensor");
    setCellProps({ title });
  }, [setCellProps, id]);

  useEffect(() => {
    if (sensorData && sensorData.device) {
      const currentDevice = clone(sensorData.device);
      const initialDevice = clone(sensorData.device);
      setCurrentSensor(currentDevice);
      setInitialSensor(initialDevice);
    }
    if (id && sensorData?.device === null) {
      navigate("/view/Sensors");
    }
  }, [sensorData]);

  useEffect(() => {
    if (probeData?.probeSensorDevice) {
      const propertiesFormRef = getFormRef(propertiesRef);
      let aspects;
      if (propertiesFormRef) {
        const propertiesValue = propertiesFormRef.getValues();

        aspects = probeData.probeSensorDevice.aspects.map((aspect) => {
          if (aspect.__typename === "DFA_Sensor" && aspect.type === DeviceFunctionalAspectType.Sensor) {
            aspect.category = propertiesValue["category"];
          }
          return aspect;
        });
      }
      onUpdateProperties({
        config: probeData.probeSensorDevice.config,
        aspects
      });
      setActiveTab(1);
      !id && setIsFirstProbe(true);
    }
  }, [probeData]);

  async function createSensor(device: DeviceInput): Promise<void> {
    setUpdating(true);
    try {
      const { data } = await createDevice({ variables: { device } });
      if (data) {
        const deviceId = data.createSensorDevice.id;
        data.createSensorDevice.warning && Log.error(data.createSensorDevice.warning);
        try {
          await updateCachedDevice(deviceId);
        }
        catch (e) {
          console.error("Updating cache error:", e);
        }
        finally {
          setUpdating(false);
          onCreated?.(deviceId);
        }
      }
      else {
        setUpdating(false);
      }
    }
    catch (e) {
      setUpdating(false);
    }
  }

  async function updateSensor(id: string, input: DeviceInput): Promise<void> {
    setUpdating(true);
    try {
      const { data } = await updateDevice({ variables: { id, device: input } });
      if (data) {
        data.updateDevice.warning && Log.error(data.updateDevice.warning);
        data.updateDevice.warningFields && updateWarningFields(data.updateDevice.warningFields);
        try {
          await updateCachedDevice(id ?? "");
        }
        catch (e) {
          console.error("Updating cache error:", e);
        }
        finally {
          const updatedDevice = clone(currentSensor);
          setInitialSensor(updatedDevice);
          setUpdating(false);
        }
      }
      else {
        setUpdating(false);
      }
    }
    catch (e) {
      setUpdating(false);
    }
  }

  function updateWarningFields(warningFields?: WarningField[]) {
    const propertiesFormRef = getFormRef(propertiesRef);
    if (!warningFields || warningFields.length === 0 || !currentSensor || !propertiesFormRef || !initialSensor) {
      return;
    }

    const zoneUpdated = !warningFields.some(ef => ef.field === "zoneId");
    if (!zoneUpdated) {
      propertiesFormRef.setValue("zoneId", initialSensor.zone?.id);
      onUpdateProperties({ zone: initialSensor.zone });
    }

    const exceptionSetIdField = warningFields.find(ef => ef.field === "setId");
    if (exceptionSetIdField) {
      const exceptionSetIds: string[] = JSON.parse(exceptionSetIdField.value || "");
      const newDeviceSets = getEntityWithoutExceptions<ObjectDescriptor[]>(exceptionSetIds, currentSensor.set || [], initialSensor.set || []);
      onUpdateProperties({ set: newDeviceSets });
    }
  }

  function back() {
    navigate(-1);
  }

  function onActiveTabChange(activeIndex: number) {
    if (sensorLoading) return;
    validateSettings(activeIndex) && setActiveTab(activeIndex);
    isFirstProbe && setIsFirstProbe(false);
  }

  function validateSettings(tabIndex: number): boolean {
    const propertiesFormRef = getFormRef(propertiesRef);
    if (!propertiesFormRef?.validate()) return false;
    if (!id && !probeData?.probeSensorDevice) return false;
    if (tabIndex !== 1 && !isSensorHasAssignedEvents()) return false;
    return true;
  }

  function onUpdateProperties(property: DeviceProperties) {
    if ("category" in property) {
      setCurrentSensor(prevSensor => {
        const newSensor = { ...prevSensor };
        newSensor?.aspects?.map(aspect => {
          if (aspect.__typename === "DFA_Sensor" && aspect.type === DeviceFunctionalAspectType.Sensor) {
            Object.assign(aspect, property);
          }
          return aspect;
        });

        return newSensor;
      });

      return;
    }
    setCurrentSensor(prevSensor => ({...prevSensor, ...property}));
  }

  function onProbe() {
    const propertiesFormRef = getFormRef(propertiesRef);
    if (!propertiesFormRef?.validate()) return;

    const values = propertiesFormRef.getValues();
    probeDevice({
      variables: {
        input: {
          configType: DeviceBaseConfigType.Sensor,
          connect: {
            authType: AuthType.NoAuth,
            host: values["host"],
            port: values["port"],
            user: values["user"],
            pass: values["pass"]
          },
          platformId: values["platformId"],
          sensorModel: values["model"]
        }
      }
    });

  }

  function onCreateUpdate(anyWay: boolean = false) {
    const propertiesFormRef = getFormRef(propertiesRef);
    if (!propertiesFormRef?.getIsValid()) {
      onActiveTabChange(0);
      return;
    }

    if (!anyWay) {
      const sameSensors = getSameSensors();
      if (sameSensors) {
        setSensorExist(true);
        setSameSensorNames(sameSensors.map(sensor => sensor.name));
        return;
      }
    }

    const input = getSensorInput();
    if (!id) {
      createSensor(input);
    }
    else {
      updateSensor(id, input);
    }
  }

  function getSameSensors(): SensorShort[] | undefined {
    const formRef = getFormRef(propertiesRef);
    if (!sensors || !formRef) {
      return undefined;
    }

    const values = formRef.getValues();
    const platformId = values["platformId"];
    if (!sensors.some(dev => dev.platform?.id === platformId)) {
      return undefined;
    }

    const host = values["host"];
    const port = values["port"];
    const sameDevices = sensors.filter(dev =>
      dev.platform?.id === platformId &&
      dev.config.connect.host === host &&
      port === dev.config.connect.port &&
      dev.id !== id);

    if (sameDevices?.length === 0) {
      return undefined;
    }

    return sameDevices;
  }

  function getSensorInput(): DeviceInput {
    const propertiesFormRef = getFormRef(propertiesRef);
    if (!propertiesFormRef || !currentSensor) throw new Error("Invalid form reference");

    const values = propertiesFormRef.getValues();
    const config = getSensorConfigInput();
    const aspects = currentSensor.aspects?.map(aspect => aspectToAspectInput(aspect));
    const { position } = currentSensor;
    const setIds = currentSensor.set?.map(set => set.id);

    const input: DeviceInput = {
      name: values["name"],
      aspects,
      config,
      enabled: values["enabled"] === "on",
      location: values["location"],
      zoneId: values["zoneId"],
      platformId: values["platformId"],
      position: position ? {lat: position.lat, lng: position.lng, alt: position.alt} : null,
      setId: setIds
    };

    return input;
  }

  function getSensorConfigInput(): DeviceBaseConfigInput {
    if (!currentSensor || !currentSensor.config?.configType) {
      return { configType: DeviceBaseConfigType.Sensor, connect: { authType: AuthType.NoAuth } };
    }

    const configType = currentSensor.config.configType;
    // eslint-disable-next-line @typescript-eslint/naming-convention
    const { __typename, ...connectFields } = currentSensor.config.connect;
    const configInput: DeviceBaseConfigInput = {
      configType,
      make: currentSensor.config.make,
      connect: { ...connectFields }
    };

    return configInput;
  }

  function assignDevices(ids: string[]) {
    setCurrentSensor(prevSensor => {
      const newSensor = { ...prevSensor };
      newSensor?.aspects?.map(aspect => {
        if (aspect.__typename === "DFA_Sensor" && aspect.type === DeviceFunctionalAspectType.Sensor) {
          aspect.associatedDevices = ids;
        }
        return aspect;
      });

      return newSensor;
    });
  }

  function getAssignedDevices(): string[] {
    let deviceIds: string[] = [];
    if (currentSensor && currentSensor.aspects) {
      currentSensor.aspects.forEach(aspect => {
        if (aspect.__typename === "DFA_Sensor" && aspect.type === DeviceFunctionalAspectType.Sensor) {
          if (Array.isArray(aspect.associatedDevices)) {
            deviceIds = aspect.associatedDevices || [];
          }
        }
      });
    }

    return deviceIds;
  }

  function assignEvents(events: string[]) {
    setCurrentSensor(prevSensor => {
      const newSensor = { ...prevSensor };
      newSensor?.aspects?.map(aspect => {
        if (aspect.__typename === "DFA_Sensor" && aspect.type === DeviceFunctionalAspectType.Sensor) {
          aspect.userEvents = events;
        }
        return aspect;
      });

      return newSensor;
    });
  }

  function assignSets(sets: ObjectDescriptor[]) {
    onUpdateProperties({ set: sets });
  }

  function getEvents(): { toAssign: SensorEvent[], assigned: string[] } {
    const sensorEvents: { toAssign: SensorEvent[], assigned: string[] } = { toAssign: [], assigned: [] };
    if (currentSensor && currentSensor.aspects) {
      currentSensor.aspects.forEach(aspect => {
        if (aspect.__typename === "DFA_Sensor" && aspect.type === DeviceFunctionalAspectType.Sensor) {
          sensorEvents.assigned = aspect.userEvents || [];
          sensorEvents.toAssign = aspect.sensorEvents || [];
        }
      });
    }

    return sensorEvents;
  }

  function isSensorHasAssignedEvents(): boolean {
    return getEvents().assigned.length > 0;
  }

  function isSensorWitnessesChanged(): boolean {
    if (!id || !currentSensor || !initialSensor) return false;
    const currentSensorWitnesses = (currentSensor.aspects?.find(aspect => aspect.type === DeviceFunctionalAspectType.Sensor) as Dfa_Sensor)?.associatedDevices;
    const initialWitnesses = (initialSensor.aspects?.find(aspect => aspect.type === DeviceFunctionalAspectType.Sensor) as Dfa_Sensor)?.associatedDevices;
    return diff(currentSensorWitnesses || [], initialWitnesses || []).length !== 0;
  }

  function isUserEventsChanged(): boolean {
    if (!id || !currentSensor || !initialSensor) return false;
    const currentUserEvents = (currentSensor.aspects?.find(aspect => aspect.type === DeviceFunctionalAspectType.Sensor) as Dfa_Sensor)?.userEvents;
    const initialUserEvents = (initialSensor.aspects?.find(aspect => aspect.type === DeviceFunctionalAspectType.Sensor) as Dfa_Sensor)?.userEvents;
    return diff(currentUserEvents || [], initialUserEvents || []).length !== 0;
  }

  function isSensorConfigChanged(): boolean {
    if (!currentSensor || !initialSensor) return false;

    const formRef = getFormRef(propertiesRef);
    if (!formRef || !initialSensor || !initialSensor.config) return false;

    const values = formRef.getValues();
    const { config } = initialSensor;

    return config.make !== values["model"] ||
      config.connect.host !== values["host"] ||
      config.connect.port !== values["port"] ||
      config.connect.user !== values["user"] ||
      config.connect.pass !== values["pass"];
  }

  function isSensorPositionChanged(): boolean {
    if (!currentSensor || !initialSensor) return false;

    const formRef = getFormRef(positionRef);
    if (!formRef) return false;

    const values = formRef.getValues();
    const { position } = initialSensor;

    return position?.lat !== values["lat"] || position?.lng !== values["lng"];
  }

  function isSensorSetsChanged(): boolean {
    if (!currentSensor || !initialSensor) return false;

    const setDiff = diff(currentSensor.set || {}, initialSensor.set || {});

    return setDiff.length > 0;
  }

  function resetSameSensors() {
    setSensorExist(false);
    setSameSensorNames([]);
  }

  const assignedDevices = getAssignedDevices();
  const events = getEvents();

  const isPropertiesValid = getFormRef(propertiesRef)?.getIsValid();
  const isPropertiesChanged = getFormRef(propertiesRef)?.getIsChanged();
  const isWitnessesChanged = isSensorWitnessesChanged();
  const isEventsChanged = isUserEventsChanged();
  const isEventsAssigned = isSensorHasAssignedEvents();
  const isConfigChanged = isSensorConfigChanged();
  const isPositionChanged = isSensorPositionChanged();
  const isSetsChanged = isSensorSetsChanged();

  const isSensorChanged = isPropertiesChanged || isWitnessesChanged || isEventsChanged || isPositionChanged || isSetsChanged;
  const fetching = probeLoading || createLoading || updateLoading || updating;
  const loading = sensorLoading || avatarLoading || zoneLoading;

  const panes = [
    {
      menuItem: __("Sensor Properties"),
      pane:
  <Tab.Pane key="sensorProperties">
    { !!probeError && <Message error content={probeError.message} /> }

    <SensorProperties
      ref={propertiesRef}
      sensor={initialSensor}
      zoneData={zoneData}
      avatarData={avatarData}
      error={zoneError || avatarError}
      onChange={onUpdateProperties}
    />
  </Tab.Pane>
    },
    {
      menuItem: __("Events"),
      pane:
  <Tab.Pane key="userEvents">
    <UserEvents
      assignedEvents={events.assigned}
      eventsToAssign={events.toAssign}
      onChange={assignEvents}
      firstProbe={isFirstProbe}
    />
  </Tab.Pane>
    },
    {
      menuItem: __("Witnesses"),
      pane:
  <Tab.Pane key="sensorWitnesses">
    <SensorWitnesses
      assignedIds={assignedDevices}
      onChange={assignDevices}
    />
  </Tab.Pane>
    },
    {
      menuItem: !accessConfig?.limitedGISAccess ? __("GIS") : null,
      pane:
  <Tab.Pane key="sensorPosition">
    <SensorPosition
      ref={positionRef}
      sensor={currentSensor}
      onChange={onUpdateProperties}
      mountMap={activeTab === 3}
    />
  </Tab.Pane>
    },
    {
      menuItem: !accessConfig?.limitedSetsAccess ? __("Sets") : null,
      pane: (
        <Tab.Pane key="sets">
          <ControlSetPanels
            setDeviceSets={assignSets}
            deviceSets={currentSensor?.set ? currentSensor.set as ObjectDescriptor[] : undefined}
          />
        </Tab.Pane>
      )
    }
  ];

  return (
    <Segment loading={fetching} error={sensorError} className="SensorSettings">
      <ButtonsWrapper>
        <Button onClick={() => back()}>
          <Icon name="arrow left"/>
          {__("Back")}
        </Button>

        { activeTab === 0 &&
          <Popup
            trigger={
              <Button className="play-test-btn" positive onClick={() => onProbe()}>
                <Icon name="play"/>{__("Probe")}
              </Button>}
            content={__("Sensor configuration was changed and should be reprobed")}
            position="right center"
            open={!!id && !probeData && isConfigChanged}
          /> }

        { !id && probeData?.probeSensorDevice &&
        <>
          { activeTab > 0 &&
          <Button  disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab - 1)}>
            <Icon name="arrow alternate circle left"/>{__("Back")}
          </Button> }
          { (probeData?.probeSensorDevice && activeTab < panes.length - 1) &&
          <Button disabled={!!probeError || (activeTab === 1 && !isEventsAssigned)} positive onClick={() => onActiveTabChange(activeTab + 1)}>
            <Icon name="arrow alternate circle right"/>{__("Next")}
          </Button> }
        </> }

        { (id || (!id && probeData?.probeSensorDevice && isEventsAssigned)) &&
          <Button
            positive
            className="final-create-btn"
            disabled={(!isPropertiesChanged && !id) || !isSensorChanged || !isPropertiesValid || sensorLoading || !isEventsAssigned}
            onClick={() => onCreateUpdate()}
          >
            <Icon name={!id ? "plus" : "check"}/>{!id ? __("Add") : __("Save")}
          </Button> }
      </ButtonsWrapper>

      { !!createError && <Message error content={createError.message} /> }
      { !!updateError && <Message error content={updateError.message} /> }

      <Segment loading={loading} className="SensorSettings-TabWrapper">
        <Tab
          className="SensorSettings-Tab"
          panes={panes}
          renderActiveOnly={false}
          activeIndex={activeTab}
          onTabChange={(e, { activeIndex }) => typeof activeIndex === "number" && onActiveTabChange(activeIndex)}
        />
      </Segment>

      <SameDevicesModal
        deviceType="sensor"
        names={sameSensorNames}
        open={sensorExist}
        okButtonTittle={id ? __("Save anyway")  : __("Add Anyway")}
        onCancel={resetSameSensors}
        onClose={resetSameSensors}
        onOk={() => {
          onCreateUpdate(true);
          resetSameSensors();
        }}
      />
    </Segment>
  );
};

export default SensorSettings;

function getFormRef(ref: React.RefObject<ComponentWithFormRef>): AutoForm | undefined {
  if (!ref.current) return undefined;
  const formRef: AutoForm | null = ref.current.getFormRef();
  if (formRef) return formRef;
  return undefined;
}
