import React, { useCallback, useEffect, useRef, useState } from "react";
import { Button, Form, Icon, Input, Label, List, Popup } from "semantic-ui-react";
import moment from "moment";
import rome from "@bevacqua/rome";
import { __ } from "@solid/libs/i18n";
import { UUID } from "@solid/types";
import { formatDateTime } from "@core/utils";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";

import "./style.css";

export enum SpanValue {
  FROM = "from",
  TO = "to"
}

type CalendarValues = {
  start: number,
  end: number
};

type ReportPeriodProps = {
  deviceId: UUID,
  start: number,
  end: number,
  plotsLength: number,
  getMetrics: () => void,
  handleChangeStart: (start: number) => void,
  handleChangeEnd: (end: number) => void,
  updateReport: () => void
};

const ReportPeriod = ({
  deviceId,
  start,
  end,
  plotsLength,
  getMetrics,
  handleChangeStart,
  handleChangeEnd,
  updateReport,
}: ReportPeriodProps) => {
  const [valueFromError, setValueFromError] = useState<string>();
  const [valueToError, setValueToError] = useState<string>();
  const [isCalendarFromOpen, setIsCalendarFromOpen] = useState<boolean>(false);
  const [isCalendarToOpen, setIsCalendarToOpen] = useState<boolean>(false);
  const calendarRefFrom = useRef<any>();
  const calendarRefTo = useRef<any>();
  const calendarFromPlaceholderRef = useRef<HTMLDivElement>(null);
  const calendarToPlaceholderRef = useRef<HTMLDivElement>(null);
  const calendarElemRef = useRef<HTMLDivElement>();
  const [calendarValues, setCalendarValues] = useState<CalendarValues>();
  const [isReportUpdated, setIsReportUpdated] = useState<boolean>(true);
  const [isMetricUpdated, setIsMetricUpdated] = useState<boolean>(false);
  const [isDurationChanged, setIsDurationChanged] = useState<boolean>(false);

  useEffect(() => {
    createCalendar(calendarFromPlaceholderRef.current, calendarRefFrom, SpanValue.FROM);
    createCalendar(calendarToPlaceholderRef.current, calendarRefTo, SpanValue.TO);
    return () => destroyCalendar();
  }, []);

  useEffect(() => {
    setValueFromError(isValidDate(start, end));
    setValueToError(isValidDate(start, end));

    if (calendarValues?.end !== end || calendarValues?.start !== start) {
      setIsMetricUpdated(false);
      isReportUpdated && !!plotsLength && setIsReportUpdated(false);
      setIsDurationChanged(true);
    } else {
      setIsDurationChanged(false);
    }

    setCalendarValues({start, end});
  }, [start, end]);

  const createCalendar = useCallback((holderRef: HTMLElement | null, calendarRef: React.MutableRefObject<any>, val: SpanValue) => {
    if (!holderRef) {
      return;
    }

    const elem = document.createElement("div");
    elem.className = "ReportPeriod-Calendar";
    holderRef.appendChild(elem);
    calendarElemRef.current = elem;

    calendarRef.current = rome(elem, {
      max: moment(Date.now())
    });

    if (calendarRef.current) {
      calendarRef.current.hide();

      calendarRef.current.on("data", (timeStr: string) => {
        const value = moment(timeStr);
        const time = moment(value).valueOf();
        if (val === "from") {
          handleChangeStart(time);
        } else {
          handleChangeEnd(time);
        }
      });
    }
  }, [start, end]);

  const showCalendar = (calendarRef: React.MutableRefObject<any>, val: SpanValue) => {
    if (isCalendarFromOpen && val === SpanValue.TO) {
      hideCalendar(calendarRefFrom, SpanValue.FROM);
    }
    if (isCalendarToOpen && val === SpanValue.FROM) {
      hideCalendar(calendarRefTo, SpanValue.TO);
    }

    calendarRef.current?.show();
    if (val === SpanValue.FROM) {
      setIsCalendarFromOpen(true);
    } else {
      setIsCalendarToOpen(true);
    }
  };

  const onUpdateReport = useCallback(() => {
    setIsReportUpdated(true);
    updateReport();
  }, [start, end]);

  const onUpdateMetrics = useCallback(() => {
    setIsMetricUpdated(true);
    getMetrics();
  }, [start, end]);

  function isValidDate(startFrom: number, endTo: number): string | undefined {
    const start = moment(startFrom);
    const end =  moment(endTo);
    if (start.isValid() && end.isValid() && start.valueOf() < end.valueOf()) {
      return undefined;
    }
    return __("Invalid Date");
  }

  function destroyCalendar() {
    hideCalendar(calendarRefFrom, SpanValue.FROM);
    hideCalendar(calendarRefTo, SpanValue.TO);
    if (calendarRefFrom.current) {
      calendarRefFrom.current.destroy();
      calendarRefFrom.current = undefined;
    }
    if (calendarRefTo.current) {
      calendarRefTo.current.destroy();
      calendarRefTo.current = undefined;
    }
    if (calendarElemRef.current) {
      calendarElemRef.current.remove();
      calendarElemRef.current = undefined;
    }
  }

  function hideCalendar(calendarRef: React.MutableRefObject<any>, val: SpanValue) {
    calendarRef.current?.hide();
    if (val === SpanValue.FROM) {
      setIsCalendarFromOpen(false);
    } else {
      setIsCalendarToOpen(false);
    }
  }

  return (
    <div className="ReportPeriod">
      <Form className="ReportPeriod-Form" onSubmit={(e) => { e.preventDefault(); }} key={`RP_${deviceId}`}>
        <Form.Field inline className="ReportPeriod-Start" error={!!valueFromError}>
          <List.Item>{__("Start")}</List.Item>
          <Input value={formatDateTime(new Date(start))} autoFocus
            label={
              <>
                { !isCalendarFromOpen ?
                  <Button className="ReportPeriod-CalendarButton" icon onClick={() => showCalendar(calendarRefFrom, SpanValue.FROM)}>
                    <FontAwesomeIcon icon="calendar-alt"/>
                  </Button> :
                  <Button className="ReportPeriod-CalendarButton" icon onClick={() => hideCalendar(calendarRefFrom, SpanValue.FROM)}>
                    <FontAwesomeIcon icon="times"/>
                  </Button> }
                <div ref={calendarFromPlaceholderRef} className="ReportPeriod-CalendarPlaceholder"/>
              </>}
            labelPosition="right"
            readOnly
            />
          { !!valueFromError && <Label pointing prompt>{valueFromError}</Label> }
        </Form.Field>
        <Form.Field inline className="ReportPeriod-End" error={!!valueToError}>
          <List.Item>{__("End")}</List.Item>
          <Input value={formatDateTime(new Date(end))} autoFocus
            label={
              <>
                { !isCalendarToOpen ?
                  <Button className="ReportPeriod-CalendarButton" icon onClick={() => showCalendar(calendarRefTo, SpanValue.TO)}>
                    <FontAwesomeIcon icon="calendar-alt"/>
                  </Button> :
                  <Button className="ReportPeriod-CalendarButton" icon onClick={() => hideCalendar(calendarRefTo, SpanValue.TO)}>
                    <FontAwesomeIcon icon="times"/>
                  </Button> }
                <div ref={calendarToPlaceholderRef} className="ReportPeriod-CalendarPlaceholder"/>
              </>}
            labelPosition="right"
            readOnly/>
          { !!valueToError && <Label pointing prompt>{valueToError}</Label> }
        </Form.Field>
        {!valueFromError && !valueToError && isDurationChanged && !isReportUpdated && plotsLength !== 0 &&
          <Form.Field className="ReportPeriod-UpdateReport">
            <Popup content={__("Regenerate report")} trigger={
              <Button
                className="ReportPeriod-UpdateReport-Button"
                icon
                disabled={!!valueFromError || !!valueToError || !isDurationChanged || isReportUpdated || plotsLength === 0}
                onClick={() => onUpdateReport()}>
                <Icon name="sync alternate"/>
              </Button>}/>
          </Form.Field>}
      </Form>
      <Form.Field className="ReportPeriod-GetMetricsButton">
        <Button
          positive
          disabled={!!valueFromError || !!valueToError || !isDurationChanged || isMetricUpdated}
          onClick={() => onUpdateMetrics()}>
          <Icon name="file alternate outline"/>
          {__("Get Metrics")}
        </Button>
      </Form.Field>
    </div>
  );
};

export default ReportPeriod;
