import React, { useCallback, useEffect, useRef, useState } from "react";
import { UUID, Type  } from "@solid/types";
import { Utils } from "@solid/libs/utils";
import { AvatarMetricName, CameraMetricName, MetricDimension } from "@generated/graphql";
import ReportPeriod from "./ReportPeriod";
import ReportMetrics from "./ReportMetrics/ReportMetrics";
import PlotWrapper from "./PlotWrapper/PlotWrapper";

export type PlotParams = {
  id: string,
  typeName: string,
  deviceId: UUID,
  name: AvatarMetricName | CameraMetricName,
  dimension: MetricDimension,
  title: string,
  start: number,
  end: number,
  dfaId?: number,
};

type ReportProps = {
  reportType: Type.avatar | Type.camera,
  deviceId: UUID
};

const defaultDuration = 24 * 60 * 60 * 1000;

const Report = ({reportType, deviceId}: ReportProps) => {
  const [start, setStart] = useState<number>(Date.now() - defaultDuration);
  const [end, setEnd] = useState<number>(Date.now());
  const [isMetricsUpdated, setIsMetricUpdated] = useState<boolean>(false);
  const [updateMetrics, setUpdateMetrics] = useState(false);
  const [plotWidth, setPlotWidth] = useState<number>(800);
  const reportsWrapperRef = useRef<any>();
  const resizeObserverRef = useRef<ResizeObserver>();
  const [plots, setPlots] = useState<PlotParams[]>([]);

  useEffect(() => {
    const root = reportsWrapperRef.current;
    if (root) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === root) {
            setPlotWidth(Math.floor(entry.contentRect.width));
            break;
          }
        }
      }));
      resizeObserverRef.current.observe(root);
    }

    return () => {
      root && resizeObserverRef.current?.unobserve(root);
    };
  }, []);

  const createReport = useCallback((value: string) => {
    if (!value) {
      return;
    }

    setUpdateMetrics(false);
    const params = value.split(",");

    const plotParams: PlotParams = {
      id: `${deviceId}_${params[1]}`,
      typeName: params[0],
      deviceId,
      name: params[1] as AvatarMetricName | CameraMetricName,
      dimension: params[3] as MetricDimension,
      title: params[2],
      start,
      end,
      dfaId: params[0] === "CameraMetric" ? Number(params[4]) : undefined,
    };

    const plotIndex = plots.findIndex(plot => plot.name === plotParams.name);

    if (plotIndex < 0) {
      setPlots(values => {
        return [plotParams, ...values];
      });
    }
  }, [plots, reportType, start, end]);

  const updateReport = useCallback(() => {
    setPlots(values => {
      values.forEach(plot => {
        plot.start = start;
        plot.end = end;
      });
      return [...values];
    });
  }, [end, start]);

  const getMetrics = () => {
    setIsMetricUpdated(true);
    setUpdateMetrics(true);
  };

  function onClosePlot(id: string) {
    const newPlots = plots.filter(plot => plot.id !== id);
    setPlots(newPlots);
  }

  function handleChangeStart(start: number) {
    setUpdateMetrics(false);
    setIsMetricUpdated(false);
    setStart(start);
  }

  function handleChangeEnd(end: number) {
    setUpdateMetrics(false);
    setIsMetricUpdated(false);
    setEnd(end);
  }

  return (
    <div className="Report" key={`report_${deviceId}`}>
      <ReportPeriod
        deviceId={deviceId}
        start={start}
        end={end}
        plotsLength={plots.length}
        getMetrics={getMetrics}
        handleChangeStart={handleChangeStart}
        handleChangeEnd={handleChangeEnd}
        updateReport={updateReport}/>
      <ReportMetrics
        metricType={reportType}
        deviceId={deviceId}
        isMetricsUpdated={isMetricsUpdated}
        updateMetrics={updateMetrics}
        start={start}
        end={end}
        createReport={createReport}/>
      <div className="Report-Wrapper" ref={reportsWrapperRef}>
        {plots.map(plot =>
          <PlotWrapper
            key={`pw_${plot.id}`}
            plotType={reportType}
            plotId={plot.id}
            plotTitle={plot.title}
            plotDimension={plot.dimension}
            plotWidth={plotWidth}
            metricName={plot.name}
            deviceId={deviceId}
            from={plot.start}
            to={plot.end}
            onClosePlot={onClosePlot}/>)}
      </div>
    </div>
  );
};

export default Report;
