import React, { useEffect, useState } from "react";
import { Button, Segment } from "semantic-ui-react";
import { AvatarMetricName, AvatarReportQueryVariables, CameraMetricName, CameraReportQueryVariables, MetricDimension, useAvatarReportLazyQuery, useCameraReportLazyQuery } from "@generated/graphql";
import Plot from "components/Plot";
import { UUID, Type } from "@solid/types";

import "./style.css";

type PlotWrapperProps = {
  plotType: Type.avatar | Type.camera,
  plotId: UUID,
  plotTitle: string,
  plotDimension: MetricDimension | undefined,
  plotWidth: number,
  metricName: AvatarMetricName | CameraMetricName,
  deviceId: UUID,
  from: number,
  to: number,
  dfaId?: number,
  onClosePlot: (id: string) => void,
};

const PlotWrapper = ({
  plotType,
  plotId,
  plotTitle,
  plotDimension,
  plotWidth,
  metricName,
  deviceId,
  from,
  to,
  dfaId,
  onClosePlot,
}: PlotWrapperProps) => {
  const [getAvatarReport, {data: avatarData, loading: avatarLoading, error: avatarError}] = useAvatarReportLazyQuery();
  const [getCameraReport, {data: cameraData, loading: cameraLoading, error: cameraError}] = useCameraReportLazyQuery();
  const [firstLoading, setFirstLoading] = useState<boolean>(true);
  const [plotData, setPlotData] = useState<[xValues:number[], ...yValues:(number | null | undefined | string)[][]]>();

  useEffect(() => {
    if (plotType === Type.avatar) {
      const input: AvatarReportQueryVariables = {
        id: deviceId,
        from: new Date(from),
        to: new Date(to),
        metricName: metricName as AvatarMetricName
      };
      getAvatarReport({ variables: input });
    } else {
      const input: CameraReportQueryVariables = {
        id: deviceId,
        from: new Date(from),
        to: new Date(to),
        metricName: metricName as CameraMetricName,
        dfaId: dfaId || 1,
      };
      getCameraReport({variables: input});
    }
  }, [from, to, plotType]);

  useEffect(() => {
    if (!avatarLoading && !avatarError && avatarData) {
      const xValues: number[] = avatarData.avatarReport.map(metric => metric.time);
      const yValues: Array<number | null> = avatarData.avatarReport.map(metric => metric.value);

      setPlotData([xValues, yValues]);
      setFirstLoading(false);
    }
  }, [avatarData, avatarLoading, avatarError]);

  useEffect(() => {
    if (!cameraLoading && !cameraError && cameraData) {
      if (plotDimension !== MetricDimension.Status) {
        const xValues: number[] = cameraData.numeric.map(metric => {
          if (metric.__typename === "NumericReport") {
            return metric.time;
          }
          return 0;
        });
        const yValues: Array<number | null | undefined> = cameraData.numeric.map(metric => {
          if (metric.__typename === "NumericReport") {
            return metric.value;
          }
          return null;
        });
        setPlotData([xValues, yValues]);
        setFirstLoading(false);
      } else {
        const xValues: number[] = cameraData.status.map(metric => {
          if (metric.__typename === "StatusReport") {
            return metric.time;
          }
          return 0;
        });
        const yValues: Array<string | null | undefined> = cameraData.status.map(metric => {
          if (metric.__typename === "StatusReport") {
            return metric.value;
          }
          return null;
        });
        setPlotData([xValues, yValues]);
        setFirstLoading(false);
      }
    }
  }, [cameraData, cameraError, cameraLoading]);

  return (
    <Segment loading={firstLoading} className="Reports-FirstLoading-Stub" key={`rfls_${plotId}`}>
      {plotData &&
      <Segment loading={avatarLoading || cameraLoading} className="Reports-Plot" key={`rp_${plotId}`}>
        <Plot
          key={`plt_${plotId}`}
          id={plotId}
          title={plotTitle}
          data={plotData}
          widthRef={plotWidth}
          dimension={plotDimension}
          plotType={plotType}/>
        <Button icon="close" className="Reports-Plot-CloseButton"
          onClick={() => onClosePlot(plotId)}/>
      </Segment>}
    </Segment>
  );
};

export default PlotWrapper;
