import React, { useState, useEffect } from "react";
import { Table, Button, Icon, Checkbox, Popup } from "semantic-ui-react";
import ListText from "components/Admin/Helpers/ListText";
import {__} from "@solid/libs/i18n";

import "./style.css";

export type ObjectItem = {
  id: string;
  name: string;
  children?: ObjectItem[];
};

type AssignObjectsProps = {
  items?: ObjectItem[];
  assignedIdSet: Set<string>;
  hasChildren?: boolean;
  readOnly?: boolean;
  itemSingleText: string;
  itemPluralText: string;
  parentSingleText: string;
  childPluralText?: string;
  selectPrompt?: string;
  removePrompt?: string;
  maxChildren?: number;
  onAssignedIdSetChange: (idSet: Set<string>) => void;
};

const AssignObjects = ({
  items,
  assignedIdSet,
  hasChildren = true,
  readOnly = false,
  itemSingleText,
  itemPluralText,
  parentSingleText,
  childPluralText,
  selectPrompt,
  removePrompt,
  maxChildren = 3,
  onAssignedIdSetChange
}: AssignObjectsProps) =>
{
  const [assignedItems, setAssignedItems] = useState<ObjectItem[]>([]);
  const [availableItems, setAvailableItems] = useState<ObjectItem[]>([]);
  const [selectedIdSet, setSelectedIdSet] = useState(new Set<string>());
  const [availOpen, setAvailOpen] = useState(false);

  useEffect(() => {
    if (items) {
      setAssignedItems(items
        .filter(item => assignedIdSet.has(item.id))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })));
      setAvailableItems(items
        .filter(item => !assignedIdSet.has(item.id))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })));
    }
  }, [items, assignedIdSet]);

  function selectUnselect(item: ObjectItem, select: boolean): void {
    setSelectedIdSet(value => {
      const newValue = new Set<string>(value);
      if (select) {
        newValue.add(item.id);
      }
      else {
        newValue.delete(item.id);
      }
      return newValue;
    });
  }

  function assignObjects(): void {
    if (selectedIdSet.size === 0) {
      return;
    }

    setAvailOpen(false);

    const ids = Array.from(selectedIdSet.keys());
    const idSet = new Set<string>(assignedIdSet);
    for (const id of ids) {
      idSet.add(id);
    }
    onAssignedIdSetChange(idSet);
    setSelectedIdSet(new Set<string>());
  }

  function removeObject(item: ObjectItem): void {
    const idSet = new Set<string>(assignedIdSet);
    idSet.delete(item.id);
    onAssignedIdSetChange(idSet);
  }

  return (
    <div className="AssignObjects">
      <div className="AssignObjects-Assigned" style={availOpen ? { maxHeight: "50%" } : undefined}>
        {!readOnly &&
        <div className="AssignObjects-TopTableButtons">
          <Button onClick={() => setAvailOpen(true)}>
            {__("Assign {{item}}", {item: itemPluralText})}
          </Button>
        </div>}
        <div className="AssignObjects-Table">
          <Table celled compact>
            <Table.Header>
              <Table.Row>
                <Table.HeaderCell className="AssignObjects-ActionCell"/>
                <Table.HeaderCell width={hasChildren ? 6 : 16}>{__("Name")}</Table.HeaderCell>
                {hasChildren &&
                <Table.HeaderCell width={10}>{childPluralText}</Table.HeaderCell>}
              </Table.Row>
            </Table.Header>

            <Table.Body>
              {assignedItems.map(item =>
                <Table.Row key={item.id}>
                  <Table.Cell collapsing>
                    {!readOnly &&
                    <Popup trigger={
                      <Icon name="remove" className="AssignObjects-IconButton" onClick={() => removeObject(item)}/>
                      }
                      content={removePrompt ?? __("Remove {{item}} from the {{list}}", {item: itemSingleText, list: parentSingleText})}
                    />}
                  </Table.Cell>
                  <Table.Cell>
                    {item.name}
                  </Table.Cell>
                  {hasChildren &&
                  <Table.Cell>
                    {!!item.children &&
                    <ListText items={item.children} maxItems={maxChildren} modalHeader={childPluralText}/>}
                  </Table.Cell>}
                </Table.Row>)}

              {assignedItems.length === 0 &&
              <Table.Row>
                <Table.Cell colSpan={hasChildren ? 3 : 2} textAlign="center">
                  {__("No {{item}} assigned", {item: itemPluralText.toLocaleLowerCase()})}
                </Table.Cell>
              </Table.Row>}
            </Table.Body>
          </Table>
        </div>
      </div>

      {availOpen &&
      <div className="AssignObjects-Available">
        <div className="AssignObjects-SelectPrompt">
          {selectPrompt ?? __("Select {{item}} to assign to the {{list}}", {item: itemPluralText.toLocaleLowerCase(), list: parentSingleText.toLocaleLowerCase()})}
        </div>
        <div className="AssignObjects-Table">
          <Table celled compact>
            <Table.Header>
              <Table.Row>
                <Table.HeaderCell className="AssignObjects-ActionCell"/>
                <Table.HeaderCell width={hasChildren ? 6 : 16}>{__("Name")}</Table.HeaderCell>
                {hasChildren &&
                <Table.HeaderCell width={10}>{childPluralText}</Table.HeaderCell>}
              </Table.Row>
            </Table.Header>

            <Table.Body>
              {availableItems.map(item =>
                <Table.Row key={item.id}>
                  <Table.Cell collapsing>
                    <Checkbox checked={selectedIdSet.has(item.id)} onChange={(e, data) => selectUnselect(item, !!data.checked)}/>
                  </Table.Cell>
                  <Table.Cell>
                    {item.name}
                  </Table.Cell>
                  {hasChildren &&
                  <Table.Cell>
                    {!!item.children &&
                    <ListText items={item.children} maxItems={maxChildren} modalHeader={childPluralText}/>}
                  </Table.Cell>}
                </Table.Row>)}

              {availableItems.length === 0 &&
              <Table.Row>
                <Table.Cell colSpan={hasChildren ? 3 : 2} textAlign="center">
                  {__("No available {{item}} found", {item: itemPluralText.toLocaleLowerCase()})}
                </Table.Cell>
              </Table.Row>}
            </Table.Body>
          </Table>
        </div>
        <div className="AssignObjects-BottomTableButtons">
          <Button positive disabled={selectedIdSet.size === 0} onClick={assignObjects}>{__("Select")}</Button>
          <Button onClick={() => setAvailOpen(false)}>{__("Cancel")}</Button>
        </div>
      </div>}
    </div>
  );
};

export default AssignObjects;
