import React, { useState, useEffect, useRef } from "react";
import { Segment, Form, Button, Icon, Message, Tab, Modal, Header } from "semantic-ui-react";
import {
  GroupsQuery,
  useGroupsQuery,
  useUsersQuery,
  usePoliciesQuery,
  useCreateGroupMutation,
  useUpdateGroupMutation,
  GroupsDocument,
  UsersDocument,
  UsersQuery
} from "generated/graphql";
import { PropType } from "utils";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import AssignObjects from "components/Admin/Helpers/AssignObjects";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import HelpMD from "./help.md";
import {useApolloClient} from "@apollo/client";

import "./style.css";
import produce from "immer";
import type {ArrayElement} from "@solid/types";

type CreateUpdateGroupProps = {
  group?: Group;
  onBack?: () => void;
};

type Group = PropType<GroupsQuery, "groups">[0];

const CreateUpdateGroup = ({ group, onBack }: CreateUpdateGroupProps) => {
  const client = useApolloClient();
  const { data: groupsData, error: groupsError, loading: groupsLoading } = useGroupsQuery();
  const { data: usersData, error: usersError, loading: usersLoading } = useUsersQuery();
  const { data: policiesData, error: policiesError, loading: policiesLoading } = usePoliciesQuery();
  const [userIdSet, setUserIdSet] = useState(new Set<string>());
  const [policyIdSet, setPolicyIdSet] = useState(new Set<string>());
  const [createGroup, { error: createError, loading: createLoading }] = useCreateGroupMutation();
  const [updateGroup, { data: updateData, error: updateError, loading: updateLoading }] = useUpdateGroupMutation();
  const [name, setName] = useState(group?.name ?? "");
  const [nameError, setNameError] = useState("");
  const [activeTab, setActiveTab] = useState(0);
  const [changesDialogOpen, setChangesDialogOpen] = useState(false);
  const backOnUpdateRef = useRef(false);
  const nextTabRef = useRef(0);
  const prevTabRef = useRef(-1);
  const tabCount = 3;

  useEffect(() => {
    setName(group?.name ?? "");
    setUserIdSet(new Set<string>(group?.users.map(user => user.id) ?? []));
    setPolicyIdSet(new Set<string>(group?.policies.map(policy => policy.id) ?? []));
  }, [group]);

  useEffect(() => {
    if (updateData) {
      prevTabRef.current = -1;
      backOnUpdateRef.current && onBack && onBack();
    }
  }, [updateData]);

  useEffect(() => {
    if (updateError && prevTabRef.current >= 0) {
      setActiveTab(prevTabRef.current);
      prevTabRef.current = -1;
    }
  }, [updateError]);

  function getNameError(name: string): string {
    let error = "";
    const groupId = group?.id;
    if (!name.trim()) {
      error = __("Group name should be not empty");
    }
    else if (groupsData?.groups.find(group => group.id !== groupId && group.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("Group with the same name already exists");
    }
    return error;
  }

  const cancel = () => {
    onBack && onBack();
  };

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  async function onCreateUpdate(backOnUpdate = false): Promise<void> {
    const groupName = name.trim();
    if (!group) {
      for (let i = 0; i < tabCount; i++) {
        if (!validateTab(i)) {
          setActiveTab(i);
          return;
        }
      }

      const response = await createGroup({
        variables: {
          group: {
            name: groupName,
            addUserIds: Array.from(userIdSet.keys()),
            addPolicyIds: Array.from(policyIdSet.keys())
          }
        }
      });

      // update cache
      // update GroupsDocument, UsersDocument
      const groupId = response.data?.createGroup.id;
      if (groupId) {
        // update groups
        const groupsCacheData = client.readQuery<GroupsQuery>({
          query: GroupsDocument
        });

        const allUsers = usersData?.users ?? [];
        const selectedUsers = allUsers.filter((user) => userIdSet.has(user.id));

        const allPolicies = policiesData?.policies ?? [];
        const selectedPolicies = allPolicies.filter((policy) => policyIdSet.has(policy.id));

        const newGroup: ArrayElement<GroupsQuery["groups"]> = {
          __typename: "Group",
          id: groupId,
          name: groupName,
          users: selectedUsers.map((user) => { return { __typename: "User", id: user.id, name: user.name }; }),
          policies: selectedPolicies.map((policy) => { return { __typename: "Policy", id: policy.id, name: policy.name, isSystemManaged: policy.isSystemManaged }; }),
          isSystemManaged: false
        };

        const groupsCache = groupsCacheData?.groups ?? [];
        const groups = produce(groupsCache, (draft) => {
          draft.push(newGroup);
        });

        client.writeQuery<GroupsQuery>({
          query: GroupsDocument,
          data: {
            groups
          }
        });

        // update users
        if (selectedUsers.length > 0) {
          const usersCacheData = client.readQuery<UsersQuery>({
            query: UsersDocument
          });

          const usersCache = usersCacheData?.users ?? [];

          const addUserIndex: number[] = [];
          usersCache.forEach((user, userIndex) => {
            if (userIdSet.has(user.id)) {
              addUserIndex.push(userIndex);
            }
          });
          const users = produce(usersCache, (draft) => {
            for (const userIndex of addUserIndex) {
              draft[userIndex].groups.push(newGroup);
            }
          });

          client.writeQuery<UsersQuery>({
            query: UsersDocument,
            data: {
              users
            }
          });
        }

        if (response.data?.createGroup.warning) {
          Log.error(response.data.createGroup.warning);
        }
        onBack && onBack();
      }
    }
    else {
      if (!validateTab(activeTab)) {
        return;
      }

      backOnUpdateRef.current = backOnUpdate;
      prevTabRef.current = activeTab;

      const groupId = group.id;

      let response;
      switch (activeTab) {
        case 0:
          response = await updateGroup({
            variables: {
              id: groupId,
              group: {name: groupName}
            }
          });

          // update cache
          // update GroupsDocument, UsersDocument?
          if (response.data?.updateGroup) {
            const groupsCacheData = client.readQuery<GroupsQuery>({
              query: GroupsDocument
            });

            const groupsCache = groupsCacheData?.groups ?? [];
            const groupIndex = groupsCache.findIndex((group) => group.id === groupId);
            const groups = produce(groupsCache, (draft) => {
              draft[groupIndex].name = groupName;
            });

            client.writeQuery<GroupsQuery>({
              query: GroupsDocument,
              data: {
                groups
              }
            });

            // TODO: update name of group in user list
          }
          break;
        case 1:
          const addUserIds = Array.from(userIdSet.keys()).filter(id => !group.users.some(u => u.id === id));
          const removeUserIds = group.users.filter(user => !userIdSet.has(user.id)).map(user => user.id);
          response = await updateGroup({
            variables: {
              id: group.id,
              group: {
                addUserIds,
                removeUserIds
              }
            }
          });

          // update cache
          // update GroupsDocument, UsersDocument
          if (response.data?.updateGroup) {
            // update groups
            const groupsCacheData = client.readQuery<GroupsQuery>({
              query: GroupsDocument
            });

            const allUsers = usersData?.users ?? [];
            const selectedUsers = allUsers.filter((user) => userIdSet.has(user.id));

            const groupsCache = groupsCacheData?.groups ?? [];
            const groupIndex = groupsCache.findIndex((group) => group.id === groupId);
            const groups = produce(groupsCache, (draft) => {
              draft[groupIndex].users = selectedUsers.map((user) => { return { __typename: "User", id: user.id, name: user.name }; });
            });

            client.writeQuery<GroupsQuery>({
              query: GroupsDocument,
              data: {
                groups
              }
            });

            // update users
            const usersCacheData = client.readQuery<UsersQuery>({
              query: UsersDocument
            });

            const changedGroup = produce(group, (draft) => {
              draft.users = selectedUsers;
            });

            // add and remove user from groups
            const usersCache = usersCacheData?.users ?? [];
            const addUserIndex: number[] = [];
            const removeUserIndexToGroupIndex = new Map<number, number>();
            usersCache.forEach((user, userIndex) => {
              if (addUserIds.includes(user.id)) {
                const groupIndex = user.groups.findIndex((group) => group.id === groupId);
                if (groupIndex === -1) {
                  addUserIndex.push(userIndex);
                }
              }
              if (removeUserIds.includes(user.id)) {
                const groupIndex = user.groups.findIndex((group) => group.id === groupId);
                if (groupIndex !== -1) {
                  removeUserIndexToGroupIndex.set(userIndex, groupIndex);
                }
              }
            });

            if (addUserIndex.length > 0 || removeUserIndexToGroupIndex.size > 0) {
              const users = produce(usersCache, (draft) => {
                // add to groups
                for (const groupIndex of addUserIndex) {
                  draft[groupIndex].groups.push(changedGroup);
                }
                // remove from groups
                for (const [userIndex, groupIndex] of removeUserIndexToGroupIndex) {
                  draft[userIndex].groups.splice(groupIndex, 1);
                }
              });

              client.writeQuery<UsersQuery>({
                query: UsersDocument,
                data: {
                  users
                }
              });
            }
          }
          break;
        case 2:
          response = await updateGroup({
            variables: {
              id: group.id,
              group: {
                addPolicyIds: Array.from(policyIdSet.keys()).filter(id => !group.policies.some(p => p.id === id)),
                removePolicyIds: group.policies.filter(policy => !policyIdSet.has(policy.id)).map(policy => policy.id)
              }
            }
          });

          // update cache
          // update GroupsDocument
          if (response.data?.updateGroup) {
            const groupsCacheData = client.readQuery<GroupsQuery>({
              query: GroupsDocument
            });

            const allPolicies = policiesData?.policies ?? [];
            const policies = allPolicies.filter((policy) => policyIdSet.has(policy.id));

            const groupsCache = groupsCacheData?.groups ?? [];
            const groupIndex = groupsCache.findIndex((group) => group.id === groupId);
            const groups = produce(groupsCache, (draft) => {
              draft[groupIndex].policies = policies.map((policy) => { return { __typename: "Policy", id: policy.id, name: policy.name, isSystemManaged: policy.isSystemManaged }; });
            });

            client.writeQuery<GroupsQuery>({
              query: GroupsDocument,
              data: {
                groups
              }
            });
          }
          break;
      }
    }
  }

  function onActiveTabChange(index: number | string | undefined): void {
    if (typeof index !== "number") {
      return;
    }

    if (!group) {
      if (index < activeTab || validateTab(activeTab)) {
        setActiveTab(index);
      }
    }
    else {
      if (!hasChangesOnTab(activeTab)) {
        setActiveTab(index);
        return;
      }

      if (!validateTab(activeTab)) {
        return;
      }

      nextTabRef.current = index;
      setChangesDialogOpen(true);
    }
  }

  function hasChangesOnTab(tabIndex: number): boolean {
    if (!group) {
      switch (tabIndex) {
        case 0: return !!name.trim();
        case 1: return userIdSet.size > 0;
        case 2: return policyIdSet.size > 0;
        default: return false;
      }
    }
    else {
      switch (tabIndex) {
        case 0: return name.trim() !== group.name;
        case 1: return group.users.some(user => !userIdSet.has(user.id)) ||
          Array.from(userIdSet.keys()).some(id => !group.users.some(u => u.id === id));
        case 2: return group.policies.some(policy => !policyIdSet.has(policy.id)) ||
          Array.from(policyIdSet.keys()).some(id => !group.policies.some(p => p.id === id));
        default: return false;
      }
    }
  }

  function validateTab(tabIndex: number): boolean {
    let result = true;
    switch (tabIndex) {
      case 0:
        const nameError = getNameError(name);
        if (nameError) {
          setNameError(nameError);
          result = false;
        }
        break;
    }
    return result;
  }

  function discardTabChanges(tabIndex: number): void {
    switch (tabIndex) {
      case 0:
        setName(group?.name ?? "");
        setNameError("");
        break;
      case 1:
        setUserIdSet(new Set<string>(group?.users.map(user => user.id) ?? []));
        break;
      case 2:
        setPolicyIdSet(new Set<string>(group?.policies.map(policy => policy.id) ?? []));
        break;
    }
  }

  const error = createError ?? updateError;

  const panes = [
    {
      menuItem: __("Group Properties"),
      render: () => (
        <Tab.Pane>
          <Form className="CreateUpdateGroup-Form" onSubmit={e => { e.preventDefault(); }}>
            <Form.Field
              control={Form.Input}
              label={__("Name")}
              placeholder={__("Name")}
              autoFocus
              value={name}
              error={nameError ? { content: nameError, pointing: "below" } : undefined}
              readOnly={group?.isSystemManaged}
              onInput={onNameInput}
            />
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Users"),
      render: () => (
        <Tab.Pane>
          <AssignObjects
            items={usersData?.users.map(user => ({ ...user, children: user.groups }))}
            assignedIdSet={userIdSet}
            readOnly={group?.isSystemManaged}
            itemSingleText={__("User")}
            itemPluralText={__("Users")}
            parentSingleText={__("Group")}
            childPluralText={__("Groups")}
            onAssignedIdSetChange={idSet => setUserIdSet(idSet)}
          />
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Policies"),
      render: () => (
        <Tab.Pane>
          <AssignObjects
            items={policiesData?.policies.map(policy => ({ ...policy, children: policy.statements }))}
            assignedIdSet={policyIdSet}
            readOnly={group?.isSystemManaged}
            itemSingleText={__("Policy")}
            itemPluralText={__("Policies")}
            parentSingleText={__("Group")}
            childPluralText={__("Statements")}
            onAssignedIdSetChange={idSet => setPolicyIdSet(idSet)}
          />
        </Tab.Pane>
      )
    }
  ];

  return (
    <Segment className="CreateUpdateGroup">
      <WithQueryStatus error={groupsError || usersError || policiesError} loading={groupsLoading || usersLoading || policiesLoading}>
        {!!error && <Message error content={error.message}/>}

        <div className="CreateUpdateGroup-TopButtons">
          <Button onClick={() => cancel()}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
          {!group &&
          <>
            <Button disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab - 1)}>
              <Icon name="arrow alternate circle left"/>{__("Back")}
            </Button>
            {activeTab < tabCount - 1 &&
            <Button positive onClick={() => onActiveTabChange(activeTab + 1)}>
              <Icon name="arrow alternate circle right"/>{__("Next")}
            </Button>}
          </>}
          {(!!group || activeTab >= tabCount - 1) && !group?.isSystemManaged &&
          <Button
            positive
            disabled={!!group && !hasChangesOnTab(activeTab)}
            onClick={() => onCreateUpdate(true)}
          >
            <Icon name={!group ? "plus" : "check"}/>{!group ? __("Create") : __("Save")}
          </Button>}
        </div>

        <Tab
          panes={panes}
          className="CreateUpdateGroup-Tab"
          activeIndex={activeTab}
          onTabChange={(e, { activeIndex }) => onActiveTabChange(activeIndex)}
        />

        <Modal open={changesDialogOpen} onClose={() => setChangesDialogOpen(false)}>
          <Header>{__("Unsaved Changes")}</Header>
          <Modal.Content>{__("Group has unsaved changes. Would you like to save the changes?")}</Modal.Content>
          <Modal.Actions>
            <Button positive onClick={async () => {
              await onCreateUpdate();
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="check"/>{__("Save")}
            </Button>
            <Button negative onClick={() => {
              discardTabChanges(activeTab);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="undo"/>{__("Discard")}
            </Button>
            <Button onClick={() => setChangesDialogOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {(createLoading || updateLoading) && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment>
  );
};

export default CreateUpdateGroup;
