import React, { FC, useEffect, useMemo, useRef, useState } from "react";
import { __, clone } from "@solid/libs";
import { Button, Icon, Message, Popup, Segment, Tab } from "semantic-ui-react";
import ButtonsWrapper from "../ButtonsWrapper";
import { useNavigate } from "react-router-dom";
import produce from "immer";

import { AutoForm } from "components/AutoForm";
import GatewayDevices from "./GatewayDevices/GatewayDevices";
import { WidgetProps } from "components/Widgets";
import {
  AssociatedGatewayDevice,
  AssociatedGatewayDeviceInput,
  AuthType,
  DeviceBaseConfigInput,
  DeviceBaseConfigType,
  DeviceFunctionalAspectInput,
  DeviceFunctionalAspectType,
  DeviceInput,
  GatewayDeviceInput,
  GatewayDirectory,
  GatewayEventInput,
  GatewayModel,
  SubscribedGatewayEventInput,
  useCreateGatewayMutation,
  useDeviceLazyQuery,
  useDevicesByAspectTypesShortQuery,
  useProbeGatewayDirectoriesLazyQuery,
  useProbeGatewayLazyQuery,
  useUpdateDeviceMutation,
} from "@generated/graphql";
import { GatewayEvents } from "./GatewayEvents/GatewayEvents";
import { aspectToAspectInput, Device, useDeviceActions } from "@core/actions";
import GatewayAuth from "./GatewayAuth/GatewayAuth";
import GatewayProperties from "./GatewayProperties/GatewayProperties";

import "./style.css";

export type DeviceProperties = {
  [Property in keyof Device]?: Device[Property];
};

export interface ComponentWithFormRef {
  getFormRef: () => AutoForm | null;
}

interface GatewaySettingsProps extends WidgetProps {
  id?: string;
  onCreated?: (id: string) => void;
}

const GatewaySettings: FC<GatewaySettingsProps> = ({ id, cellProps, setCellProps, onCreated }) => {
  const navigate = useNavigate();
  const [getGateway, { data: gatewayData /*error: gatewayError*/ }] = useDeviceLazyQuery({ fetchPolicy: "no-cache" });
  const authRef = useRef<ComponentWithFormRef>(null);
  const propertiesRef = useRef<ComponentWithFormRef>(null);
  const {
    data: avatarData,
    error: avatarError,
    loading: avatarLoading,
  } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] } });
  const [probeDeviceDirectories, { data: probeDirectoryData, error: probeDirectoriesError, loading: probeDirectoriesLoading }] = useProbeGatewayDirectoriesLazyQuery({
    fetchPolicy: "no-cache",
  });
  const [deviceList, setDeviceList] = useState<GatewayDeviceInput[]>([]);
  const [eventList, setEventList] = useState<GatewayEventInput[]>([]);
  const [associatedGatewayDevices, setAssociatedGatewayDevices] = useState<AssociatedGatewayDeviceInput[]>([]);
  const [subscribedEvents, setSubscribedEvents] = useState<SubscribedGatewayEventInput[]>([]);
  const [getDevices, { data: devices, error: devicesError, loading: devicesLoading }] = useProbeGatewayLazyQuery({ fetchPolicy: "no-cache" });
  const [directoriesData, setDirectoriesData] = useState<GatewayDirectory[]>([]);
  const [activeDirectory, setActiveDirectory] = useState<GatewayDirectory>();
  const [createGateway, { /*data: createData,*/ error: createError, loading: createLoading }] = useCreateGatewayMutation();
  const [updateGateway, { /*data: updateData,*/ /*error: updateError,*/ loading: updateLoading }] = useUpdateDeviceMutation();
  const [authState, setAuthState] = useState<boolean>(false);

  const { updateCachedDevice } = useDeviceActions({ skipSubscription: true });

  const [initialGateway, setInitialGateway] = useState<DeviceProperties>();
  const [currentGateway, setCurrentGateway] = useState<DeviceProperties>();
  const [activeTab, setActiveTab] = useState<number>(0);
  const [aspect, setAspect] = useState<DeviceFunctionalAspectInput>();

  function onUpdateProperties(property: DeviceProperties) {
    setCurrentGateway((prevGateway) => ({ ...prevGateway, ...property }));
  }

  useEffect(() => {
    id && getGateway({ variables: { id } });
  }, [id]);

  useEffect(() => {
    if (gatewayData && gatewayData.device) {
      const currentDevice = clone(gatewayData.device);
      const initialDevice = clone(gatewayData.device);
      console.log(currentDevice);
      setCurrentGateway(currentDevice);
      setInitialGateway(initialDevice);
      currentDevice.aspects.map((aspect) => {
        if (aspect.__typename === "DFA_Gateway" && aspect.type === DeviceFunctionalAspectType.Gateway) {
          setAspect(aspectToAspectInput(aspect));
          setDirectoriesData(aspect.directories);
          setActiveDirectory(aspect.directory);
          setAssociatedGatewayDevices(aspect.associatedGatewayDevices.map((device) => ({ id: device.id, associatedDevices: device.associatedDevices })));
          setSubscribedEvents(aspect.subscribedEvents.map((event) => ({ id: event.id, type: event.type })));
          setEventList(aspect.events.map((event) => ({ id: event.id, name: event.name, type: "" })));
          setDeviceList(aspect.devices.map((device) => ({ id: device.id, name: device.name })));
        }
      });
    }
    if (id && gatewayData?.device === null) {
      navigate("/view/Gateways");
    }
  }, [gatewayData]);

  function isDevicesChanged(): boolean | undefined {
    if (!initialGateway) return true;
    return initialGateway.aspects?.every((aspect) => {
      if (aspect.__typename === "DFA_Gateway" && aspect.type === DeviceFunctionalAspectType.Gateway) {
        if (aspect.subscribedEvents.length !== subscribedEvents.length) return false;
        if (aspect.associatedGatewayDevices.length !== associatedGatewayDevices.length) return false;
        let isChanged: boolean = true;
        aspect.subscribedEvents.every((aspectEvent) => {
          subscribedEvents.some((subscribedEvent) => {
            if (aspectEvent.id === subscribedEvent.id) {
              if (aspectEvent.type !== subscribedEvent.type) {
                isChanged = false;
                return;
              }
            }
          });
        });
        if (!isChanged) return isChanged;

        aspect.associatedGatewayDevices.forEach((aspectDevice) =>
          associatedGatewayDevices.forEach((associatedGatewayDevice) => {
            if (aspectDevice.id === associatedGatewayDevice.id) {
              console.log("ASPECT: ", aspectDevice, "AssociatedGateway: ", associatedGatewayDevice);
              if (aspectDevice.associatedDevices.length !== associatedGatewayDevice.associatedDevices.length) {
                isChanged = false;
                return;
              }
              const isEqual = associatedGatewayDevice.associatedDevices.every((element) => {
                return aspectDevice.associatedDevices.includes(element);
              });
              isChanged = isEqual;
              return;
            }
          })
        );
        if (!isChanged) return isChanged;
      }
      return true;
    });
  }

  useEffect(() => {
    if (!setCellProps || !id) return;
    const title: string = !id ? __("Create Gateway") : __("Configure Gateway");
    setCellProps({ title });
  }, [setCellProps, id]);

  function back() {
    navigate(-1);
  }

  async function getDirectories() {
    const propertiesFormRef = getFormRef(propertiesRef);
    if (!propertiesFormRef?.validate()) return;

    const values = propertiesFormRef?.getValues();
    try {
      const directories = await probeDeviceDirectories({
        variables: {
          input: {
            connect: {
              authType: AuthType.NoAuth,
              protocol: values["protocol"],
              host: values["host"],
              port: values["port"],
            },
            platformId: values["platformId"],
            model: values["model"],
          },
        },
      });
      if (directories.error) {
        throw new Error(directories.error.message);
      }
      setAuthState(true);
      setActiveTab(1);
    } catch (e) {
      console.log("ERROR: ", e);
    }
  }

  function getGatewayInput(): DeviceInput {
    const propertiesFormRef = getFormRef(propertiesRef);
    const authFormRef = getFormRef(authRef);

    if (!propertiesFormRef || !authFormRef || !currentGateway) throw new Error("Invalid form reference");
    const authValues = authFormRef.getValues();
    const propertyValues = propertiesFormRef.getValues();
    const config = getGatewayConfigInput();

    const updatedAspect = produce(aspect, (draft) => {
      if (!draft) return;
      let directories;
      //TODO: get directories from aspect, not from form
      authFormRef?.props.schema.find((schema) => {
        if (schema?.name === "directories") {
          directories = schema.dataSource;
        }
      });
      draft.devices = deviceList;
      draft.directories = directories;
      draft.directoryId = authValues["directories"];
      draft.events = eventList;
      if (associatedGatewayDevices) {
        draft.associatedGatewayDevices = associatedGatewayDevices;
      }
      if (subscribedEvents) {
        draft.subscribedEvents = subscribedEvents;
      }
    });
    Object.assign(config.connect, {
      user: authValues["gatewayLogin"],
      pass: authValues["gatewayPass"],
    });
    const input: DeviceInput = {
      name: propertyValues["name"],
      enabled: propertyValues["enabled"] === "on",
      location: propertyValues["location"],
      platformId: propertyValues["platformId"],
      aspects: updatedAspect ? [updatedAspect] : [],
      config,
    };
    return input;
  }

  function getGatewayConfigInput(): DeviceBaseConfigInput {
    if (!currentGateway || !currentGateway.config?.configType) {
      return { configType: DeviceBaseConfigType.Gateway, connect: { authType: AuthType.NoAuth } };
    }

    const configType = currentGateway.config.configType;
    // eslint-disable-next-line @typescript-eslint/naming-convention
    const { __typename, ...connectFields } = currentGateway.config.connect;
    const configInput: DeviceBaseConfigInput = {
      configType,
      model: currentGateway.config.model,
      connect: { ...connectFields },
    };

    return configInput;
  }

  useEffect(() => {
    console.log("initialGateway", initialGateway);
    console.log(subscribedEvents);
  }, [initialGateway]);

  async function onAddGateway() {
    if (!aspect) return;

    const propertiesFormRef = getFormRef(propertiesRef);
    const authFormRef = getFormRef(authRef);
    if (!propertiesFormRef?.validate() || !authFormRef?.validate()) return;
    const propertyValues = propertiesFormRef?.getValues();
    const authValues = authFormRef?.getValues();
    const updatedAspect = produce(aspect, (draft) => {
      let directories;
      authFormRef?.props.schema.forEach((schema) => {
        if (schema?.name === "directories") {
          directories = schema.dataSource;
        }
      });
      draft.devices = deviceList;
      draft.directories = directories;
      draft.directoryId = authValues["directories"];
      draft.events = eventList;
      if (associatedGatewayDevices) {
        draft.associatedGatewayDevices = associatedGatewayDevices;
      }
      if (subscribedEvents) {
        draft.subscribedEvents = subscribedEvents;
      }
    });
    try {
      const { data } = await createGateway({
        variables: {
          device: {
            name: propertyValues["name"],
            location: propertyValues["location"],
            enabled: propertyValues["enabled"] === "on",
            aspects: [updatedAspect],
            platformId: propertyValues["platformId"],
            config: {
              configType: DeviceBaseConfigType.Gateway,
              connect: {
                authType: AuthType.NoAuth,
                protocol: propertyValues["protocol"],
                host: propertyValues["host"],
                port: propertyValues["port"],
                user: authValues["gatewayLogin"],
                pass: authValues["gatewayPass"],
              },
              model: GatewayModel.Lenel,
            },
          },
        },
      });
      if (data) {
        const deviceId = data.createGatewayDevice.id;
        try {
          onCreated?.(deviceId);
          await updateCachedDevice(deviceId);
        } catch (e) {
          console.log("Updating cache error:", e);
        }
      }
    } catch (e) {
      console.log(e);
    }
  }

  async function onUpdateGateway() {
    if (!id) return;
    const input = getGatewayInput();
    try {
      const { data } = await updateGateway({
        variables: {
          id,
          device: input,
        },
      });
      if (data?.updateDevice.updated) {
        try {
          const updatedGateway = clone(currentGateway);
          setInitialGateway(updatedGateway);
          await updateCachedDevice(id ?? "");
        } catch (e) {
          console.log("Updating cache error:", e);
        }
      }
    } catch (e) {
      console.log(e);
    }
  }

  async function onProbe() {
    const lenelVersion = aspect?.lenelVersion || currentGateway?.version;
    if (!lenelVersion) throw Error("Can't detect Lenel version");
    const propertiesFormRef = getFormRef(propertiesRef);
    const authFormRef = getFormRef(authRef);

    if (!propertiesFormRef?.validate() || !authFormRef?.validate()) return;
    const propertyValues = propertiesFormRef?.getValues();
    const authValues = authFormRef?.getValues();
    try {
      const devices = await getDevices({
        variables: {
          input: {
            connect: {
              authType: AuthType.NoAuth,
              protocol: propertyValues["protocol"],
              host: propertyValues["host"],
              port: propertyValues["port"],
              user: authValues["gatewayLogin"],
              pass: authValues["gatewayPass"],
            },
            platformId: propertyValues["platformId"],
            model: propertyValues["model"],
            version: lenelVersion,
            directoryId: authValues["directories"],
          },
        },
      });

      if (devices.error) {
        throw new Error(devices.error.message);
      }
      authState && setAuthState(false);
      setActiveTab(2);
    } catch (e) {
      console.log("ERROR: ", e);
    }
  }

  useEffect(() => {
    if (devices?.probeGateway) {
      const updatedDevices = devices?.probeGateway.devices.map((device) => {
        return {
          name: device.name,
          id: device.id,
        };
      });
      const updatedEvents = devices?.probeGateway.events.map((event) => {
        return {
          name: event.name,
          id: event.id,
          type: "",
        };
      });
      setDeviceList(updatedDevices);
      setEventList(updatedEvents);
    }
  }, [devices]);

  function updateAssociatedDevices(selectedDevice: AssociatedGatewayDevice, cameras: string[]) {
    if (!cameras) return;
    const devicesArray = produce(associatedGatewayDevices, (draft) => {
      const index = draft.findIndex((device) => device.id === selectedDevice.id);
      if (index === -1) {
        draft.push({
          id: selectedDevice.id,
          associatedDevices: cameras,
        });
      } else if (cameras.length) {
        draft[index].associatedDevices = cameras;
      } else {
        draft.splice(index, 1);
      }
    });
    setAssociatedGatewayDevices(devicesArray);

    setCurrentGateway((prevGateway) => {
      const newGateway = { ...prevGateway };
      newGateway?.aspects?.map((aspect) => {
        if (aspect.__typename === "DFA_Gateway" && aspect.type === DeviceFunctionalAspectType.Gateway) {
          aspect.associatedGatewayDevices = devicesArray;
          return aspect;
        }
        return aspect;
      });

      return newGateway;
    });
  }

  function updateSubscribedEvents(selectedEvent: SubscribedGatewayEventInput | SubscribedGatewayEventInput[], type: string) {
    const eventsArray = Array.isArray(selectedEvent) ? selectedEvent : produce(subscribedEvents, (draft) => {
      const index = draft.findIndex((event) => event.id === selectedEvent.id);
      if (index === -1) {
        draft.push({
          id: selectedEvent.id,
          type,
        });
      } else if (type !== "NONE") {
        draft[index].type = type;
      } else {
        draft.splice(index, 1);
      }
    });
    setSubscribedEvents(eventsArray);
    setCurrentGateway((prevGateway) => {
      const newGateway = { ...prevGateway };
      newGateway?.aspects?.map((aspect) => {
        if (aspect.__typename === "DFA_Gateway" && aspect.type === DeviceFunctionalAspectType.Gateway) {
          aspect.subscribedEvents = eventsArray;
          return aspect;
        }
        return aspect;
      });

      return newGateway;
    });
  }

  /*useEffect(() => {
    if (createData && !createError && !loading) {
      onCreated?.(createData.createGatewayDevice.id);
    }
  }, [createData]);*/

  useEffect(() => {
    if (probeDirectoryData?.probeGatewayDirectories) {
      probeDirectoryData?.probeGatewayDirectories.deviceProbe.aspects.map((aspect) => {
        if (aspect.__typename === "DFA_Gateway" && aspect.type === DeviceFunctionalAspectType.Gateway) {
          setAspect(aspectToAspectInput(aspect));
        }
      });
      setDirectoriesData(probeDirectoryData?.probeGatewayDirectories.directories);
    }
  }, [probeDirectoryData]);

  function onActiveTabChange(activeIndex: number) {
    validateSettings(activeIndex) && setActiveTab(activeIndex);
  }

  const isAuthChanged = isGatewayAuthChanged();
  const isConfigChanged = isGatewayConfigChanged();
  const isPropertiesChanged = getFormRef(propertiesRef)?.getIsChanged();

  function validateSettings(tabIndex: number): boolean {
    const propertiesFormRef = getFormRef(propertiesRef);
    const authFormRef = getFormRef(authRef);
    switch (activeTab) {
      case 0: {
        if ((propertiesFormRef?.validate() && !id && probeDirectoryData) || (!isConfigChanged && id)) {
          return true;
        }
        return false;
      }
      case 1: {
        if ((authFormRef?.validate() && !id && devices && !authState) || (!isAuthChanged && id && !authState) || (authState && tabIndex === 0)) {
          return true;
        }
        return false;
      }
      default:
        return true;
    }
  }

  function isGatewayConfigChanged(): boolean {
    if (!currentGateway || !initialGateway) return false;

    const formRef = getFormRef(propertiesRef);
    if (!formRef || !initialGateway || !initialGateway.config) return false;

    const values = formRef.getValues();
    const { config } = initialGateway;

    return (
      config.model !== values["model"] ||
      config.connect.host !== values["host"] ||
      config.connect.port !== values["port"] ||
      config?.connect.protocol !== values["protocol"]
    );
  }

  function isGatewayAuthChanged(): boolean {
    if (!currentGateway || !initialGateway) return false;
    const formRef = getFormRef(authRef);
    if (!formRef || !initialGateway || !initialGateway.config || !activeDirectory) return false;

    const { config } = initialGateway;

    const authConf = config?.connect;
    const values = formRef.getValues();
    return authConf?.user !== values["gatewayLogin"] || authConf?.pass !== values["gatewayPass"] || activeDirectory.id !== values["directories"];
  }

  const panes = [
    {
      menuItem: __("Gateway Properties"),
      pane: (
        <Tab.Pane key="gatewayProperties">
          {!!probeDirectoriesError && <Message error content={probeDirectoriesError.message} />}
          {!!avatarError && <Message error content={avatarError.message} />}
          <GatewayProperties avatars={avatarData} ref={propertiesRef} gateway={initialGateway} onChange={onUpdateProperties} />
        </Tab.Pane>
      ),
    },
    {
      menuItem: __("Gateway Authentication"),
      pane: (
        <Tab.Pane key="gatewayAuth">
          {!!devicesError && <Message error content={devicesError.message} />}
          <GatewayAuth ref={authRef} directories={directoriesData} gateway={initialGateway} onChange={onUpdateProperties} getDirectories={getDirectories} />
        </Tab.Pane>
      ),
    },
    {
      menuItem: __("Devices"),
      pane: (
        <Tab.Pane key="gatewayDevices">
          {!!createError && <Message error content={createError.message} />}
          {deviceList && (
            <GatewayDevices
              devices={deviceList}
              associatedDevices={associatedGatewayDevices}
              updateAssociatedDevices={(selectedDevice, cameras) => updateAssociatedDevices(selectedDevice, cameras)}
            />
          )}
        </Tab.Pane>
      ),
    },
    {
      menuItem: __("Events"),
      pane: (
        <Tab.Pane key="gatewayEvents">
          {!!createError && <Message error content={createError.message} />}
          {eventList && (
            <GatewayEvents
              events={eventList}
              subscribedEvents={subscribedEvents}
              updateSubscribedEvents={(selectedEvent, type) => updateSubscribedEvents(selectedEvent, type)}
            />
          )}
        </Tab.Pane>
      ),
    },
  ];

  const saveDisabled = useMemo<boolean>(() => {
    const deviceChanged = isDevicesChanged();
    if (authState) return true;
    if (deviceChanged && !isPropertiesChanged) return true;
    if (isConfigChanged) return true;
    if (isAuthChanged) return true;
    return false;
  }, [authState, isPropertiesChanged, isConfigChanged, isAuthChanged, associatedGatewayDevices, subscribedEvents]);

  const loading = probeDirectoriesLoading || devicesLoading || avatarLoading || createLoading || updateLoading;
  const isReprobe = id && isConfigChanged;

  return (
    <Segment className="GatewaySettings" loading={loading}>
      <ButtonsWrapper>
        <Button onClick={() => back()}>
          <Icon name="arrow left" />
          {__("Back")}
        </Button>

        {activeTab === 0 && (
          <Popup
            trigger={
              <Button className="probe-test-btn" positive onClick={() => getDirectories()}>
                <Icon name="play" />
                {__("Probe")}
              </Button>
            }
            content={__("Gateway configuration was changed and should be reprobed")}
            position="right center"
            open={!!isReprobe}
          />
        )}
        {activeTab > 0 && !id && (
          <Button disabled={activeTab <= 0} onClick={() => setActiveTab(activeTab - 1)}>
            <Icon name="arrow alternate circle left" />
            {__("Back")}
          </Button>
        )}
        {activeTab === 1 && (
          <Popup
            trigger={
              <Button className="auth-test-btn" positive onClick={() => onProbe()}>
                <Icon name="play" />
                {__("Auth")}
              </Button>
            }
            content={__("Gateway configuration was changed and should be reprobed")}
            position="right center"
            open={!!((id && authState) || (id && isAuthChanged))}
          />
        )}
        {!id ? (
          activeTab >= 2 && (
            <Popup
              trigger={
                <Button className="add-test-btn" positive onClick={() => onAddGateway()}>
                  <Icon name="play" />
                  {__("Add")}
                </Button>
              }
              content={__("Gateway configuration was changed and should be reprobed")}
              position="right center"
            />
          )
        ) : (
          <Popup
            trigger={
              <Button className="save-test-btn" positive onClick={() => onUpdateGateway()} disabled={saveDisabled}>
                <Icon name="check" />
                {__("Save")}
              </Button>
            }
            content={__("Gateway configuration was changed and should be reprobed")}
            position="right center"
          />
        )}
      </ButtonsWrapper>
      <Tab
        className="GatewaySettings-Tab"
        panes={panes}
        renderActiveOnly={false}
        activeIndex={activeTab}
        onTabChange={(e, { activeIndex }) => typeof activeIndex === "number" && onActiveTabChange(activeIndex)}
      />
    </Segment>
  );
};

export default GatewaySettings;

function getFormRef(ref: React.RefObject<ComponentWithFormRef>): AutoForm | undefined {
  if (!ref.current) return undefined;
  const formRef: AutoForm | null = ref.current.getFormRef();
  if (formRef) return formRef;
  return undefined;
}
