import React, { useState, useEffect, useRef } from "react";
import { Segment, Icon, Popup, Ref } from "semantic-ui-react";
import { useLocation, useNavigate, useParams } from "react-router-dom";
import { Table, Column, AutoSizer } from "react-virtualized";
import {useDevicesByAspectTypesShortQuery, DeviceFunctionalAspectType, HealthStatus} from "@generated/graphql";
import { WidgetProps } from "components/Widgets";
import WithQueryStatus from "components/WithQueryStatus";
import {
  DeviceShort,
  useLinkDeviceInfo,
  getLinkChannelStatusText,
  linkHealthStatusText,
  getLinkStatusUpdateAge
} from "@core/actions";
import ConfigureLinkWrapper from "./ConfigureLink/ConfigureLinkWrapper";
import {__} from "@solid/libs/i18n";
import ListFilter from "components/ListFilter";
import { RouteParams } from "@core/types";
import { ConfigureLinkTabRoute } from "./ConfigureLink/ConfigureLink";
import DeviceInfoText from "./DeviceInfoText";
import DeviceInfoTrigger from "components/Admin/AdminLinkList/DeviceInfoTrigger";

import "./style.css";

type AdminLinkListProps = WidgetProps;

const refetchInterval = 60000;
export const defaultActiveTab = ConfigureLinkTabRoute.Properties;

const AdminLinkList = ({ cellProps, setCellProps }: AdminLinkListProps) => {
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", linkId = "", tabId = defaultActiveTab } = useParams<RouteParams>();
  const { loading, error, data, refetch } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] } });
  const [devices, setDevices] = useState<DeviceShort[]>();
  const { getLinkDevicesCounts, loading: infoLoading, error: infoError } = useLinkDeviceInfo();
  const refetchIntervalRef = useRef<NodeJS.Timeout | undefined>();
  const rootRef = useRef<HTMLDivElement>(null);
  const [selectedId, setSelectedId] = useState("");
  const selectedLink = selectedId ? devices?.find(device => device.id === selectedId) : undefined;
  const [configOpen, setConfigOpen] = useState(false);
  const [filterVisible, setFilterVisible] = useState<boolean>(false);
  const [searchText, setSearchText] = useState("");

  useEffect(() => {
    if (data?.devicesByAspectTypes) {
      const devices = getDevices(searchText);
      setDevices(devices);
    }
  }, [data, searchText]);

  useEffect(() => {
    if (!setCellProps || configOpen) {
      return;
    }

    const title =
      <div className="AdminCameraList-Title">
        <div className="AdminCameraList-TitleName">{cellProps?.title}</div>

        <Popup
          trigger={
            <Icon
              name="search"
              className="AdminCameraList-TitleIcon"
              onClick={() => setFilterVisible(visible => !visible)}
            />}
          content={filterVisible ? __("Hide Filter") : __("Show Filter")}
          position="bottom right"
        />
      </div>;

    setCellProps({ title });
  }, [filterVisible, configOpen, selectedLink, setCellProps]);

  useEffect(() => {
    if (linkId && devices) {
      const isLinkExist = devices.some(device => device.id === linkId);
      if (isLinkExist) {
        setConfigOpen(true);
        setSelectedId(linkId);
      }
    }
  }, [devices, linkId]);

  useEffect(() => {
    if (viewId && configOpen) {
      setConfigOpen(false);
    }
  }, [viewId]);

  useEffect(() => {
    refetchIntervalRef.current = setInterval(() => refetch(), refetchInterval);

    return () => {
      if (refetchIntervalRef.current) {
        clearInterval(refetchIntervalRef.current);
        refetchIntervalRef.current = undefined;
      }
    };
  }, []);

  useEffect(() => {
    if (configOpen && refetchIntervalRef.current) {
      clearInterval(refetchIntervalRef.current);
      refetchIntervalRef.current = undefined;
    }
    if (!configOpen && !refetchIntervalRef.current) {
      refetchIntervalRef.current = setInterval(() => refetch(), refetchInterval);
    }
  }, [configOpen]);

  /* eslint-disable react/no-unused-prop-types */
  function getActionsContent({ rowData }: { rowData: DeviceShort }) {
    return (
      <Popup trigger={
        <Icon name="edit" className="AdminLinkList-IconButton" onClick={() => onEditClick(rowData.id)}/>
        }
        content={__("Configure")}
      />
    );
  }

  function getHealthStatusContent({ rowData }: { rowData: DeviceShort }) {
    return (
      <span className={rowData.healthStatus === HealthStatus.Off ? "AdminLinkList-Red" : undefined}>
        {linkHealthStatusText[rowData.healthStatus]}
      </span>
    );
  }

  function getChannelStatusContent({ rowData }: { rowData: DeviceShort }) {
    const status = getLinkChannelStatusText(rowData);
    const age = getLinkStatusUpdateAge(rowData);
    return (
      <>
        <span>{status || __("N/A")}</span>
        {!!age && <>&nbsp;<span> - {age}</span></>}
      </>
    );
  }

  function getDevicesInfoContent({ rowData }: { rowData: DeviceShort }) {
    const { camera, sensor, gateway, total } = getLinkDevicesCounts(rowData);

    if (total === 0) {
      return <span className="AdminLinkList-DeviceInfo">{__("No devices registered")}</span>;
    }

    return (
      <Popup
        trigger={
          <div className="AdminLinkList-DeviceInfoWrapper">
            { camera.total > 0 && <DeviceInfoTrigger countByStatus={camera.countByStatus} total={camera.total} icon="video" /> }
            { sensor.total > 0 && <DeviceInfoTrigger countByStatus={sensor.countByStatus} total={sensor.total} icon="bell" /> }
            { gateway.total > 0 && <DeviceInfoTrigger countByStatus={gateway.countByStatus} total={gateway.total} icon="server" /> }
          </div>
          }
        >
        <b>{__("Total")} - {total}</b>
        { camera.total > 0 && <DeviceInfoText countByStatus={camera.countByStatus} title={__("Cameras")} /> }
        { sensor.total > 0 && <DeviceInfoText countByStatus={sensor.countByStatus} title={__("Sensors")} /> }
        { gateway.total > 0 && <DeviceInfoText countByStatus={gateway.countByStatus} title={__("Gateways")} /> }
      </Popup>
    );
  }
  /* eslint-enable react/no-unused-prop-types */

  function onEditClick(id: string): void {
    setSelectedId(id);
    setConfigOpen(true);
    !pathname.includes(`/view/link/edit/${id}/${tabId}`) && navigate(`/view/link/edit/${id}/${tabId}`);
  }

  function getDevices(searchText?: string): DeviceShort[] {
    const text = searchText?.toLocaleUpperCase();
    const list: DeviceShort[] = data?.devicesByAspectTypes
      ?.filter(link =>
        (!text
          || link.name.toLocaleUpperCase().includes(text)
          || link.healthStatus.toLocaleUpperCase().includes(text)
          || getChannelStatusString(link).toLocaleUpperCase().includes(text)
          || link.config.connect.host?.toLocaleUpperCase().includes(text)
          || getNoDevicesString(link).toLocaleUpperCase().includes(text)
        ))
      .sort((a, b) => {
        return a.name.localeCompare(b.name, undefined, { sensitivity: "base" });
      })
      ?? [];
    return list;
  }

  function getNoDevicesString(link: DeviceShort): string {
    const { total } = getLinkDevicesCounts(link);
    const result = total === 0 ? __("No devices registered") : "";
    return result;
  }

  function getChannelStatusString(link: DeviceShort): string {
    const status = getLinkChannelStatusText(link);
    const age = getLinkStatusUpdateAge(link);
    const result = `${status || __("N/A")}${age || ""}`;
    return result;
  }

  return !configOpen ?
    <Ref innerRef={rootRef}>
      <Segment className="AdminLinkList">
        <WithQueryStatus
          loading={loading || infoLoading}
          error={error || infoError}>

          {filterVisible &&
          <ListFilter
            searchText={searchText}
            nameFilter
            rootRef={rootRef}
            onSearchTextChange={text => setSearchText(text)}
            filterTextPlaceholder={__("Filter by text")}
          />}

          { devices &&
          <div className="AdminLinkList-Grid">
            <AutoSizer style={{ width: "100%", height: "100%" }}>
              {({ width, height }) => (
                <Table
                  className="AdminLinkList-Table"
                  width={width}
                  height={height}
                  rowHeight={34}
                  headerHeight={34}
                  rowCount={devices.length}
                  rowGetter={({ index }) => devices[index]}>
                  <Column
                    dataKey="actions"
                    label=""
                    width={38}
                    flexShrink={0}
                    cellRenderer={getActionsContent}
                />
                  <Column
                    dataKey="name"
                    label={__("Name")}
                    width={100}
                    flexGrow={1}
                />
                  <Column
                    dataKey="healthStatus"
                    label={__("Status")}
                    width={80}
                    flexShrink={0}
                    cellRenderer={getHealthStatusContent}
                />
                  <Column
                    dataKey="channelStatus"
                    label={__("Connection Status")}
                    width={240}
                    flexShrink={0}
                    cellRenderer={getChannelStatusContent}
                />
                  <Column
                    dataKey="ipAddress"
                    label={__("IP Address")}
                    width={130}
                    flexGrow={0}
                    cellDataGetter={({ rowData }: { rowData: DeviceShort }) => rowData.config.connect.host ?? ""}
                />
                  <Column
                    dataKey="devicesInfo"
                    label={__("Devices")}
                    width={300}
                    flexGrow={1}
                    cellRenderer={getDevicesInfoContent}
                />
                </Table>
              )}
            </AutoSizer>
          </div>}
        </WithQueryStatus>
      </Segment>
    </Ref> :
    selectedLink && data &&
      <ConfigureLinkWrapper
        deviceId={selectedLink.id}
        links={data.devicesByAspectTypes}
        onBack={() => setConfigOpen(false)}
        cellProps={cellProps}
        setCellProps={setCellProps}
    />;
};

export default AdminLinkList;
