import React, { useEffect, useMemo, useRef, useState } from "react";
import { Button, Form, Icon, Input, List, Popup, Segment } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";

import { StreamShare, useGetDeviceSharedStreamLazyQuery } from "@generated/graphql";
import { healthStatusText } from "@core/actions";
import { __ } from "@solid/libs";
import { UUID } from "@solid/types";
import useIsMounted from "@core/useIsMounted";

import EntityTable, { EntityTableProps } from "components/EntityTable";

import "./style.css";

type SharingStreamProps = {
  deviceId: UUID;
  sharedStream?: StreamShare[] | null;
  autoUpdate?: boolean;
  onChange: (sharedStream: StreamShare[]) => void;
};

const updateInterval = 5000;
const maxUrlCount = 3;
const urlRegEx = /(rtmp):\/\/(?:([^\s@\/]+?)[@])?([^\s\/:]+)(?:[:]([0-9]+))?(?:(\/[^\s?#]+)([?][^\s#]+)?)?([#]\S*)?/;
const defaultValidUrlLength = "rtmp://xxx".length;

const SharingStream = ({ sharedStream, deviceId, onChange, autoUpdate = false }: SharingStreamProps) => {
  const [getSharedStream, { data, error, loading }] = useGetDeviceSharedStreamLazyQuery({ variables: { id: deviceId }, fetchPolicy: "network-only" });
  const isMounted = useIsMounted();

  const [rtmpUrl, setRtmpUrl] = useState<string>("");
  const [errorUrl, setErrorUrl] = useState<string>();
  const [currentStreams, setCurrentStreams] = useState<StreamShare[]>([]);

  const updateIntervalRef = useRef<NodeJS.Timeout | undefined>(undefined);

  const sharedStreamsTableProps: EntityTableProps = useMemo(() => {
    return {
      columns: ["", "Server", "Status"],
      emptyLine: __("No shared streams"),
      rows: currentStreams.reduce((acc, item, index) => {
        acc.push([
          <Popup
            content={__("Remove sharing")}
            trigger={
              <Icon
                name="remove"
                className="SharingStream-RemoveIcon"
                onClick={() => removeStreamSharing(index)}
              />
            }
          />,
          <span>&nbsp;{item.url}</span>,
          <span>&nbsp;{item.status ? healthStatusText[item.status] : ""}{item.error ? `(${item.error})` : ""}</span>
        ]);
        return acc;
      }, [] as React.ReactNode[][])
    };
  }, [currentStreams]);

  useEffect(() => {
    const currentStreams = Array.from(sharedStream ?? []).sort((a, b) => a.id - b.id);
    setCurrentStreams(currentStreams);
  }, [sharedStream]);

  useEffect(() => {
    if (autoUpdate && !updateIntervalRef.current && currentStreams.length !== 0) {
      updateIntervalRef.current = setInterval(() => {
        isMounted() && getSharedStream();
      }, updateInterval);
    }
    if ((!autoUpdate || currentStreams.length === 0) && updateIntervalRef.current) {
      clearInterval(updateIntervalRef.current);
      updateIntervalRef.current = undefined;
    }

    return () => {
      updateIntervalRef.current && clearInterval(updateIntervalRef.current);
      updateIntervalRef.current = undefined;
    };
  }, [autoUpdate, currentStreams]);

  useEffect(() => {
    if (!loading && !error && data && data.device?.aspects) {
      const aspectWithSharedStream = data.device.aspects.find(aspect => aspect.__typename === "DFA_Media" && aspect.sharedStream);
      if (aspectWithSharedStream && aspectWithSharedStream.__typename === "DFA_Media") {
        const currentUrls = currentStreams.map(stream => stream.url);
        const updatedSharedStreams = aspectWithSharedStream.sharedStream?.filter(stream => currentUrls.includes(stream.url));

        if (updatedSharedStreams && updatedSharedStreams.length > 0) {
          setCurrentStreams(prevStreams => {
            const updatedStreams: StreamShare[] = [];
            let streamsUpdated = false;

            for (const currentStream of prevStreams) {
              const updatedStreamIndex = updatedSharedStreams.findIndex(stream => stream.url === currentStream.url);
              if (updatedStreamIndex >= 0) {
                const updatedStream = updatedSharedStreams[updatedStreamIndex];
                streamsUpdated = !streamsUpdated ?
                  (updatedStream.error !== currentStream.error ||
                    updatedStream.status !== currentStream.status) :
                  streamsUpdated;
                updatedStreams.push(updatedStream);
              }
              else {
                updatedStreams.push(currentStream);
              }
            }

            return streamsUpdated ? updatedStreams : prevStreams;
          });
        }
      }
    }
  }, [data, loading, error]);

  function removeStreamSharing(streamIndex: number) {
    const newStreams = Array.from(currentStreams);
    newStreams.splice(streamIndex, 1);
    setCurrentStreams(newStreams);
    onChange(newStreams);
  }

  function addStreamUrl(url: string) {
    url = url.trim();
    if (isUrlCanAdded(url)) {
      const id = Array.from(currentStreams)[currentStreams.length - 1]?.id + 1 ?? 1;
      const newStream = { id, url };
      setErrorUrl("");
      setRtmpUrl("");
      onChange([...currentStreams, newStream]);
      setCurrentStreams(prevStream => {
        const newStreams = Array.from(prevStream);
        newStreams.push(newStream);
        return newStreams;
      });
    }
  }

  function isUrlCanAdded(url: string): boolean {
    const urlExist = currentStreams.some(stream => stream.url === url);
    if (urlExist) {
      setErrorUrl(__("URL is already exist"));
      return false;
    }

    const allowNewUrl = currentStreams.length < maxUrlCount;
    if (!allowNewUrl) {
      setErrorUrl(__("For Device maximum allowed {{count}} URLs", { count: maxUrlCount }));
      return false;
    }

    const urlValid = urlRegEx.exec(url);
    if (!urlValid) {
      setErrorUrl(__("Invalid RTMP URL"));
      return false;
    }

    return true;
  }

  return (
    <Segment className="SharingStream">
      <Form className="SharingStream-Form">
        <Form.Field inline className="SharingStream-Field" error={!!errorUrl}>
          <div className="SharingStream-FieldLabel">
            <List.Item>{__("RTMP URL")}</List.Item>
            { !!errorUrl && <span className="SharingStream-ErrorUrl">&nbsp;{errorUrl}</span> }
          </div>
          <Input
            autoFocus
            value={rtmpUrl}
            labelPosition="right"
            onChange={(event) => setRtmpUrl(event.currentTarget.value)}
            label={
              <Button
                icon
                disabled={rtmpUrl.length < defaultValidUrlLength}
                onClick={() => addStreamUrl(rtmpUrl)}
              >
                <FontAwesomeIcon icon="plus"/>
              </Button>
            }
          />
        </Form.Field>
      </Form>

      <EntityTable {...sharedStreamsTableProps} />
    </Segment>
  );
};

export default SharingStream;
