import React, { useState, useEffect, useRef, useMemo } from "react";
import { Card, Icon, Input, Checkbox, Label, Button, Message, Dropdown } from "semantic-ui-react";
import classNames from "classnames";
import { useSnapshot } from "@core/actions";
import { DiscoveredDevice, DeviceUserInput } from "../CameraDiscovery";
import Loading from "components/Loading";
import {__} from "@solid/libs/i18n";
import { MediaStreamType } from "@generated/graphql";

type CameraOnvifProps = {
  device: DiscoveredDevice;
  onUserInput: (device: DiscoveredDevice, input: DeviceUserInput) => void;
  onTestClick: (device: DiscoveredDevice) => void;
};

const CameraOnvif = ({ device, onUserInput, onTestClick }: CameraOnvifProps) => {
  const [showPassword, setShowPassword] = useState(false);
  const imageRef = useRef<HTMLImageElement>(null);
  const { imageSrc, loading: snapshotLoading } = useSnapshot({ load: !!device.device, deviceId: device.device?.id ?? "", imageRef });
  const snapshotRef = useRef<HTMLImageElement>(null);
  const streamOptions = useMemo(() => device.aspects
    .filter(aspect => aspect.template.id === "video")
    .map(({ name }) => ({ key: name, value: name, text: name })), [device.aspects]);

  const snapshot = useMemo(() => {
    if (device.snapshots.length === 0) {
      return "";
    }

    let snapshot = device.snapshots[0];
    const streamIndex = device.aspects.findIndex(aspect => aspect.name === device.userInput.streamName);
    if (device.snapshots.length > 0 && streamIndex >= 0) {
      snapshot = device.snapshots[streamIndex];
    }

    return snapshot;
  }, [device]);

  const audioAspects = useMemo(() => {
    return device.aspects.filter(aspect => aspect.__typename === "DFA_Media" && aspect.streamType === MediaStreamType.Audio);
  }, [device]);

  const audioOptions = useMemo(() => {
    if (audioAspects.length > 0) {
      const currentStream = device.aspects.find(aspect =>
        aspect.__typename === "DFA_Media" &&
        aspect.streamType === MediaStreamType.Video &&
        device.userInput.streamName === aspect.name &&
        aspect.onvifProfile
      );

      const audioAspect = audioAspects.find(aspect =>
        aspect.__typename === "DFA_Media" &&
        aspect.onvifProfile &&
        //@ts-ignore
        currentStream?.onvifProfile === aspect.onvifProfile
      );

      if (audioAspect) {
        return [
          {key: audioAspect.name, value: audioAspect.name, text: audioAspect.name},
          {key: `${audioAspect.name}_off`, value: false, text: "none"}
        ];
      }

      return [{key: "_off", value: false, text: "none"}];
    }

    return [];
  }, [audioAspects, device, onUserInput]);

  useEffect(() => {
    setTimeout(() => {
      if (snapshotRef.current) {
        snapshotRef.current.src = "";
        snapshotRef.current.src = `data:image/jpeg;base64, ${snapshot}`;
      }
    }, 250);
  }, [snapshot]);

  return (
    <Card className="CameraDiscovery-Camera" color={device.device ? "green" : undefined}>
      <div className="CameraDiscovery-CameraHeader">
        <Checkbox
          disabled={!device.userInput.isProbed}
          checked={device.userInput.isSelected}
          onChange={(e, { checked }) => onUserInput(device, { ...device.userInput, isSelected: !!checked })}
        />

        {!!device.device &&
        <Label color="blue" content={<>{__("Registered as")}<br/>{`'${device.device.name}'`}</>}/>}
      </div>

      <div className="CameraDiscovery-CameraImage">
        {!!snapshot && <img ref={snapshotRef} src={`data:image/jpeg;base64, ${snapshot}`} alt=""/>}

        {!snapshot &&
        <>
          <img ref={imageRef} src={imageSrc} alt="" className={classNames({ "hidden": !imageSrc })}/>
          {snapshotLoading && <Loading/>}

          {!imageSrc &&
          <>
            <Icon name="image outline" size="huge"/>
            <div>{__("To test the camera stream, enter username and password below and click the Test button or the checkbox")}</div>
          </>}
        </>}
      </div>

      <Card.Content>
        {!!device.userInput.validationError &&
        <Message error content={device.userInput.validationError}/>}

        <div className="CameraDiscovery-CameraFields">
          <div className="CameraDiscovery-CameraFieldLabel">{__("Name")}</div>
          <Input fluid value={device.userInput.name} onChange={e => onUserInput(device, { ...device.userInput, name: e.currentTarget.value })}/>

          <div className="CameraDiscovery-CameraFieldLabel">{__("IP")}</div>
          <span className="CameraDiscovery-CameraFieldValue">{device.config.connect.host ?? ""}</span>

          <div className="CameraDiscovery-CameraFieldLabel">{__("Username")}</div>
          <Input fluid value={device.userInput.userName} onChange={e => onUserInput(device, { ...device.userInput, userName: e.currentTarget.value })}/>

          <div className="CameraDiscovery-CameraFieldLabel">{__("Password")}</div>
          <Input
            fluid
            type={showPassword ? undefined : "password"}
            value={device.userInput.password}
            label={
              <Button icon onClick={() => setShowPassword(value => !value)}>
                <Icon name={showPassword ? "eye slash" : "eye"}/>
              </Button>}
            labelPosition="right"
            onChange={e => onUserInput(device, { ...device.userInput, password: e.currentTarget.value })}
          />


          <div className={classNames("CameraDiscovery-CameraFieldLabel", {"disabled": !streamOptions || streamOptions.length === 0})}>
            {__("Video")}
          </div>
          <Dropdown
            fluid
            selection
            options={streamOptions}
            value={device.userInput.streamName}
            disabled={!streamOptions || streamOptions.length === 0}
            onChange={(e, { value }) => typeof value === "string" && onUserInput(device, { ...device.userInput, streamName: value })}
            />

          <div className={classNames("CameraDiscovery-CameraFieldLabel", {"disabled": !audioAspects || audioOptions.length <= 1})}>
            {__("Audio")}
          </div>
          <Dropdown
            fluid
            selection
            options={audioOptions}
            value={device.userInput.audio}
            disabled={!audioAspects || audioOptions.length <= 1}
            onChange={(e, { value }) => {
              (typeof value === "string" || typeof value === "boolean") && onUserInput(device, { ...device.userInput, audio: value });
            }}
            />
        </div>

        <Button content={__("Test")} className="CameraDiscovery-TestButton" onClick={() => onTestClick(device)}/>
      </Card.Content>

      {!!device.userInput.isLoading && <Loading/>}
    </Card>
  );
};

export default CameraOnvif;
